/**
 * @file event-switches.c
 * Switch event provider for the Mode Control Entity
 * <p>
 * Copyright © 2007 Nokia Corporation.  All rights reserved.
 * <p>
 * @author David Weinehall <david.weinehall@nokia.com>
 */
#include <glib.h>

#include <string.h>			/* strncmp(), strlen() */

#include "mce.h"
#include "event-switches.h"

#include "mce-io.h"			/* mce_register_io_monitor_string(),
					 * mce_unregister_io_monitor()
					 */
#include "datapipe.h"			/* execute_datapipe() */

/** ID for the lockkey I/O monitor */
static gconstpointer lockkey_iomon_id = NULL;

/** ID for the keyboard slide I/O monitor */
static gconstpointer kbd_slide_iomon_id = NULL;

/** ID for the headphone jack sense I/O monitor */
//static gconstpointer headphone_iomon_id = NULL;

/** ID for the lid cover I/O monitor */
static gconstpointer lid_cover_iomon_id = NULL;

/** ID for the usb cable I/O monitor */
//static gconstpointer usb_cable_iomon_id = NULL;

/** ID for the mmc cover I/O monitor */
//static gconstpointer mmc_cover_iomon_id = NULL;

/** ID for the battery cover I/O monitor */
//static gconstpointer bat_cover_iomon_id = NULL;

/** Does the device have a flicker key? */
gboolean has_flicker_key = FALSE;

/**
 * I/O monitor callback for the lock flicker key
 *
 * @param data The new data
 * @param bytes_read Unused
 */
static void lockkey_cb(gpointer data, gsize bytes_read)
{
	gint lockkey_state;

	(void)bytes_read;

	if (!strncmp(data, MCE_FLICKER_KEY_ACTIVE,
		     strlen(MCE_FLICKER_KEY_ACTIVE))) {
		lockkey_state = 1;
	} else {
		lockkey_state = 0;
	}

	(void)execute_datapipe(&lockkey_pipe,
			       GINT_TO_POINTER(lockkey_state), FALSE, TRUE);
}

/**
 * I/O monitor callback for the keyboard slide
 *
 * @param data The new data
 * @param bytes_read Unused
 */
static void kbd_slide_cb(gpointer data, gsize bytes_read)
{
	gint slide_state;

	(void)bytes_read;

	if (!strncmp(data, MCE_KBD_SLIDE_OPEN, strlen(MCE_KBD_SLIDE_OPEN))) {
		slide_state = 1;
	} else {
		slide_state = 0;
	}

	(void)execute_datapipe(&keyboard_slide_pipe,
			       GINT_TO_POINTER(slide_state), FALSE, TRUE);
}

#if 0
/**
 * I/O monitor callback for the headphone jack sense
 *
 * @param data The new data
 * @param bytes_read Unused
 */
static void headphone_cb(gpointer data, gsize bytes_read)
{
	gint headphone_state;

	(void)bytes_read;

	if (!strncmp(data, MCE_HEADPHONE_CONNECTED,
		     strlen(MCE_HEADPHONE_CONNECTED))) {
		headphone_state = 1;
	} else {
		headphone_state = 0;
	}
}
#endif

/**
 * I/O monitor callback for the lid cover
 *
 * @param data The new data
 * @param bytes_read Unused
 */
static void lid_cover_cb(gpointer data, gsize bytes_read)
{
	cover_state_t lid_cover_state;

	(void)bytes_read;

	if (!strncmp(data, MCE_LID_COVER_OPEN, strlen(MCE_LID_COVER_OPEN))) {
		lid_cover_state = COVER_OPEN;
	} else {
		lid_cover_state = COVER_CLOSED;
	}

	(void)execute_datapipe(&lid_cover_pipe,
			       GINT_TO_POINTER(lid_cover_state),
			       FALSE, TRUE);
}

#if 0
/**
 * I/O monitor callback for the USB cable
 *
 * @param data The new data
 * @param bytes_read Unused
 */
static void usb_cable_cb(gpointer data, gsize bytes_read)
{
	gint cable_state;

	(void)bytes_read;

	if (!strncmp(data, MCE_USB_CABLE_CONNECTED,
		     strlen(MCE_USB_CABLE_CONNECTED))) {
		cable_state = 1;
	} else {
		cable_state = 0;
	}
}

/**
 * I/O monitor callback for the MMC cover
 *
 * @param data The new data
 * @param bytes_read Unused
 */
static void mmc_cover_cb(gpointer data, gsize bytes_read)
{
	(void)data;
	(void)bytes_read;
}

/**
 * I/O monitor callback for the battery cover
 *
 * @param data The new data
 * @param bytes_read Unused
 */
static void battery_cover_cb(gpointer data, gsize bytes_read)
{
	(void)data;
	(void)bytes_read;
}
#endif

/**
 * Init function for the switches component
 *
 * @return TRUE on success, FALSE on failure
 */
gboolean mce_switches_init(void)
{
	gboolean status = FALSE;

	/* Set default values, in case these are not available */
	(void)execute_datapipe(&lid_cover_pipe,
			       GINT_TO_POINTER(COVER_OPEN),
			       FALSE, TRUE);

	/* Register I/O monitors */
	// FIXME: error handling?
	lockkey_iomon_id =
		mce_register_io_monitor_string(-1, MCE_FLICKER_KEY_STATE_PATH,
					       FALSE, TRUE, lockkey_cb);
	kbd_slide_iomon_id =
		mce_register_io_monitor_string(-1, MCE_KBD_SLIDE_STATE_PATH,
					       FALSE, TRUE, kbd_slide_cb);
//	headphone_iomon_id =
//		mce_register_io_monitor_string(-1, MCE_HEADPHONE_STATE_PATH,
//					       FALSE, TRUE, headphone_cb);
	lid_cover_iomon_id =
		mce_register_io_monitor_string(-1,
					       MCE_LID_COVER_STATE_PATH,
					       FALSE, TRUE, lid_cover_cb);
//	usb_cable_iomon_id =
//		mce_register_io_monitor_string(-1, MCE_USB_CABLE_STATE_PATH,
//					       FALSE, TRUE, usb_cable_cb);
//	mmc_cover_iomon_id =
//		mce_register_io_monitor_activity(-1, MCE_MMC_COVER_STATE_PATH,
//					         FALSE, mmc_cover_cb);
//	bat_cover_iomon_id =
//		mce_register_io_monitor_activity(-1,
//						 MCE_BATTERY_COVER_STATE_PATH,
//					         FALSE, battery_cover_cb);

	if (lockkey_iomon_id != NULL)
		has_flicker_key = TRUE;

	status = TRUE;

	return status;
}

/**
 * Exit function for the switches component
 */
void mce_switches_exit(void)
{
	/* Unregister I/O monitors */
//	mce_unregister_io_monitor(bat_cover_iomon_id);
//	mce_unregister_io_monitor(mmc_cover_iomon_id);
//	mce_unregister_io_monitor(usb_cable_iomon_id);
	mce_unregister_io_monitor(lid_cover_iomon_id);
//	mce_unregister_io_monitor(headphone_iomon_id);
	mce_unregister_io_monitor(kbd_slide_iomon_id);
	mce_unregister_io_monitor(lockkey_iomon_id);
}
