/**
 * @file event-keypress.c
 * Key press event provider for the Mode Control Entity
 * <p>
 * Copyright © 2004-2007 Nokia Corporation.  All rights reserved.
 * <p>
 * @author David Weinehall <david.weinehall@nokia.com>
 */
#include <glib.h>

#include <stdlib.h>			/* exit(), EXIT_FAILURE */
#include <unistd.h>			/* close() */
#include <linux/input.h>		/* struct input_event, EV_KEY */

#include "mce.h"
#include "event-keypress.h"

#include "mce-io.h"			/* mce_register_io_monitor_chunk(),
					 * mce_unregister_io_monitor()
					 */
#include "mce-log.h"			/* mce_log(), LL_* */
#include "datapipe.h"			/* execute_datapipe() */
#include "input_event.h"		/* get_event_fd */

guint16 power_keycode = POWER_BUTTON;

/** ID for the keypress I/O monitor */
static gconstpointer keypress_iomon_id = NULL;

/** ID for the [power] key I/O monitor */
static gconstpointer powerkey_iomon_id = NULL;

/** File descriptor for keyboard/keypad */
static int eventfd1 = -1;

/** File descriptor for power button */
static int eventfd2 = -1;

/**
 * I/O monitor callback for the touchscreen
 *
 * @param data The new data
 * @param bytes_read The number of bytes read
 */
static void keypress_cb(gpointer data, gsize bytes_read)
{
	struct input_event *ev;

	ev = data;

	if (bytes_read != sizeof (struct input_event)) {
		goto EXIT;
	}

	/* Ignore non-keypress events */
	if (ev->type != EV_KEY) {
		goto EXIT;
	}

	/* Remap [power] key */
	if (ev->code == power_keycode) {
		ev->code = POWER_BUTTON;
	}

	/* For now there's no reason to cache the keypress */
	(void)execute_datapipe(&keypress_pipe, &ev, FALSE, FALSE);

EXIT:
	return;
}

/**
 * Init function for the keypress component
 *
 * @return TRUE on success, FALSE on failure
 */
gboolean mce_keypress_init(void)
{
	gboolean status = FALSE;
	const gchar *event1 = NULL;
	const gchar *event2 = NULL;

	/* Find out what event handler handles the keyboard/keypad */
	if ((eventfd1 = get_event_fd(OMAP_KEYBOARD_N810)) != -1) {
		event1 = OMAP_KEYBOARD_N810;
		has_keyboard = TRUE;
	} else if ((eventfd1 = get_event_fd(OMAP_KEYBOARD_N810_OLD)) != -1) {
		event1 = OMAP_KEYBOARD_N810_OLD;
		has_keyboard = TRUE;
	} else if ((eventfd1 = get_event_fd(OMAP_KEYPAD_N800)) != -1) {
		event1 = OMAP_KEYPAD_N800;
	} else if ((eventfd1 = get_event_fd(OMAP_KEYPAD_N800_OLD)) != -1) {
		event1 = OMAP_KEYPAD_N800_OLD;
	} else if ((eventfd1 = get_event_fd(OMAP_KEYPAD_770)) != -1) {
		event1 = OMAP_KEYPAD_770;
	} else if ((eventfd1 = get_event_fd(AT_KEYBOARD)) != -1) {
		event1 = AT_KEYBOARD;
	}

	if (eventfd1 != -1) {
		mce_log(LL_DEBUG,
			"Got event file descriptor for `%s'",
			event1);
	} else {
		mce_log(LL_ERR,
			"Failed to get a keypad event file descriptor");
	}

	/* If we fail to get a file descriptor,
	 * don't attempt to register an I/O-channel for it
	 */
	if (eventfd1 != -1) {
		/* Register I/O monitors */
		// FIXME: error handling?
		keypress_iomon_id =
			mce_register_io_monitor_chunk(eventfd1, event1, FALSE, FALSE, keypress_cb, sizeof (struct input_event));
	}

	if ((eventfd2 = get_event_fd(RETU_POWERBUTTON)) != -1) {
		event2 = RETU_POWERBUTTON;
	} else if ((eventfd2 = get_event_fd(AT_KEYBOARD)) != -1) {
		event2 = AT_KEYBOARD;
		power_keycode = KEY_F9;
	}

	if (eventfd2 != -1) {
		mce_log(LL_DEBUG,
			"Got event file descriptor for `%s'",
			event2);
	} else {
		mce_log(LL_ERR,
			"Failed to get a keypad event file descriptor");
	}

	/* If both file descriptors point at the same file,
	 * close the second one
	 */
	if (event2 == event1) {
		close(eventfd2);
		eventfd2 = -1;
	}

	/* If we fail to get a file descriptor,
	 * don't try to register an I/O-channel for it
	 */
	if (eventfd2 != -1) {
		/* Register I/O monitors */
		// FIXME: error handling?
		powerkey_iomon_id =
			mce_register_io_monitor_chunk(eventfd2, event2, FALSE, FALSE, keypress_cb, sizeof (struct input_event));
	}

	status = TRUE;

	return status;
}

/**
 * Exit function for the keypress component
 */
void mce_keypress_exit(void)
{
	/* Unregister I/O monitors */
	mce_unregister_io_monitor(powerkey_iomon_id);
	mce_unregister_io_monitor(keypress_iomon_id);

	/* Close file descriptors */
	if (eventfd1 != -1) {
		close(eventfd1);
		eventfd1 = -1;
	}

	if (eventfd2 != -1) {
		close(eventfd2);
		eventfd2 = -1;
	}

	return;
}
