/**
 * @file devlock.c
 * This file implements the device lock component
 * of the Mode Control Entity
 * <p>
 * Copyright © 2004-2007 Nokia Corporation.  All rights reserved.
 * <p>
 * @author David Weinehall <david.weinehall@nokia.com>
 */
#include <glib.h>

#include <cal.h>			/* cal_init(), cal_read-block(),
					 * cal_finish(),
					 * struct cal
					 */
#include <crypt.h>			/* crypt() */
#include <stdlib.h>			/* free(), exit(), EXIT_FAILURE */
#include <string.h>			/* strcmp(), strdup(), strlen() */
#ifndef _XOPEN_SOURCE
#define _XOPEN_SOURCE			/* crypt() */
#endif /* _XOPEN_SOURCE */
#include <unistd.h>			/* crypt() */

#include <mce/mode-names.h>
#include <systemui/dbus-names.h>
#include <systemui/devlock-dbus-names.h>

#include "mce.h"
#include "devlock.h"

#include "mce-log.h"			/* mce_log(), LL_* */
#include "mce-conf.h"			/* mce_conf_get_int() */
#include "mce-dbus.h"			/* mce_dbus_handler_add(),
					 * dbus_send(),
					 * dbus_send_message(),
					 * dbus_new_method_reply(),
					 * dbus_new_signal(),
					 * dbus_message_get_args(),
					 * dbus_message_append_args(),
					 * dbus_message_unref(),
					 * dbus_error_init(),
					 * dbus_error_free(),
					 * DBUS_MESSAGE_TYPE_METHOD_CALL,
					 * DBUS_TYPE_BOOLEAN,
					 * DBUS_TYPE_UINT32, DBUS_TYPE_INT32,
					 * DBUS_TYPE_STRING,
					 * DBUS_TYPE_INVALID,
					 * DBusMessage, DBusError,
					 * dbus_bool_t,
					 * dbus_uint32_t, dbus_int32_t
					 */
#include "mce-dsme.h"			/* request_normal_shutdown(),
					 * set_device_autolock(),
					 * setup_display_settings()
					 */
#include "mce-gconf.h"			/* mce_gconf_set_bool(),
					 * mce_gconf_set_int(),
					 * mce_gconf_notifier_add()
					 */
#include "datapipe.h"			/* append_input_trigger_to_datapipe(),
					 * append_output_trigger_to_datapipe(),
					 * remove_input_trigger_from_datapipe()
					 * remove_output_trigger_from_datapipe()
					 */

/** CAL-data for device lock code */
typedef struct {
	int version;		/**< Version number of the structure */
	gchar passwd[11];	/**< The password hash (DES) */
} passwd_struct;

/** Internal copy of the device lock unlock failure count */
static gint device_lock_failed = DEFAULT_DEVICE_LOCK_FAILED;
/** Internal copy of the device lock total unlock failure count */
static gint device_lock_total_failed = DEFAULT_DEVICE_LOCK_TOTAL_FAILED;

/** Internal copy of the device autolock enabled/disabled status */
static gboolean device_autolock_enabled = DEFAULT_DEVICE_AUTOLOCK_ENABLED;
/** GConf Callback ID for the device autolock enabled/disabled status */
static guint device_autolock_enabled_cb_id = 0;

/** Internal copy of the device autolock timeout */
static gint device_autolock_timeout = DEFAULT_DEVICE_AUTOLOCK_TIMEOUT;
/** GConf Callback ID for the device autolock timeout */
static guint device_autolock_timeout_cb_id = 0;

/** Should the shutdown query be asked when cancel is pressed? */
static gboolean devlock_query_enabled = TRUE;

/** Timeout ID for password failure timer */
static guint devlock_timeout_cb_id = 0;
/** Timeout ID for shutdown query timer */
static guint shutdown_timeout_cb_id = 0;

/** Delay to use when (device_lock_failed % 4) == 0 */
static gint lock_delay_0 = DEFAULT_LOCK_DELAY_0;
/** Delay to use when (device_lock_failed % 4) == 1 */
static gint lock_delay_1 = DEFAULT_LOCK_DELAY_1;
/** Delay to use when (device_lock_failed % 4) == 2 */
static gint lock_delay_2 = DEFAULT_LOCK_DELAY_2;
/** Delay to use when (device_lock_failed % 4) == 3 */
static gint lock_delay_3 = DEFAULT_LOCK_DELAY_3;

/** Timeout before automaic shutdown from shutdown query */
static gint shutdown_timeout = DEFAULT_SHUTDOWN_TIMEOUT;

static gboolean enable_devlock(void);

/**
 * Query the device autolock status
 *
 * @return TRUE if the device lock is enabled,
 *         FALSE if the device lock is disabled
 */
static gboolean is_device_autolock_enabled(void)
{
	return device_autolock_enabled;
}

/**
 * Query the device lock status
 *
 * @return TRUE if the device lock is enabled,
 *         FALSE if the device lock is disabled
 */
static gboolean is_devlock_enabled(void)
{
	return ((mce_get_submode_int32() & MCE_DEVLOCK_SUBMODE) != 0);
}

/**
 * Disable the device autolock
 */
static void disable_device_autolock(void)
{
	set_device_autolock(device_autolock_timeout, FALSE);
}

/**
 * Restore the device autolock
 */
static void restore_device_autolock(void)
{
	set_device_autolock(device_autolock_timeout, device_autolock_enabled);
}

/**
 * GConf callback for device lock related settings
 *
 * @param gce Unused
 * @param id Connection ID from gconf_client_notify_add()
 * @param entry The modified GConf entry
 * @param data Unused
 */
static void devlock_gconf_cb(GConfClient *const gce, const guint id,
			     GConfEntry *const entry, gpointer const data)
{
	GConfValue *gcv = gconf_entry_get_value(entry);

	(void)gce;
	(void)data;

	if (id == device_autolock_enabled_cb_id) {
		device_autolock_enabled = gconf_value_get_bool(gcv);
		restore_device_autolock();
	} else if (id == device_autolock_timeout_cb_id) {
		/* Convert to seconds for DSME */
		device_autolock_timeout = gconf_value_get_int(gcv) * 60;
		restore_device_autolock();
	} else {
		mce_log(LL_WARN, "Spurious GConf value received; confused!");
	}
}

/**
 * Send the device lock mode
 *
 * @param method_call A DBusMessage to reply to;
 *                    pass NULL to send a device lock mode signal instead
 * @return TRUE on success, FALSE on failure
 */
static gboolean mce_send_devlock_mode(DBusMessage *const method_call)
{
	DBusMessage *msg = NULL;
	const gchar *modestring;
	gboolean status = FALSE;

	if (is_devlock_enabled() == TRUE)
		modestring = MCE_DEVICE_LOCKED;
	else
		modestring = MCE_DEVICE_UNLOCKED;

	/* If method_call is set, send a reply,
	 * otherwise, send a signal
	 */
	if (method_call != NULL) {
		msg = dbus_new_method_reply(method_call);
	} else {
		/* devicelock_mode_ind */
		msg = dbus_new_signal(MCE_SIGNAL_PATH, MCE_SIGNAL_IF,
				      MCE_DEVLOCK_MODE_SIG);
	}

	/* Append the new mode */
	if (dbus_message_append_args(msg,
				     DBUS_TYPE_STRING, &modestring,
				     DBUS_TYPE_INVALID) == FALSE) {
		mce_log(LL_CRIT,
			"Failed to append %sargument to D-Bus message for %s",
			method_call ? "reply " : "",
			method_call ? MCE_DEVLOCK_MODE_GET :
				      MCE_DEVLOCK_MODE_SIG);
		dbus_message_unref(msg);
		goto EXIT;
	}

	/* Send the message */
	status = dbus_send_message(msg);

EXIT:
	return status;
}

/**
 * D-Bus reply handler for device lock UI enabling
 *
 * @param pending_call The DBusPendingCall
 * @param data Unused
 * @return TRUE on success, FALSE on failure
 */
static void devlock_ui_open_reply_dbus_cb(DBusPendingCall *pending_call,
					  void *data)
{
	DBusMessage *reply;
	dbus_int32_t retval;
	DBusError error;

	/* Register error channel */
	dbus_error_init(&error);

	(void)data;

	mce_log(LL_DEBUG, "Received device lock UI reply");

	if ((reply = dbus_pending_call_steal_reply(pending_call)) == NULL) {
		mce_log(LL_ERR,
			"Device lock reply callback invoked, "
			"but no pending call available");
		goto EXIT;
	}

	/* Make sure we didn't get an error message */
	if (dbus_message_is_error(reply, DBUS_ERROR_UNKNOWN_METHOD) == TRUE) {
		mce_log(LL_ERR,
			"Device lock D-Bus method not handled");
		goto EXIT2;
	} else if (dbus_message_is_error(reply,
					 DBUS_ERROR_INVALID_ARGS) == TRUE) {
		mce_log(LL_ERR,
			"Incorrect arguments passed to "
			"device lock D-Bus method");
		goto EXIT2;
	}

	/* Extract reply */
	if (dbus_message_get_args(reply, &error,
				  DBUS_TYPE_INT32, &retval,
				  DBUS_TYPE_INVALID) == FALSE) {
		// XXX: should we return an error instead?
		mce_log(LL_CRIT,
			"Failed to get reply from %s: %s",
			SYSTEMUI_DEVLOCK_OPEN_REQ,
			error.message);
		dbus_error_free(&error);
		goto EXIT2;
	}

	switch (retval) {
	case DEVLOCK_REPLY_LOCKED:
		enable_devlock();
		break;

	case DEVLOCK_REPLY_VERIFY:
		mce_add_submode_int32(MCE_VERIFY_SUBMODE);
		break;

	case DEVLOCK_REPLY_FAILED:
	default:
		/* Someone else is using the devlock UI; ignore */
		break;
	}

EXIT2:
	dbus_message_unref(reply);

EXIT:
	dbus_pending_call_unref(pending_call);

	return;
}

/**
 * Enable the device lock
 *
 * @param mode The mode to open in; valid modes:
 *             DEVLOCK_QUERY_ENABLE (open device lock dialog in normal mode)
 *             DEVLOCK_QUERY_OPEN (open device lock dialog in no input mode)
 *             DEVLOCK_QUERY_NOTE (open device lock confirm cancel dialog)
 * @return TRUE on success, FALSE on failure
 */
static gboolean call_devlock_ui(const dbus_uint32_t mode)
{
	const gchar *const cb_service = MCE_SERVICE;
	const gchar *const cb_path = MCE_REQUEST_PATH;
	const gchar *const cb_interface = MCE_REQUEST_IF;
	const gchar *const cb_method = MCE_DEVLOCK_CB_REQ;

	/* com.nokia.system_ui.request.devlock_open */
	return dbus_send(SYSTEMUI_SERVICE, SYSTEMUI_REQUEST_PATH,
			 SYSTEMUI_REQUEST_IF, SYSTEMUI_DEVLOCK_OPEN_REQ,
			 devlock_ui_open_reply_dbus_cb,
			 DBUS_TYPE_STRING, &cb_service,
			 DBUS_TYPE_STRING, &cb_path,
			 DBUS_TYPE_STRING, &cb_interface,
			 DBUS_TYPE_STRING, &cb_method,
			 DBUS_TYPE_UINT32, &mode,
			 DBUS_TYPE_INVALID);
}

/**
 * Timeout callback for shutdown query
 *
 * @param data Unused
 * @return Always returns FALSE, to disable the timeout
 */
static gboolean shutdown_timeout_cb(gpointer data)
{
	(void)data;

	shutdown_timeout_cb_id = 0;

	request_normal_shutdown();

	/* FALSE here is just to disable the timeout */
	return FALSE;
}

/**
 * Timeout callback for device lock query
 *
 * @param data Unused
 * @return Always returns FALSE, to disable the timeout
 */
static gboolean devlock_timeout_cb(gpointer data)
{
	(void)data;

	devlock_timeout_cb_id = 0;

	/* Reopen in normal mode */
	call_devlock_ui(DEVLOCK_QUERY_ENABLE);
	devlock_query_enabled = TRUE;

	/* FALSE here is just to disable the timeout */
	return FALSE;
}

/**
 * Setup delay for device lock
 */
static void devlock_delay(void)
{
	guint delay = 0;

	switch (device_lock_failed % 4) {
	default:
	case 0:
		delay = lock_delay_0;
		break;

	case 1:
		delay = lock_delay_1;
		break;

	case 2:
		delay = lock_delay_2;
		break;

	case 3:
		delay = lock_delay_3;
		break;
	}

	/* Multiply by 1000, since the delays are in milliseconds */
	delay *= 1000;

	devlock_timeout_cb_id = g_timeout_add(delay, devlock_timeout_cb, NULL);
	devlock_query_enabled = FALSE;
}

/**
 * Update the count of failed and total failed login attempts
 */
static void update_password_count(void)
{
	/* Ignore failure to store the count */
	(void)mce_gconf_set_int(MCE_GCONF_DEVICE_LOCK_FAILED_PATH,
			       device_lock_failed);
	(void)mce_gconf_set_int(MCE_GCONF_DEVICE_LOCK_TOTAL_FAILED_PATH,
			       device_lock_total_failed);
}

/**
 * Internal device lock enable
 *
 * Note: since device_lock_failed == 0 means that the device is unlocked,
 * set the value to 4 by default to indicate that the device is locked
 */
static void enable_devlock_internal(void)
{
	if (device_lock_failed == 0) {
		device_lock_failed = 4;
		update_password_count();
	}
}

/**
 * Internal device lock disable
 *
 * Note: device_lock_failed == 0 means that the device is unlocked
 */
static void disable_devlock_internal(void)
{
	device_lock_failed = 0;
	update_password_count();
}

/**
 * Enable the device lock
 *
 * @return TRUE on success, FALSE on failure
 */
static gboolean enable_devlock(void)
{
	mce_add_submode_int32(MCE_DEVLOCK_SUBMODE);
	mce_rem_submode_int32(MCE_VERIFY_SUBMODE);
	// XXX: error handling?
	(void)mce_send_devlock_mode(NULL);
	enable_devlock_internal();

	return TRUE;
}

/**
 * Request enabling of the device lock
 *
 * @return TRUE on success, FALSE on failure
 */
static gboolean request_devlock(void)
{
	/* Enabling the device lock when already enabled is OK
	 * since this happens on boot and when re-entering
	 * normal state from actdead state
	 */
	return call_devlock_ui(DEVLOCK_QUERY_ENABLE);
}

/**
 * Disable the device lock
 *
 * @return TRUE on success, FALSE on failure
 */
static gboolean disable_devlock(void)
{
	gboolean status = FALSE;

	/* Disabling the device lock when already disabled is OK
	 * since this happens when entering actdead state
	 */

	/* com.nokia.system_ui.request.devlock_close */
	if (dbus_send(SYSTEMUI_SERVICE, SYSTEMUI_REQUEST_PATH,
		      SYSTEMUI_REQUEST_IF, SYSTEMUI_DEVLOCK_CLOSE_REQ, NULL,
		      DBUS_TYPE_INVALID) == FALSE)
		goto EXIT;

	mce_rem_submode_int32(MCE_DEVLOCK_SUBMODE);
	mce_rem_submode_int32(MCE_VERIFY_SUBMODE);
	(void)mce_send_devlock_mode(NULL);
	status = TRUE;

EXIT:
	return status;
}

/**
 * Return result from device lock code validation
 *
 * @param method_call D-Bus message containing the code and salt to validate
 * @return TRUE if the code matches,
 *         FALSE if the code does not match or an error occurs
 */
static gboolean validate_devlock_code(DBusMessage *const method_call)
{
	struct cal *cal_data = NULL;
	gboolean result = FALSE;
	gchar *ccode = NULL;
	gchar *code = NULL;
	gchar *salt = NULL;
	DBusError error;

	/* Register error channel */
	dbus_error_init(&error);

	mce_log(LL_DEBUG, "Received validate devlock code request");

	/* Extract device lock code and salt */
	if (dbus_message_get_args(method_call, &error,
				  DBUS_TYPE_STRING, &code,
				  DBUS_TYPE_STRING, &salt,
				  DBUS_TYPE_INVALID) == FALSE) {
		// XXX: should we return an error instead?
		mce_log(LL_CRIT,
			"Failed to get argument from %s: %s",
			MCE_DEVLOCK_VALIDATE_CODE_REQ,
			error.message);
		dbus_error_free(&error);
		goto EXIT;
	}

	/* Retrieve the password stored in CAL */
	if (cal_init(&cal_data) >= 0) {
		void *ptr = NULL;
		unsigned long len;

		if (cal_read_block(cal_data, "lock_code", &ptr, &len,
				   CAL_FLAG_USER) == 0) {
			passwd_struct *tmp = (passwd_struct *)ptr;

			/* Correctness checks */
			if ((len == sizeof (passwd_struct)) &&
			    (tmp != NULL) &&
			    (tmp->version == 1) &&
			    ((tmp->passwd[5] == '\0') ||
			     (tmp->passwd[6] == '\0') ||
			     (tmp->passwd[7] == '\0') ||
			     (tmp->passwd[8] == '\0') ||
			     (tmp->passwd[9] == '\0') ||
			     (tmp->passwd[10] == '\0'))) {
				ccode = strdup(crypt(tmp->passwd, salt));
			}

			free(tmp);
		} else {
			mce_log(LL_INFO,
				"cal_read_block() (lock_code) failed");
		}

		cal_finish(cal_data);
	} else {
		mce_log(LL_ERR,
			"cal_init() failed");
	}

	/* If we failed to read code from CAL, fallback to "12345" */
	if (ccode == NULL) {
		gchar *tmp;

		/* No memory to allocate crypt password; send TRUE */
		if ((tmp = strdup(crypt("12345", salt))) == NULL) {
			result = TRUE;
		} else {
			if (strcmp(tmp + strlen(salt), code) == 0)
				result = TRUE;

			free(tmp);
		}
	} else {
		if (strcmp(ccode + strlen(salt), code) == 0)
			result = TRUE;

		free(ccode);
	}

EXIT:
	return result;
}

/**
 * Return result from device lock code validation
 *
 * @param method_call D-Bus message containing the code and salt to validate
 * @return TRUE on success, FALSE on failure
 */
static gboolean mce_send_devlock_code_validation(DBusMessage *const method_call)
{
	DBusMessage *msg = NULL;
	dbus_bool_t result = validate_devlock_code(method_call);
	gboolean status = FALSE;

	mce_log(LL_DEBUG, "Received devlock code validation request");

	/* Create a reply */
	msg = dbus_new_method_reply(method_call);

	/* Append the result of the validation */
	if (dbus_message_append_args(msg,
				     DBUS_TYPE_BOOLEAN, &result,
				     DBUS_TYPE_INVALID) != TRUE) {
		mce_log(LL_CRIT,
			"Failed to append argument to D-Bus message reply "
			"for %s",
			MCE_DEVLOCK_VALIDATE_CODE_REQ);
		dbus_message_unref(msg);
		goto EXIT;
	}

	/* Send the message */
	status = dbus_send_message(msg);

EXIT:
	return status;
}

/**
 * D-Bus callback for device lock code validation method call
 *
 * @param msg The D-Bus message
 * @return TRUE on success, FALSE on failure
 */
static gboolean validate_devlock_code_req_dbus_cb(DBusMessage *const msg)
{
	gboolean status = FALSE;

	mce_log(LL_DEBUG, "Received devlock validate code request");

	if (mce_send_devlock_code_validation(msg) == FALSE)
		goto EXIT;

	status = TRUE;

EXIT:
	return status;
}

/**
 * Change the device lock code
 *
 * @return TRUE on success, FALSE on failure
 */
static gboolean change_devlock_code(const char *new_code)
{
	gboolean status = FALSE;
	struct cal *cal_data;
	passwd_struct new_entry;

	if (cal_init(&cal_data) < 0) {
		mce_log(LL_ERR, "cal_init() failed");
		goto EXIT;
	}

	if (strlen(new_code) > 10) {
		mce_log(LL_ERR,
			"New device lock code "
			"is > 10 characters long");
		goto EXIT2;
	}

	memset(&new_entry, 0, sizeof (passwd_struct));
	new_entry.version = 1;
	memcpy(new_entry.passwd, new_code, strlen(new_code));

	if (cal_write_block(cal_data, "lock_code",
			    &new_entry, sizeof (passwd_struct),
			    CAL_FLAG_USER) < 0) {
		mce_log(LL_ERR, "Failed to write new lock code to CAL");
		goto EXIT2;
	}

	status = TRUE;

EXIT2:
	cal_finish(cal_data);

EXIT:
	return status;
}

/**
 * D-Bus callback for device lock code change method call
 *
 * @param msg The D-Bus message
 * @return TRUE on success, FALSE on failure
 */
static gboolean change_devlock_code_req_dbus_cb(DBusMessage *const msg)
{
	DBusMessage *reply = NULL;
	gboolean status = FALSE;
	dbus_bool_t result = FALSE;

	mce_log(LL_DEBUG, "Received devlock change code request");

	if (validate_devlock_code(msg) == TRUE) {
		char *discard = NULL;
		char *newcode = NULL;
		DBusError error;

		/* Register error channel */
		dbus_error_init(&error);

		/* Extract device lock code and salt */
		if (dbus_message_get_args(msg, &error,
					  DBUS_TYPE_STRING, &discard,
					  DBUS_TYPE_STRING, &discard,
					  DBUS_TYPE_STRING, &newcode,
					  DBUS_TYPE_INVALID) == FALSE) {
			// XXX: should we return an error instead?
			mce_log(LL_CRIT,
				"Failed to get argument from %s: %s",
				MCE_DEVLOCK_CHANGE_CODE_REQ,
				error.message);
			dbus_error_free(&error);
		} else {
			result = change_devlock_code(newcode);
		}
	}

	/* Create a reply */
	reply = dbus_new_method_reply(msg);

	/* Append the result of the validation */
	if (dbus_message_append_args(reply,
				     DBUS_TYPE_BOOLEAN, &result,
				     DBUS_TYPE_INVALID) != TRUE) {
		mce_log(LL_CRIT,
			"Failed to append reply argument to D-Bus message "
			"for %s",
			MCE_DEVLOCK_CHANGE_CODE_REQ);
		dbus_message_unref(reply);
		goto EXIT;
	}

	/* Send the message */
	status = dbus_send_message(reply);

EXIT:
	return status;
}

/**
 * D-Bus callback for the get devicelock mode method call
 *
 * @param msg The D-Bus message
 * @return TRUE on success, FALSE on failure
 */
static gboolean devlock_mode_get_req_dbus_cb(DBusMessage *const msg)
{
	gboolean status = FALSE;

	mce_log(LL_DEBUG, "Received devlock mode get request");

	/* Try to send a reply that contains the current devlock mode */
	if (mce_send_devlock_mode(msg) == FALSE)
		goto EXIT;

	status = TRUE;

EXIT:
	return status;
}

/**
 * D-Bus callback from SystemUI device lock
 *
 * @param msg The D-Bus message
 * @return TRUE on success, FALSE on failure
 */
static gboolean systemui_devlock_dbus_cb(DBusMessage *const msg)
{
	dbus_int32_t result = INT_MAX;
	gboolean status = FALSE;
	DBusError error;

	/* Register error channel */
	dbus_error_init(&error);

	mce_log(LL_DEBUG, "Received devlock callback");

	if (dbus_message_get_args(msg, &error,
				  DBUS_TYPE_INT32, &result,
				  DBUS_TYPE_INVALID) == FALSE) {
		// XXX: should we return an error instead?
		mce_log(LL_CRIT,
			"Failed to get argument from %s: %s",
			MCE_DEVLOCK_CB_REQ,
			error.message);
		dbus_error_free(&error);
		goto EXIT;
	}

	mce_log(LL_DEBUG, "devlock callback value: %d", result);

	switch (result) {
	case DEVLOCK_RESPONSE_LOCKED:
		/* Should only happen on transition from VERIFY to LOCKED */
		enable_devlock();
		break;

	case DEVLOCK_RESPONSE_SHUTDOWN:
		/* Request for shutdown */
		if (shutdown_timeout_cb_id != 0) {
			g_source_remove(shutdown_timeout_cb_id);
			shutdown_timeout_cb_id = 0;
		}

		request_normal_shutdown();
		break;

	case DEVLOCK_RESPONSE_NOSHUTDOWN:
		g_source_remove(shutdown_timeout_cb_id);
		shutdown_timeout_cb_id = 0;
		call_devlock_ui(DEVLOCK_QUERY_ENABLE);
		break;

	case DEVLOCK_RESPONSE_CORRECT:
		/* Correct password */
		disable_devlock();
		disable_devlock_internal();
		break;

	case DEVLOCK_RESPONSE_INCORRECT:
		/* Incorrect password -- open in no input mode */
		call_devlock_ui(DEVLOCK_QUERY_OPEN);

		/* OK, the risk of someone actually trying to
		 * loop these counters is remote, but let's not
		 * leave it open anyway...
		 */
		if (device_lock_failed < G_MAXINT)
			device_lock_failed++;

		if (device_lock_total_failed < G_MAXINT)
			device_lock_total_failed++;

		update_password_count();
		devlock_delay();
		break;

	case DEVLOCK_RESPONSE_CANCEL:
		/* User pressed cancel; either from the device lock
		 * directly or from its shutdown query
		 */
		if (mce_get_mode_int32() == MCE_VOIP_MODE_INT32) {
			(void)mce_send_devlock_mode(NULL);
		} else if (devlock_query_enabled == TRUE) {
			/* Offer to shutdown */
			call_devlock_ui(DEVLOCK_QUERY_NOTE);
			shutdown_timeout_cb_id =
				g_timeout_add(shutdown_timeout,
					      shutdown_timeout_cb, NULL);
		}

		break;

	default:
		/* Reopen in normal mode */
		call_devlock_ui(DEVLOCK_QUERY_ENABLE);
		break;
	}

	status = TRUE;

EXIT:
	return status;
}

/**
 * Startup device lock
 * will exit the mainloop if an error occurs
 *
 * @return TRUE on success (on failure, the function exits)
 */
static gboolean mce_devlock_startup(void)
{
	/* Enable device lock if necessary */
	/** @todo Is the second part of this statement really necessary? */
	if ((is_devlock_enabled() == TRUE) || (device_lock_failed != 0)) {
		/* If the display settings haven't been setup yet,
		 * we need to do it now
		 */
		setup_display_settings();

		if (request_devlock() == FALSE) {
			mce_log(LL_CRIT, "Failed to lock device");
			g_main_loop_quit(mainloop);
			exit(EXIT_FAILURE);
		}

		mce_log(LL_DEBUG, "Enabling device lock");
	}

	return TRUE;
}

/**
 * Shutdown device lock
 *
 * @todo the call to disable_devlock needs error handling
 */
static void mce_devlock_shutdown(void)
{
	if ((is_devlock_enabled() == TRUE) || (device_lock_failed != 0)) {
		enable_devlock_internal();
	}

	if (shutdown_timeout_cb_id != 0)
		g_source_remove(shutdown_timeout_cb_id);

	shutdown_timeout_cb_id = 0;

	// FIXME: error handling?
	(void)disable_devlock();
	mce_log(LL_DEBUG, "Disabling device lock");

	/* Disable device autolock */
	disable_device_autolock();
}

/**
 * Handle VoIP mode
 *
 * @param data The system mode stored in a pointer
 */
static void voip_mode_trigger(gconstpointer data)
{
	system_mode_t system_mode = GPOINTER_TO_INT(data);

	if ((system_mode == MCE_VOIP_MODE_INT32) &&
	    ((mce_get_submode_int32() & MCE_VERIFY_SUBMODE) != 0))
		disable_devlock();
}

/**
 * Handle device lock state
 * XXX: Add a filter as well, that manages device lock policy
 *
 * @param data The device lock state stored in a pointer
 */
static void device_lock_trigger(gconstpointer data)
{
	lock_state_t device_lock_state = GPOINTER_TO_INT(data);

	switch (device_lock_state) {
	case LOCK_OFF:
		disable_devlock();
		break;

	case LOCK_ON:
		request_devlock();
		break;

	default:
		break;
	}
}

/**
 * Handle system state change
 *
 * @param data The system state stored in a pointer
 */
static void system_state_trigger(gconstpointer data)
{
	system_state_t system_state = GPOINTER_TO_INT(data);

	switch (system_state) {
	case MCE_STATE_USER:
		restore_device_autolock();

		/* If the device autolock is enabled, lock device */
		if (is_device_autolock_enabled() == TRUE) {
			mce_add_submode_int32(MCE_DEVLOCK_SUBMODE);
		}

		mce_devlock_startup();
		break;

	case MCE_STATE_SHUTDOWN:
	case MCE_STATE_ACTDEAD:
	case MCE_STATE_REBOOT:
		mce_devlock_shutdown();
		break;

	default:
		break;
	}
}

/**
 * Init function for the devicelock component
 *
 * @return TRUE on success, FALSE on failure
 */
gboolean mce_devlock_init(void)
{
	gboolean status = FALSE;

	/* Append triggers/filters to datapipes */
	append_input_trigger_to_datapipe(&mode_pipe,
					 voip_mode_trigger);
	append_output_trigger_to_datapipe(&system_state_pipe,
					  system_state_trigger);
	append_output_trigger_to_datapipe(&device_lock_pipe,
					  device_lock_trigger);

	/* Device lock; number of attempts since last successful login */
	/* Since we've set a default, error handling is unnecessary */
	(void)mce_gconf_get_int(MCE_GCONF_DEVICE_LOCK_FAILED_PATH,
				&device_lock_failed);

	/* Enable devicelock if it was shutdown from device lock mode */
	if (device_lock_failed != 0)
		mce_add_submode_int32(MCE_DEVLOCK_SUBMODE);

	/* Device lock; total number of failed attempts */
	/* Since we've set a default, error handling is unnecessary */
	(void)mce_gconf_get_int(MCE_GCONF_DEVICE_LOCK_TOTAL_FAILED_PATH,
				&device_lock_total_failed);

	/* Device autolock enabled */
	/* Since we've set a default, error handling is unnecessary */
	(void)mce_gconf_get_bool(MCE_GCONF_DEVICE_AUTOLOCK_ENABLED_PATH,
				 &device_autolock_enabled);

	/* Enable device lock if autolock is enabled */
	if (is_device_autolock_enabled() == TRUE)
		mce_add_submode_int32(MCE_DEVLOCK_SUBMODE);

	(void)mce_send_devlock_mode(NULL);

	/* Add a notifier for autolock */
	if (mce_gconf_notifier_add(MCE_GCONF_LOCK_PATH,
				   MCE_GCONF_DEVICE_AUTOLOCK_ENABLED_PATH,
				   devlock_gconf_cb,
				   &device_autolock_enabled_cb_id) == FALSE)
		goto EXIT;

	/* Device autolock timeout */
	/* Since we've set a default, error handling is unnecessary */
	(void)mce_gconf_get_int(MCE_GCONF_DEVICE_AUTOLOCK_TIMEOUT_PATH,
				&device_autolock_timeout);

	/* Convert to seconds for DSME */
	device_autolock_timeout *= 60;

	if (mce_gconf_notifier_add(MCE_GCONF_LOCK_PATH,
				   MCE_GCONF_DEVICE_AUTOLOCK_TIMEOUT_PATH,
				   devlock_gconf_cb,
				   &device_autolock_timeout_cb_id) == FALSE)
		goto EXIT;

	/* validate_devicelock_code */
	if (mce_dbus_handler_add(MCE_REQUEST_IF,
				 MCE_DEVLOCK_VALIDATE_CODE_REQ,
				 NULL,
				 DBUS_MESSAGE_TYPE_METHOD_CALL,
				 validate_devlock_code_req_dbus_cb) == FALSE)
		goto EXIT;

	/* change_devicelock_code */
	if (mce_dbus_handler_add(MCE_REQUEST_IF,
				 MCE_DEVLOCK_CHANGE_CODE_REQ,
				 NULL,
				 DBUS_MESSAGE_TYPE_METHOD_CALL,
				 change_devlock_code_req_dbus_cb) == FALSE)
		goto EXIT;

	/* get_devicelock_mode */
	if (mce_dbus_handler_add(MCE_REQUEST_IF,
				 MCE_DEVLOCK_MODE_GET,
				 NULL,
				 DBUS_MESSAGE_TYPE_METHOD_CALL,
				 devlock_mode_get_req_dbus_cb) == FALSE)
		goto EXIT;

	/* devlock_callback */
	if (mce_dbus_handler_add(MCE_REQUEST_IF,
				 MCE_DEVLOCK_CB_REQ,
				 NULL,
				 DBUS_MESSAGE_TYPE_METHOD_CALL,
				 systemui_devlock_dbus_cb) == FALSE)
		goto EXIT;

	/* Get configuration options */
	lock_delay_0 = mce_conf_get_int(MCE_CONF_DEVLOCK_GROUP,
					MCE_CONF_DEVLOCK_DELAY_0,
					DEFAULT_LOCK_DELAY_0,
					NULL);

	lock_delay_1 = mce_conf_get_int(MCE_CONF_DEVLOCK_GROUP,
					MCE_CONF_DEVLOCK_DELAY_1,
					DEFAULT_LOCK_DELAY_1,
					NULL);

	lock_delay_2 = mce_conf_get_int(MCE_CONF_DEVLOCK_GROUP,
					MCE_CONF_DEVLOCK_DELAY_2,
					DEFAULT_LOCK_DELAY_2,
					NULL);

	lock_delay_3 = mce_conf_get_int(MCE_CONF_DEVLOCK_GROUP,
					MCE_CONF_DEVLOCK_DELAY_3,
					DEFAULT_LOCK_DELAY_3,
					NULL);

	shutdown_timeout = mce_conf_get_int(MCE_CONF_DEVLOCK_GROUP,
					    MCE_CONF_DEVLOCK_SHUTDOWN_TIMEOUT,
					    DEFAULT_SHUTDOWN_TIMEOUT,
					    NULL);

	status = TRUE;

EXIT:
	return status;
}

/**
 * Exit function for the devicelock component
 *
 * @todo D-Bus unregistration
 */
void mce_devlock_exit(void)
{
	/* Remove triggers/filters from datapipes */
	remove_output_trigger_from_datapipe(&device_lock_pipe,
					    device_lock_trigger);
	remove_output_trigger_from_datapipe(&system_state_pipe,
					    system_state_trigger);
	remove_input_trigger_from_datapipe(&mode_pipe,
					   voip_mode_trigger);

	if (devlock_timeout_cb_id != 0) {
		g_source_remove(devlock_timeout_cb_id);
		devlock_timeout_cb_id = 0;
	}

	if (shutdown_timeout_cb_id != 0) {
		g_source_remove(shutdown_timeout_cb_id);
		shutdown_timeout_cb_id = 0;
	}

	/* Do nothing for now */
	return;
}
