/**
 * @file devlock-blocker.c
 * Boottime device lock blocker
 * <p>
 * Copyright © 2005-2007 Nokia Corporation.  All rights reserved.
 * <p>
 * @author David Weinehall <david.weinehall@nokia.com>
 */
#include <glib.h>

#include <errno.h>			/* errno,
					 * ENOMEM
					 */
#include <stdio.h>			/* fprintf() */
#define _GNU_SOURCE
#include <getopt.h>			/* getopt_long(),
					 * struct option
					 */
#include <signal.h>			/* signal(),
					 * SIGHUP, SIGTERM, SIGUSR1, SIGUSR2,
					 * SIG_IGN
					 */
#include <stdlib.h>			/* EXIT_FAILURE */
#include <string.h>			/* strcmp() */
#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-lowlevel.h>	/* dbus_connection_setup_with_g_main */

#include <mce/dbus-names.h>
#include <mce/mode-names.h>

#include "mce.h"

#include "mce-log.h"			/* mce_log_open(), mce_log_close(),
					 * mce_log(), mce_log_set_verbosity(),
					 * LL_*
					 */

/** Name shown by --help etc. */
#define PRG_NAME			"devlock-blocker"

extern int optind;			/**< Used by getopt */
extern char *optarg;			/**< Used by getopt */

static const gchar *progname;	/**< Used to store the name of the program */

static GMainLoop *loop = NULL;	/**< The GMainLoop used by devlock-blocker */

static DBusError dbus_error;	/**< D-Bus error */
/** TRUE if dbus_error is initialised */
static gboolean error_initialised = FALSE;

/**
 * Display usage information
 */
static void usage(void)
{
	fprintf(stdout,
		_("Usage: %s [OPTION]...\n"
		  "Device lock blocker for MCE\n"
		  "\n"
		  "  -S, --session       use the session bus instead of the "
		  "system bus for D-Bus\n"
		  "      --verbose       increase debug message verbosity\n"
		  "      --quiet         decrease debug message verbosity\n"
		  "      --help          display this help and exit\n"
		  "      --version       output version information and exit\n"
		  "\n"
		  "Report bugs to <david.weinehall@nokia.com>\n"),
		progname);
}

/**
 * Display version information
 */
static void version(void)
{
	fprintf(stdout, _("%s v%s\n%s"),
		progname,
		G_STRINGIFY(PRG_VERSION),
		_("Written by David Weinehall.\n"
		  "\n"
		  "Copyright (C) 2005-2007 Nokia Corporation.  "
		  "All rights reserved.\n"));
}

/**
 * Initialise locale support
 *
 * @param name The program name to output in usage/version information
 * @return 0 on success, non-zero on failure
 */
static gint init_locales(const gchar *const name)
{
	gint status = 0;

#ifdef ENABLE_NLS
	setlocale(LC_ALL, "");

	if ((bindtextdomain(name, LOCALEDIR) == 0) && (errno == ENOMEM)) {
		status = errno;
		goto EXIT;
	}

	if ((textdomain(name) == 0) && (errno == ENOMEM)) {
		status = errno;
		return 0;
	}

EXIT:
	/* In this error-message we don't use _(), since we don't
	 * know where the locales failed, and we probably won't
	 * get a reasonable result if we try to use them.
	 */
	if (status != 0) {
		fprintf(stderr,
			"%s: `%s' failed; %s. Aborting.\n",
			name, "init_locales", g_strerror(errno));
	} else {
		progname = name;
		errno = 0;
	}
#else
	progname = name;
#endif /* ENABLE_NLS */

	return status;
}

/**
 * D-Bus message handler
 *
 * @todo Error message should be sent on failure instead of
 *       returning DBUS_HANDLER_RESULT_NOT_YET_HANDLED
 *
 * @param connection Unused
 * @param msg The D-Bus message received
 * @param user_data Unused
 * @return DBUS_HANDLER_RESULT_HANDLED if message was handled
 *         DBUS_HANDLER_RESULT_NOT_HANDLED if message was not for us or
 *         if some error occured
 */
static DBusHandlerResult msg_handler(DBusConnection *const connection,
				     DBusMessage *const msg,
				     gpointer const user_data)
{
	(void)connection;
	(void)user_data;

/* Signals from MCE */

	/* Device lock state changed */
	if (dbus_message_is_signal(msg, MCE_SIGNAL_IF,
				   MCE_DEVLOCK_MODE_SIG) == TRUE) {
		gchar *mode = NULL;

		mce_log(LL_DEBUG, "Received MCE devlock mode");

		if (dbus_message_get_args(msg, &dbus_error,
					  DBUS_TYPE_STRING, &mode,
					  DBUS_TYPE_INVALID) == FALSE) {
			// XXX: should we return an error instead?
			mce_log(LL_CRIT,
				"Failed to get argument from %s: %s",
				MCE_DEVLOCK_MODE_SIG,
				dbus_error.message);
			dbus_error_free(&dbus_error);
			return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
		}

		mce_log(LL_DEBUG, "New devlock mode: %s", mode);

		/* If mode == MCE_DEVICE_UNLOCKED, it's time to exit */
		if (strcmp(mode, MCE_DEVICE_UNLOCKED) == 0)
			g_main_loop_quit(loop);
	} else {
		/* Not intended for us */
		return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
	}

	return DBUS_HANDLER_RESULT_HANDLED;
}

/**
 * Initialise the message handler(s) used by devlock-blocker
 *
 * @param connection The D-Bus connection to add the handler(s) to
 * @return TRUE on success, FALSE on failure
 */
static gboolean dbus_init_message_handlers(DBusConnection *const connection)
{
	gboolean status = FALSE;

	/* Listen to signals from MCE */
	dbus_bus_add_match(connection,
			   "type='signal',"
			   "interface='" MCE_SIGNAL_IF "'",
			   &dbus_error);

	if (dbus_error_is_set(&dbus_error) == TRUE) {
		mce_log(LL_CRIT, "Failed to add D-Bus match for "
			"'" MCE_SIGNAL_IF "'; %s",
			dbus_error.message);
		goto EXIT;
	}

	if (dbus_connection_add_filter(connection, msg_handler,
				       NULL, NULL) == FALSE) {
		mce_log(LL_CRIT, "Failed to add D-Bus filter");
		goto EXIT;
	}

	status = TRUE;

EXIT:
	return status;
}

/**
 * Main
 *
 * @param argc Number of command line arguments
 * @param argv Array with command line arguments
 * @return 0 on success, non-zero on failure.
 */
int main(int argc, char **argv)
{
	int optc;
	int opt_index;

	int verbosity = LL_DEFAULT;
	int consolelog = MCE_LOG_SYSLOG;

	gint status = 0;

	DBusBusType bus_type = DBUS_BUS_SYSTEM;
	DBusConnection *dbus_connection = NULL;

	struct option const options[] = {
		{ "session", no_argument, 0, 'S' },
		{ "quiet", no_argument, 0, 'q' },
		{ "verbose", no_argument, 0, 'v' },
		{ "help", no_argument, 0, 'h' },
		{ "version", no_argument, 0, 'V' },
		{ 0, 0, 0, 0 }
	};

	/* Initialise support for locales, and set the program-name */
	if (init_locales(PRG_NAME) != 0)
		goto EXIT;

	/* Parse the command-line options */
	while ((optc = getopt_long(argc, argv, "S",
				   options, &opt_index)) != -1) {
		switch (optc) {
		case 'S':
			bus_type = DBUS_BUS_SESSION;
			break;

		case 'q':
			if (verbosity > LL_CRIT)
				verbosity--;
			break;

		case 'v':
			if (verbosity < LL_DEBUG)
				verbosity++;
			break;

		case 'h':
			usage();
			goto EXIT;

		case 'V':
			version();
			goto EXIT;

		default:
			usage();
			status = EINVAL;
			goto EXIT;
		}
	}

	/* We don't take any non-flag arguments */
	if ((argc - optind) > 0) {
		fprintf(stderr,
			_("%s: Too many arguments\n"
			  "Try: `%s --help' for more information.\n"),
			progname, progname);
		status = EINVAL;
		goto EXIT;
	}

	mce_log_open(PRG_NAME, LOG_DAEMON,
		     (consolelog == TRUE) ? MCE_LOG_SYSLOG : MCE_LOG_STDERR);
	mce_log_set_verbosity(verbosity);

	/* Trap a few signals */
	signal(SIGHUP, SIG_IGN);
	signal(SIGTERM, SIG_IGN);
	signal(SIGUSR1, SIG_IGN);
	signal(SIGUSR2, SIG_IGN);

	/* Register error channel */
	dbus_error_init(&dbus_error);
	error_initialised = TRUE;

	/* Establish D-Bus connection */
	if ((dbus_connection = dbus_bus_get(bus_type,
					    &dbus_error)) == NULL) {
		mce_log(LL_CRIT, "Failed to open connection to message bus");
		status = EXIT_FAILURE;
		goto EXIT;
	} else {
		gchar *mode;

		DBusMessage *msg;
		DBusMessage *reply;

		mce_log(LL_DEBUG, "Querying MCE devlock mode");

		/* Query whether device is locked or not */
		if ((msg = dbus_message_new_method_call(MCE_SERVICE,
							MCE_REQUEST_PATH,
							MCE_REQUEST_IF,
							MCE_DEVLOCK_MODE_GET)) == NULL) {
			mce_log(LL_CRIT, "Cannot allocate memory for "
				"D-Bus method call!");
			status = EXIT_FAILURE;
			goto EXIT;
		}

		mce_log(LL_DEBUG, "Got MCE devlock mode reply");
		reply = dbus_connection_send_with_reply_and_block(dbus_connection, msg, -1, &dbus_error);
		dbus_message_unref(msg);

		if ((dbus_error_is_set(&dbus_error) == TRUE) ||
		    (reply == NULL)) {
			mce_log(LL_CRIT, "Cannot call method %s; %s; exiting",
				MCE_DEVLOCK_MODE_GET,
				dbus_error.message);
			status = EXIT_FAILURE;
			goto EXIT;
		}

		if (dbus_message_get_args(reply, &dbus_error,
					  DBUS_TYPE_STRING, &mode,
					  DBUS_TYPE_INVALID) == FALSE) {
			// XXX: should we return an error instead?
			mce_log(LL_CRIT,
				"Failed to get argument from %s: %s",
				MCE_DEVLOCK_MODE_GET,
				dbus_error.message);
			dbus_message_unref(reply);
			dbus_error_free(&dbus_error);
			status = EXIT_FAILURE;
			goto EXIT;
		}

		mce_log(LL_DEBUG, "New devlock mode: %s", mode);
		dbus_message_unref(reply);

		/* If mode == MCE_DEVICE_UNLOCKED, it's time to exit */
		if (strcmp(mode, MCE_DEVICE_UNLOCKED) == 0)
			goto EXIT;
	}

	/* Register a mainloop */
	loop = g_main_loop_new(NULL, FALSE);

	/* Connect D-Bus to the mainloop */
	dbus_connection_setup_with_g_main(dbus_connection, NULL);

	/* Initialise message handlers */
	if (dbus_init_message_handlers(dbus_connection) == FALSE) {
		status = EXIT_FAILURE;
		goto EXIT;
	}

	/* Run the main loop */
	g_main_loop_run(loop);

	/* If we get here, the main loop has terminated;
	 * either because we requested or because of an error
	 */
EXIT:
	/* If we have an initialised mainloop, unreference it */
	if (loop != NULL)
		g_main_loop_unref(loop);

	/* If there is an established D-Bus connection, unreference it */
	if (dbus_connection != NULL) {
		mce_log(LL_DEBUG, "Unreferencing D-Bus connection");
		dbus_connection_unref(dbus_connection);
		dbus_connection = NULL;
	}

	/* If a D-Bus error channel is registered, free it */
	if ((error_initialised == TRUE) &&
	    (dbus_error_is_set(&dbus_error) == TRUE)) {
		mce_log(LL_DEBUG, "Unregistering D-Bus error channel");
		dbus_error_free(&dbus_error);
	}

	/* Log a farewell message and close the log */
	mce_log(LL_INFO, "Exiting...");
	mce_log_close();

	return status;
}
