/**
 * @file datapipe.h
 * Headers for the simple filter framework
 * <p>
 * Copyright © 2007 Nokia Corporation.  All rights reserved.
 * <p>
 * @author David Weinehall <david.weinehall@nokia.com>
 */
#ifndef _DATAPIPE_H_
#define _DATAPIPE_H_

#include <glib.h>

/**
 * Datapipe structure
 *
 * Only access this struct through the functions
 */
typedef struct {
	GSList *filters;		/**< The filters */
	GSList *input_triggers;		/**< Triggers called on indata */
	GSList *output_triggers;	/**< Triggers called on outdata */
	gpointer cached_data;		/**< Latest cached data */
	gpointer old_cached_data;	/**< Previous cached data */
	gsize datasize;			/**< Size of data; NULL == automagic */
	gboolean free_cache;		/**< Free the cache? */
	gboolean read_only;		/**< Datapipe is read only */
} datapipe_struct;

/* Data retrieval */
#define datapipe_get_gbool(_datapipe)	(GPOINTER_TO_INT((_datapipe).cached_data))
#define datapipe_get_gint(_datapipe)	(GPOINTER_TO_INT((_datapipe).cached_data))
#define datapipe_get_guint(_datapipe)	(GPOINTER_TO_UINT((_datapipe).cached_data))
#define datapipe_get_gsize(_datapipe)	(GPOINTER_TO_SIZE((_datapipe).cached_data))
#define datapipe_get_pointer(_datapipe)	((_datapipe).cached_data)

/* Old data retrieval */
#define datapipe_get_old_gbool(_datapipe)	(GPOINTER_TO_INT((_datapipe).old_cached_data))
#define datapipe_get_old_gint(_datapipe)	(GPOINTER_TO_INT((_datapipe).old_cached_data))
#define datapipe_get_old_guint(_datapipe)	(GPOINTER_TO_UINT((_datapipe).old_cached_data)
#define datapipe_get_old_gsize(_datapipe)	(GPOINTER_TO_SIZE((_datapipe).old_cached_data)
#define datapipe_get_old_pointer(_datapipe)	((_datapipe).old_cached_data)

/* Datapipe execution */
void execute_datapipe_input_triggers(datapipe_struct *const datapipe,
				     gpointer const indata,
				     const gboolean use_cache,
				     const gboolean cache_indata);
gconstpointer execute_datapipe_filters(datapipe_struct *const datapipe,
				       gpointer indata,
				       const gboolean use_cache);
void execute_datapipe_output_triggers(const datapipe_struct *const datapipe,
				      gconstpointer indata,
				      const gboolean use_cache);
gconstpointer execute_datapipe(datapipe_struct *const datapipe,
			       gpointer indata,
			       const gboolean use_cache,
			       const gboolean cache_indata);

/* Filters */
void append_filter_to_datapipe(datapipe_struct *const datapipe,
			       gpointer (*filter)(gpointer data));
void remove_filter_from_datapipe(datapipe_struct *const datapipe,
				 gpointer (*filter)(gpointer data));

/* Input triggers */
void append_input_trigger_to_datapipe(datapipe_struct *const datapipe,
				      void (*trigger)(gconstpointer data));
void remove_input_trigger_from_datapipe(datapipe_struct *const datapipe,
					void (*trigger)(gconstpointer data));

/* Output triggers */
void append_output_trigger_to_datapipe(datapipe_struct *const datapipe,
				       void (*trigger)(gconstpointer data));
void remove_output_trigger_from_datapipe(datapipe_struct *const datapipe,
					 void (*trigger)(gconstpointer data));

void setup_datapipe(datapipe_struct *const datapipe,
		    const gboolean read_only, const gboolean free_cache,
		    const gsize datasize, gpointer initial_data);
void free_datapipe(datapipe_struct *const datapipe);

#endif /* _DATAPIPE_H_ */
