/**
 * @file datapipe.c
 * This file implements the sinmple datapipe framework;
 * this can be used to filter data and to setup data triggers
 * <p>
 * Copyright © 2007 Nokia Corporation.  All rights reserved.
 * <p>
 * @author David Weinehall <david.weinehall@nokia.com>
 */
#include <glib.h>

#include "datapipe.h"

#include "mce-log.h"			/* mce_log(), LL_* */

/**
 * Execute the input triggers of a datapipe
 *
 * @param datapipe The datapipe to execute
 * @param indata The input data to run through the datapipe
 * @param use_cache TRUE to use data from cache, FALSE to use indata
 * @param cache_indata TRUE to cache the indata, FALSE to keep the old data
 */
void execute_datapipe_input_triggers(datapipe_struct *const datapipe,
				     gpointer const indata,
				     const gboolean use_cache,
				     const gboolean cache_indata)
{
	void (*trigger)(gconstpointer const input);
	gpointer data;
	gint i;

	if (datapipe == NULL) {
		/* Potential memory leak! */
		mce_log(LL_ERR,
			"Warning: execute_datapipe_input_triggers called "
			"without a valid datapipe");
		goto EXIT;
	}

	data = use_cache ? datapipe->cached_data : indata;

	if (cache_indata == TRUE) {
		if (datapipe->free_cache == TRUE) {
			g_free(datapipe->old_cached_data);
		}

		if (use_cache == FALSE) {
			datapipe->old_cached_data = datapipe->cached_data;
			datapipe->cached_data = data;
		} else {
			/* datapipe->cached_data already has
			 * the right content, we only need to update
			 * old_cached_data
			 */
			if (datapipe->free_cache == TRUE) {
				/* NULL == string */
				if (datapipe->datasize == 0) {
					datapipe->old_cached_data = g_strdup(data);
				} else {
					datapipe->old_cached_data = g_memdup(data, datapipe->datasize);
				}
			} else {
				datapipe->old_cached_data = data;
			}
		}
	}

	for (i = 0; (trigger = g_slist_nth_data(datapipe->input_triggers,
						i)) != NULL; i++) {
		trigger(data);
	}

EXIT:
	return;
}

/**
 * Execute the filters of a datapipe
 *
 * @param datapipe The datapipe to execute
 * @param indata The input data to run through the datapipe
 * @param use_cache TRUE to use data from cache, FALSE to use indata
 * @return The processed data
 */
gconstpointer execute_datapipe_filters(datapipe_struct *const datapipe,
				       gpointer indata,
				       const gboolean use_cache)
{
	gpointer (*filter)(gpointer input);
	gpointer data;
	gconstpointer retval = NULL;
	gint i;

	if (datapipe == NULL)
		goto EXIT;

	data = use_cache ? datapipe->cached_data : indata;

	for (i = 0; (filter = g_slist_nth_data(datapipe->filters,
					       i)) != NULL; i++) {
		gpointer tmp;

		tmp = filter(data);

		/* If the data needs to be freed, and this isn't the indata,
		 * or if we're not using the cache, then free the data
		 */
		if ((datapipe->free_cache == TRUE) &&
		    ((i > 0) || (use_cache == FALSE)))
			g_free(data);

		data = tmp;
	}

	retval = data;

EXIT:
	return retval;
}

/**
 * Execute the output triggers of a datapipe
 *
 * @param datapipe The datapipe to execute
 * @param indata The input data to run through the datapipe
 * @param use_cache TRUE to use data from cache, FALSE to use indata
 */
void execute_datapipe_output_triggers(const datapipe_struct *const datapipe,
				      gconstpointer indata,
				      const gboolean use_cache)
{
	void (*trigger)(gconstpointer input);
	gconstpointer data;
	gint i;

	if (datapipe == NULL)
		goto EXIT;

	data = use_cache ? datapipe->cached_data : indata;

	for (i = 0; (trigger = g_slist_nth_data(datapipe->output_triggers,
						i)) != NULL; i++) {
		trigger(data);
	}

EXIT:
	return;
}

/**
 * Execute the datapipe
 *
 * @param datapipe The datapipe to execute
 * @param indata The input data to run through the datapipe
 * @param use_cache TRUE to use data from cache, FALSE to use indata
 * @param cache_indata TRUE to cache the indata, FALSE to keep the old data
 * @return The processed data
 */
gconstpointer execute_datapipe(datapipe_struct *const datapipe,
			       gpointer indata,
			       const gboolean use_cache,
			       const gboolean cache_indata)
{
	gconstpointer data = NULL;

	if (datapipe == NULL)
		goto EXIT;

	execute_datapipe_input_triggers(datapipe, indata, use_cache,
					cache_indata);

	if (datapipe->read_only == TRUE) {
		data = indata;
	} else {
		data = execute_datapipe_filters(datapipe, indata, use_cache);
	}

	execute_datapipe_output_triggers(datapipe, data, FALSE);

EXIT:
	return data;
}

/**
 * Append a filter to an existing datapipe
 *
 * @param datapipe The datapipe to manipulate
 * @param filter The filter to add to the datapipe
 */
void append_filter_to_datapipe(datapipe_struct *const datapipe,
			       gpointer (*filter)(gpointer data))
{
	if (!datapipe || !filter || datapipe->read_only)
		return;

	datapipe->filters = g_slist_append(datapipe->filters, filter);
}

/**
 * Remove a filter from an existing datapipe
 * Non-existing filters are ignored
 *
 * @param datapipe The datapipe to manipulate
 * @param filter The filter to remove from the datapipe
 */
void remove_filter_from_datapipe(datapipe_struct *const datapipe,
				 gpointer (*filter)(gpointer data))
{
	if (!datapipe || !filter || datapipe->read_only)
		return;

	datapipe->filters = g_slist_remove(datapipe->filters, filter);
}

/**
 * Append an input trigger to an existing datapipe
 *
 * @param datapipe The datapipe to manipulate
 * @param trigger The trigger to add to the datapipe
 */
void append_input_trigger_to_datapipe(datapipe_struct *const datapipe,
				      void (*trigger)(gconstpointer data))
{
	if (!datapipe || !trigger)
		return;

	datapipe->input_triggers = g_slist_append(datapipe->input_triggers,
						  trigger);
}

/**
 * Remove an input trigger from an existing datapipe
 * Non-existing triggers are ignored
 *
 * @param datapipe The datapipe to manipulate
 * @param trigger The trigger to remove from the datapipe
 */
void remove_input_trigger_from_datapipe(datapipe_struct *const datapipe,
					void (*trigger)(gconstpointer data))
{
	if (!datapipe || !trigger)
		return;

	datapipe->input_triggers = g_slist_remove(datapipe->input_triggers,
						  trigger);
}

/**
 * Append an output trigger to an existing datapipe
 *
 * @param datapipe The datapipe to manipulate
 * @param trigger The trigger to add to the datapipe
 */
void append_output_trigger_to_datapipe(datapipe_struct *const datapipe,
				       void (*trigger)(gconstpointer data))
{
	if (!datapipe || !trigger)
		return;

	datapipe->output_triggers = g_slist_append(datapipe->output_triggers,
						   trigger);
}

/**
 * Remove an output trigger from an existing datapipe
 * Non-existing triggers are ignored
 *
 * @param datapipe The datapipe to manipulate
 * @param trigger The trigger to remove from the datapipe
 */
void remove_output_trigger_from_datapipe(datapipe_struct *const datapipe,
					 void (*trigger)(gconstpointer data))
{
	if (!datapipe || !trigger)
		return;

	datapipe->output_triggers = g_slist_remove(datapipe->output_triggers,
						   trigger);
}

/**
 * Initialise a datapipe
 *
 * @param datapipe The datapipe to manipulate
 * @param read_only TRUE if the datapipe is read only, FALSE if it's read/write
 * @param free_cache TRUE if the cached data needs to be freed, FALSE if not
 * @param datasize Pass size of memory to copy,
 *		   or 0 if only passing pointers or data as pointers
 * @param initial_data Initial cache content
 */
void setup_datapipe(datapipe_struct *const datapipe,
		    const gboolean read_only, const gboolean free_cache,
		    const gsize datasize, gpointer initial_data)
{
	if (!datapipe)
		return;

	datapipe->filters = NULL;
	datapipe->input_triggers = NULL;
	datapipe->output_triggers = NULL;
	datapipe->datasize = datasize;
	datapipe->read_only = read_only;

	datapipe->cached_data = initial_data;

	if (free_cache == TRUE) {
		/* NULL == string */
		if (datasize == 0) {
			datapipe->old_cached_data = g_strdup(initial_data);
		} else {
			datapipe->old_cached_data =
				g_memdup(initial_data, datasize);
		}
	} else {
		datapipe->old_cached_data = initial_data;
	}
}

/**
 * Deinitialize a datapipe
 *
 * @param datapipe The datapipe to manipulate
 */
void free_datapipe(datapipe_struct *const datapipe)
{
	/* Warn about still registered filters/triggers */
	if (datapipe->filters != NULL) {
		mce_log(LL_INFO,
			"free_datapipe() called on a datapipe that "
			"still has registered filter(s)");
	}

	if (datapipe->input_triggers != NULL) {
		mce_log(LL_INFO,
			"free_datapipe() called on a datapipe that "
			"still has registered input_trigger(s)");
	}

	if (datapipe->output_triggers != NULL) {
		mce_log(LL_INFO,
			"free_datapipe() called on a datapipe that "
			"still has registered output_trigger(s)");
	}

	if (datapipe->free_cache == TRUE) {
		g_free(datapipe->old_cached_data);
		g_free(datapipe->cached_data);
	}
}
