/**
 * @file powerkey.c
 * Connectivity logic for the Mode Control Entity
 * <p>
 * Copyright © 2004-2007 Nokia Corporation.  All rights reserved.
 * <p>
 * @author David Weinehall <david.weinehall@nokia.com>
 */
#include <glib.h>

#include <string.h>			/* strcmp() */

#include <bt-dbus.h>			/* BTCOND_SERVICE,
					 * BTCOND_REQ_PATH,
					 * BTCOND_REQ_INTERFACE,
					 * BTCOND_GET_BDA_LIST_REQ,
					 * BTCOND_CONNECTION_STATUS_REQ,
					 */
#include <wlancond-dbus.h>		/* WLANCOND_SERVICE,
					 * WLANCOND_REQ_PATH,
					 * WLANCOND_REQ_INTERFACE,
					 * WLANCOND_CONNECTION_STATUS_REQ,
					 */

#include "mce.h"
#include "connectivity.h"

#include "mce-log.h"			/* mce_log(), LL_* */
#include "mce-dbus.h"			/* dbus_send_message_with_reply_and_block(),
					 * dbus_new_method_call(),
					 * dbus_message_get_args(),
					 * dbus_message_append_args(),
					 * dbus_message_unref(),
					 * dbus_free_string_array(),
					 * dbus_error_init(),
					 * dbus_error_free(),
					 * DBUS_TYPE_BOOLEAN,
					 * DBUS_TYPE_STRING, DBUS_TYPE_ARRAY,
					 * DBUS_TYPE_INVALID,
					 * DBusMessage, DBusError,
					 * dbus_bool_t,
					 * dbus_int32_t
					 */

/**
 * Check connectivity status
 *
 * @param[out] connected Will contain TRUE if there is a possibility
 *		         of open connections, and FALSE if no connections
 *		         are open
 * @return TRUE if the call was successful,
 *         or if wlancond/btcond does not respond at all
 *         FALSE if the call failed
 */
gboolean get_connectivity_status(gboolean *connected)
{
	DBusMessage *msg = NULL;
	DBusMessage *reply = NULL;
	dbus_bool_t status;
	gchar **array;
	dbus_int32_t length;
	DBusError error;

	/* Register error channel */
	dbus_error_init(&error);

	*connected = FALSE;

	mce_log(LL_DEBUG, "Querying WLAN connection status");

	/* Query whether wlan has open connections or not */
	msg = dbus_new_method_call(WLANCOND_SERVICE,
				   WLANCOND_REQ_PATH,
				   WLANCOND_REQ_INTERFACE,
				   WLANCOND_CONNECTION_STATUS_REQ);

	if ((reply = dbus_send_message_with_reply_and_block(msg)) == NULL) {
		mce_log(LL_ERR,
			"Cannot call method %s with reply; "
			"assuming connection active",
			WLANCOND_CONNECTION_STATUS_REQ);
		*connected = TRUE;
		goto EXIT;
	}

	if (dbus_message_get_args(reply, &error,
				  DBUS_TYPE_BOOLEAN, &status,
				  DBUS_TYPE_INVALID) == FALSE) {
		mce_log(LL_ERR,
			"Failed to get reply argument from %s: %s; %s",
			"WLAN connection",
			error.message,
			"assuming connection active");
		dbus_message_unref(reply);
		dbus_error_free(&error);
		*connected = TRUE;
		goto EXIT;
	}

	dbus_message_unref(reply);

	if (status == TRUE) {
		mce_log(LL_INFO, "WLAN connection found");
		*connected = TRUE;
		goto EXIT;
	}

	/* No WLAN connections open; check bluetooth */

	mce_log(LL_DEBUG, "Querying BT connection status");

	/* Query whether bluetooth has open connections or not */
	msg = dbus_new_method_call(BTCOND_SERVICE,
				   BTCOND_REQ_PATH,
				   BTCOND_REQ_INTERFACE,
				   BTCOND_GET_BDA_LIST_REQ);

	if ((reply = dbus_send_message_with_reply_and_block(msg)) == NULL) {
		mce_log(LL_ERR,
			"Cannot call method %s with reply; "
			"assuming connection active",
			 BTCOND_GET_BDA_LIST_REQ);
		*connected = TRUE;
		goto EXIT;
	}

	if (dbus_message_get_args(reply, &error,
				  DBUS_TYPE_ARRAY, DBUS_TYPE_STRING,
				  &array, &length,
				  DBUS_TYPE_INVALID) == FALSE) {
		mce_log(LL_ERR,
			"Failed to get reply argument from %s: %s; %s",
			"BT connection",
			error.message,
			"assuming connection active");
		dbus_message_unref(reply);
		dbus_error_free(&error);
		*connected = TRUE;
		goto EXIT;
	}

	while (--length >= 0) {
		DBusMessage *reply2;
		gchar *string;
		gchar *btaddr;

		btaddr = array[length];

		msg = dbus_new_method_call(BTCOND_SERVICE,
					   BTCOND_REQ_PATH,
					   BTCOND_REQ_INTERFACE,
					   BTCOND_CONNECTION_STATUS_REQ);

		if (dbus_message_append_args(msg,
					     DBUS_TYPE_STRING, &btaddr,
					     DBUS_TYPE_INVALID) == FALSE) {
			mce_log(LL_CRIT,
				"Failed to append argument to D-Bus message "
				"for %s",
				BTCOND_CONNECTION_STATUS_REQ);
			dbus_message_unref(reply);
			dbus_message_unref(msg);
			return FALSE;
		}

		if ((reply2 = dbus_send_message_with_reply_and_block(msg)) == NULL) {
			mce_log(LL_ERR,
				"Cannot call method %s with reply; "
				"assuming connection active",
				BTCOND_CONNECTION_STATUS_REQ);
			dbus_message_unref(reply);
			*connected = TRUE;
			goto EXIT;
		}

		if (dbus_message_get_args(reply2, &error,
					  DBUS_TYPE_STRING, &string,
					  DBUS_TYPE_INVALID) == FALSE) {
			mce_log(LL_ERR,
				"Failed to get reply argument from %s: %s; %s",
				"BT connection",
				error.message,
				"assuming connection active");
			dbus_message_unref(reply2);
			dbus_message_unref(reply);
			dbus_error_free(&error);
			*connected = TRUE;
			goto EXIT;
		}

		if (strcmp(string, "connected") == 0) {
			mce_log(LL_INFO, "BT connection found on %s", btaddr);
			dbus_message_unref(reply2);
			dbus_message_unref(reply);
			*connected = TRUE;
			goto EXIT;
		}

		dbus_message_unref(reply2);
	}

	dbus_free_string_array(array);
	dbus_message_unref(reply);

EXIT:
	return TRUE;
}
