/**
 * @file test_mce-log.c
 * Module test for mce-log
 * <p>
 * Copyright © 2007 Nokia Corporation.  All rights reserved.
 * <p>
 * @author David Weinehall <david.weinehall@nokia.com>
 */
#include <glib.h>
#include <errno.h>			/* errno */
#include <stdio.h>			/* printf(),
					 * fgets(), freopen(),
					 * fseek(), ftell(),
					 * stderr
					 */
#include <string.h>			/* strcmp(), strlen() */

#include "mce-log.h"			/* mce_log_open(), mce_log_close(),
					 * mce_log_set_verbosity(), mce_log(),
					 * LL_*
					 */

#define PRG_NAME			"test_mce-log"

#define MCE_LOG_LOGFILE			G_STRINGIFY(MCE_BUILD_TEST_PATH) "/tmp/logfile"

#define BUFSIZE				1024

static FILE *logfile = NULL;

/**
 * Display expect information and keep count of test cases
 *
 * @param test The result of a test
 * @return 0 on success, non-zero on failure
 */
static gint test_expect(gint test)
{
	static gint testcase = 1;
	gint retval = 0;

	if (!test) {
		printf("[NOT OK]\n");
		retval = testcase;
	} else {
		printf("[OK]\n");
		testcase++;
	}

	return retval;
}

/**
 * Run module tests
 *
 * @return 0 on success, the first failed test on failure
 */
static int run_tests(void)
{
	char output[BUFSIZE];
	long pos, oldpos;
	gint retval;

	/* Test #1:
	 * Log an LL_DEBUG message at LL_CRIT verbosity
	 * Expected output: none
	 */
	printf("Attempting to log LL_DEBUG message at LL_CRIT verbosity: ");
	mce_log_set_verbosity(LL_CRIT);
	oldpos = ftell(logfile);
	mce_log(LL_DEBUG, "Test #1");
	pos = ftell(logfile);

	if ((retval = test_expect(pos == oldpos)))
		goto EXIT;

	/* Test #2:
	 * Log an LL_DEBUG message at maximum verbosity
	 * Expected output: "Test #2"
	 */
	printf("Attempting to log LL_DEBUG message at LL_DEBUG verbosity: ");
	mce_log_set_verbosity(LL_DEBUG);
	oldpos = ftell(logfile);
	mce_log(LL_DEBUG, "Test #2");
	pos = ftell(logfile);
	fseek(logfile, oldpos, SEEK_SET);
	fgets(output, BUFSIZE, logfile);

	if ((retval = test_expect(!strcmp(output, PRG_NAME ": Test #2\n"))))
		goto EXIT;

	/* Test #3:
	 * Log an LL_CRIT message at LL_CRIT verbosity
	 * Expected output: "Test #3"
	 */
	printf("Attempting to log LL_CRIT message at LL_CRIT verbosity: ");
	mce_log_set_verbosity(LL_CRIT);
	oldpos = ftell(logfile);
	mce_log(LL_CRIT, "Test #3");
	pos = ftell(logfile);
	fseek(logfile, oldpos, SEEK_SET);
	fgets(output, BUFSIZE, logfile);

	if ((retval = test_expect(!strcmp(output, PRG_NAME ": Test #3\n"))))
		goto EXIT;

	/* Test #4:
	 * Log an LL_CRIT message at LL_DEBUG verbosity
	 * Expected output: "Test #4"
	 */
	printf("Attempting to log LL_CRIT message at LL_DEBUG verbosity: ");
	mce_log_set_verbosity(LL_CRIT);
	oldpos = ftell(logfile);
	mce_log(LL_CRIT, "Test #4");
	pos = ftell(logfile);
	fseek(logfile, oldpos, SEEK_SET);
	fgets(output, BUFSIZE, logfile);

	if ((retval = test_expect(!strcmp(output, PRG_NAME ": Test #4\n"))))
		goto EXIT;

	/* Test #5:
	 * Log an LL_WARN message at LL_ERR verbosity
	 * Expected output: none
	 */
	printf("Attempting to log LL_WARN message at LL_ERR verbosity: ");
	mce_log_set_verbosity(LL_ERR);
	oldpos = ftell(logfile);
	mce_log(LL_WARN, "Test #5");
	pos = ftell(logfile);
	fseek(logfile, oldpos, SEEK_SET);

	if ((retval = test_expect(pos == oldpos)))
		goto EXIT;

	/* Test #6:
	 * Log an LL_WARN message at LL_INFO verbosity
	 * Expected output: "Test #6"
	 */
	printf("Attempting to log LL_WARN message at LL_INFO verbosity: ");
	mce_log_set_verbosity(LL_INFO);
	oldpos = ftell(logfile);
	mce_log(LL_WARN, "Test #6");
	pos = ftell(logfile);
	fseek(logfile, oldpos, SEEK_SET);
	fgets(output, BUFSIZE, logfile);

	if ((retval = test_expect(!strcmp(output, PRG_NAME ": Test #6\n"))))
		goto EXIT;

	/* Test #7:
	 * Log an LL_DEBUG message at (LL_DEBUG + 1) verbosity
	 * Expected output: "Test #7"
	 */
	printf("Attempting to log LL_DEBUG message at "
	       "(LL_DEBUG + 1) verbosity: ");
	mce_log_set_verbosity(LL_DEBUG + 1);
	oldpos = ftell(logfile);
	mce_log(LL_WARN, "Test #7");
	pos = ftell(logfile);
	fseek(logfile, oldpos, SEEK_SET);
	fgets(output, BUFSIZE, logfile);

	if ((retval = test_expect(!strcmp(output, PRG_NAME ": Test #7\n"))))
		goto EXIT;

	/* Test #8:
	 * Log an LL_CRIT message at LL_NONE verbosity
	 * Expected output: none
	 */
	printf("Attempting to log LL_CRIT message at LL_NONE verbosity: ");
	mce_log_set_verbosity(LL_NONE);
	oldpos = ftell(logfile);
	mce_log(LL_WARN, "Test #8");
	pos = ftell(logfile);
	fseek(logfile, oldpos, SEEK_SET);
	fgets(output, BUFSIZE, logfile);

	if ((retval = test_expect(pos == oldpos)))
		goto EXIT;

EXIT:
	return retval;
}

/**
 * Main
 *
 * @param argc Number of command line arguments
 * @param argv Array with command line arguments
 * @return 0 on success, the first failed test-case on failure
 */
int main(int argc, char **argv)
{
	int retval;

	(void)argc;
	(void)argv;

	/* Reopen stderr as a regular, R/W file */
	if ((logfile = freopen(MCE_LOG_LOGFILE, "w+", stderr)) == NULL) {
		/* We cannot trust stderr here, since we failed to reopen it */
		printf("Failed to reopen stderr as `%s'; %s. Aborting.\n",
		       MCE_LOG_LOGFILE, g_strerror(errno));

		/* Negative errno, to avoid confusion with test results */
		retval = -errno;
		goto EXIT;
	}

	/* Setup logging */
	mce_log_open(PRG_NAME, LOG_USER, MCE_LOG_STDERR);

	/* Set maximum verbosity */
	mce_log_set_verbosity(LL_DEBUG);

	retval = run_tests();

	printf("\n");

	if (retval == 0) {
		printf("All tests completed successfully");
	} else {
		printf("First failing test: #%d", retval);
	}

	printf("\n");

	/* Close the logfile */
	mce_log_close();

EXIT:
	return retval;
}
