/**
 * @file test_mce-io.c
 * Module test for mce-io
 * <p>
 * Copyright © 2007 Nokia Corporation.  All rights reserved.
 * <p>
 * @author David Weinehall <david.weinehall@nokia.com>
 */
#include <glib.h>

#include <errno.h>			/* errno */
#include <fcntl.h>			/* open() */
#include <stdio.h>			/* printf() */
#include <string.h>			/* strcmp() */

#include "mce-log.h"			/* mce_log_open(), mce_log_close(),
					 * mce_log_set_verbosity(), LL_*
					 */
#include "mce-io.h"			/* mce_write_string_to_file(),
					 * mce_write_number_string_to_file(),
					 * mce_read_string_from_file(),
					 * mce_read_number_string_from_file(),
					 * mce_register_io_monitor_chunk(),
					 * mce_unregister_io_monitor()
					 */

#define TEST_STRING			"Foo"
/* A 64-bit unsigned long + 1 should do the trick... */
#define TEST_STRING_TOO_BIG		"18446744073709551616"
#define TEST_NUMBER			42UL
#define TEST_FILE			G_STRINGIFY(MCE_BUILD_TEST_PATH) "/tmp/testfile"
#define TEST_DIRECTORY			G_STRINGIFY(MCE_BUILD_TEST_PATH) "/tmp/"

#define PRG_NAME			"test_mce-io"

/** Used by callbacks to pass test results */
static gboolean test_result = FALSE;

/** Used by callbacks to indicate updates */
static gboolean file_updated = FALSE;

/**
 * Display expect information and keep count of test cases
 *
 * @param test The result of a test
 * @return 0 on success, non-zero on failure
 */
static gint test_expect(gint test)
{
	static gint testcase = 1;
	gint retval = 0;

	if (!test) {
		printf("[NOT OK]\n");
		retval = testcase;
	} else {
		printf("[OK]\n");
		testcase++;
	}

	printf("\n");

	return retval;
}

/**
 * I/O monitor callback for activity tests
 *
 * @param data The new data
 * @param bytes_read The number of bytes read
 */
static void test_activity_cb(gpointer data, gsize bytes_read)
{
	(void)data;
	(void)bytes_read;

	file_updated = TRUE;
}

/**
 * I/O monitor callback for string tests
 *
 * @param data The new data
 * @param bytes_read The number of bytes read
 */
static void test_string_cb(gpointer data, gsize bytes_read)
{
	test_result = FALSE;

	if ((data != NULL) && (bytes_read == strlen(TEST_STRING)) &&
	    (!strcmp(data, TEST_STRING))) {
		test_result = TRUE;
	}

	file_updated = TRUE;
}

/**
 * I/O monitor callback for chunk tests
 *
 * @param data The new data
 * @param bytes_read The number of bytes read
 */
static void test_chunk_cb(gpointer data, gsize bytes_read)
{
	gchar *tmp;

	tmp = data;

	test_result = FALSE;

	if ((tmp != NULL) && (bytes_read == strlen(TEST_STRING)) &&
	    (!strcmp(tmp, TEST_STRING))) {
		test_result = TRUE;
	}

	file_updated = TRUE;
}

/**
 * Run module tests
 *
 * @return 0 on success, the first failed test on failure
 */
static int run_tests(void)
{
	gconstpointer iomon_id = NULL;
	gchar *string = NULL;
	gboolean retval;
	gulong number;
	gint status;
	int fd = -1;

	/* Test #1:
	 * Write a string to a file we have permission to write to
	 *
	 * Expected output: TRUE
	 */
	printf("Attempting to write a string (%s) to a testfile (%s): ",
	       TEST_STRING, TEST_FILE);

	retval = mce_write_string_to_file(TEST_FILE, TEST_STRING);

	if ((status = test_expect((retval == TRUE))))
		goto EXIT;

	/* Test #2:
	 * Read a string from a file we have permission to read from
	 *
	 * Expected output: "Foo"
	 */
	printf("Attempting to read a string (%s) from a testfile (%s): ",
	       TEST_STRING, TEST_FILE);

	retval = mce_read_string_from_file(TEST_FILE, &string);

	if ((status = test_expect((retval == TRUE) &&
				  (string != NULL) &&
				  (!strcmp(string, TEST_STRING)))))
		goto EXIT;

	/* Test #3:
	 * Write a string to a path where we will fail to write
	 * (in this case a directory, to make sure we fail even as root)
	 *
	 * Expected output: FALSE
	 */
	printf("Attempting to write a string (%s) "
	       "to an invalid location (%s): ",
	       TEST_STRING, TEST_DIRECTORY);

	retval = mce_write_string_to_file(TEST_DIRECTORY, TEST_STRING);

	/* Clear errno */
	errno = 0;

	if ((status = test_expect((retval == FALSE))))
		goto EXIT;

	/* Test #4:
	 * Read a string from a location we cannot read from
	 * (in this case a directory, to make sure we fail even as root)
	 *
	 * Expected output: FALSE
	 */
	printf("Attempting to read a string "
	       "from an invalid location (%s): ",
	       TEST_DIRECTORY);

	retval = mce_read_string_from_file(TEST_DIRECTORY, &string);

	/* Clear errno */
	errno = 0;

	if ((status = test_expect((retval == FALSE))))
		goto EXIT;

	/* Test #5:
	 * Write a number to a file we have permission to write to
	 *
	 * Expected output: TRUE
	 */
	printf("Attempting to write a string representation of a "
	       "valid number (%lu) to a testfile (%s): ",
	       TEST_NUMBER, TEST_FILE);

	retval = mce_write_number_string_to_file(TEST_FILE, TEST_NUMBER);

	if ((status = test_expect((retval == TRUE))))
		goto EXIT;

	/* Test #6:
	 * Read a valid number from a file we have permission to read from
	 *
	 * Expected output: 42
	 */
	printf("Attempting to read a number representation of a "
	       "numeric string (%lu) from a testfile (%s): ",
	       TEST_NUMBER, TEST_FILE);

	retval = mce_read_number_string_from_file(TEST_FILE, &number);

	if ((status = test_expect((retval == TRUE) &&
				  (number == TEST_NUMBER))))
		goto EXIT;

	/* Test #7:
	 * Write a number to a path where we will fail to write
	 * (in this case a directory, to make sure we fail even as root)
	 *
	 * Expected output: FALSE
	 */
	printf("Attempting to write a string representation of an "
	       "valid number (%lu) to an invalid location (%s): ",
	       TEST_NUMBER, TEST_DIRECTORY);

	retval = mce_write_number_string_to_file(TEST_DIRECTORY, TEST_NUMBER);

	/* Clear errno */
	errno = 0;

	if ((status = test_expect((retval == FALSE))))
		goto EXIT;

	/* Test #8:
	 * Read a number from a location we cannot read from
	 * (in this case a directory, to make sure we fail even as root)
	 *
	 * Expected output: FALSE
	 */
	printf("Attempting to read a number representation of a "
	       "numeric string from an invalid location (%s): ",
	       TEST_DIRECTORY);

	retval = mce_read_number_string_from_file(TEST_DIRECTORY, &number);

	/* Clear errno */
	errno = 0;

	if ((status = test_expect((retval == FALSE))))
		goto EXIT;

	/* Test #9:
	 * Write a string to a file we have permission to write to
	 *
	 * Expected output: TRUE
	 */
	printf("Attempting to write a string (%s) to a testfile (%s): ",
	       TEST_STRING, TEST_FILE);

	retval = mce_write_string_to_file(TEST_FILE, TEST_STRING);

	if ((status = test_expect((retval == TRUE))))
		goto EXIT;

	/* Test #10:
	 * Read an invalid number from a file we have permission to read from
	 *
	 * Expected output: FALSE or 0 depending on implementation
	 */
	printf("Attempting to read a number representation of a "
	       "non-numeric string (%s) from a testfile (%s): ",
	       TEST_STRING, TEST_FILE);

	retval = mce_read_number_string_from_file(TEST_FILE, &number);

	/* Clear errno */
	errno = 0;

	if ((status = test_expect((retval == FALSE) || (number == 0))))
		goto EXIT;

	/* Test #11:
	 * Write a numeric string to a file we have permission to write to
	 *
	 * Expected output: TRUE
	 */
	printf("Attempting to write a numeric string (%s) "
	       "to a testfile (%s): ",
	       TEST_STRING_TOO_BIG, TEST_FILE);

	retval = mce_write_string_to_file(TEST_FILE, TEST_STRING_TOO_BIG);

	if ((status = test_expect((retval == TRUE))))
		goto EXIT;

	/* Test #12:
	 * Read a numeric string > ULONG_MAX
	 * from a file we have permission to read from
	 *
	 * Expected output: FALSE
	 */
	printf("Attempting to read a number representation of a "
	       "numeric string > ULONG_MAX (%s) from a testfile (%s): ",
	       TEST_STRING_TOO_BIG, TEST_FILE);

	retval = mce_read_number_string_from_file(TEST_FILE, &number);

	/* Clear errno */
	errno = 0;

	if ((status = test_expect((retval == FALSE))))
		goto EXIT;

	/* Test #13:
	 * Register a chunk I/O-monitor by file descriptor
	 *
	 * Expected output: a pointer to the I/O-monitor
	 */
	printf("Attempting to register a chunk I/O-monitor by file descriptor "
	       "for a testfile (%s): ",
	       TEST_FILE);

	if ((fd = open(TEST_FILE, O_NONBLOCK | O_RDONLY)) != -1) {
		iomon_id = mce_register_io_monitor_chunk(fd, TEST_FILE, FALSE, TRUE, test_chunk_cb, strlen(TEST_STRING));
	}

	/* Clear errno */
	errno = 0;

	if ((status = test_expect((fd != -1) && (iomon_id != NULL))))
		goto EXIT;

	/** Reset update indication */
	file_updated = FALSE;

	/* Test #14:
	 * Write a string to a monitored file we have permission to write to
	 *
	 * Expected output: TRUE
	 */
	printf("Attempting to write a string (%s) "
	       "to a monitored testfile (%s): ",
	       TEST_STRING, TEST_FILE);

	retval = mce_write_string_to_file(TEST_FILE, TEST_STRING);

#if 0
	if ((status = test_expect((retval == TRUE))))
		goto EXIT;

	/* Test #15:
	 * Check whether we triggered the I/O-monitor or not
	 *
	 * Expected output: TRUE && TRUE
	 */
	printf("Checking if the I/O-monitor triggered with correct content: ");

	if ((status = test_expect((file_updated == TRUE) &&
				  (test_result == TRUE))))
		goto EXIT;

	/** Reset update indication */
	file_updated = FALSE;
#endif

	/* Unregister the I/O monitor */
	mce_unregister_io_monitor(iomon_id);

#if 0
	/* Test #16:
	 * Write a string to a file that we have unregistered
	 * the I/O monitor for
	 *
	 * Expected output: TRUE
	 */
	printf("Attempting to write a string (%s) "
	       "to a no longer monitored testfile (%s): ",
	       TEST_STRING, TEST_FILE);

	retval = mce_write_string_to_file(TEST_FILE, TEST_STRING);

	if ((status = test_expect((retval == TRUE))))
		goto EXIT;
#endif

	/* Test #17:
	 * Check whether we triggered the I/O-monitor or not
	 *
	 * Expected output: FALSE
	 */
	printf("Checking if the I/O-monitor triggered: ");

	if ((status = test_expect((file_updated == FALSE))))
		goto EXIT;

	/* Test #18:
	 * Register a string I/O-monitor by file descriptor
	 *
	 * Expected output: a pointer to the I/O-monitor
	 */
	printf("Attempting to register a string I/O-monitor "
	       "by file descriptor for a testfile (%s): ",
	       TEST_FILE);

	if ((fd = open(TEST_FILE, O_NONBLOCK | O_RDONLY)) != -1) {
		iomon_id = mce_register_io_monitor_string(fd, TEST_FILE, FALSE, TRUE, test_string_cb);
	}

	/* Clear errno */
	errno = 0;

	if ((status = test_expect((fd != -1) && (iomon_id != NULL))))
		goto EXIT;

	/** Reset update indication */
	file_updated = FALSE;

	/* Test #19:
	 * Write a string to a monitored file we have permission to write to
	 *
	 * Expected output: TRUE
	 */
	printf("Attempting to write a string (%s) "
	       "to a monitored testfile (%s): ",
	       TEST_STRING, TEST_FILE);

	retval = mce_write_string_to_file(TEST_FILE, TEST_STRING);

	if ((status = test_expect((retval == TRUE))))
		goto EXIT;

#if 0
	/* Test #20:
	 * Check whether we triggered the I/O-monitor or not
	 *
	 * Expected output: TRUE && TRUE
	 */
	printf("Checking if the I/O-monitor triggered with correct content: ");

	if ((status = test_expect((file_updated == TRUE) &&
				  (test_result == TRUE))))
		goto EXIT;

	/** Reset update indication */
	file_updated = FALSE;
#endif

	/* Unregister the I/O monitor */
	mce_unregister_io_monitor(iomon_id);

#if 0
	/* Test #21:
	 * Write a string to a file that we have unregistered
	 * the I/O monitor for
	 *
	 * Expected output: TRUE
	 */
	printf("Attempting to write a string (%s) "
	       "to a no longer monitored testfile (%s): ",
	       TEST_STRING, TEST_FILE);

	retval = mce_write_string_to_file(TEST_FILE, TEST_STRING);

	if ((status = test_expect((retval == TRUE))))
		goto EXIT;
#endif

	/* Test #22:
	 * Check whether we triggered the I/O-monitor or not
	 *
	 * Expected output: FALSE
	 */
	printf("Checking if the I/O-monitor triggered: ");

	if ((status = test_expect((file_updated == FALSE))))
		goto EXIT;


	/* Test #23:
	 * Register an activity I/O-monitor by file descriptor
	 *
	 * Expected output: a pointer to the I/O-monitor
	 */
	printf("Attempting to register an activity I/O-monitor "
	       "by file descriptor for a testfile (%s): ",
	       TEST_FILE);

	if ((fd = open(TEST_FILE, O_NONBLOCK | O_RDONLY)) != -1) {
		iomon_id = mce_register_io_monitor_activity(fd, TEST_FILE, FALSE, test_activity_cb);
	}

	/* Clear errno */
	errno = 0;

	if ((status = test_expect((fd != -1) && (iomon_id != NULL))))
		goto EXIT;

	/** Reset update indication */
	file_updated = FALSE;

	/* Test #24:
	 * Write a string to a monitored file we have permission to write to
	 *
	 * Expected output: TRUE
	 */
	printf("Attempting to write a string (%s) "
	       "to a monitored testfile (%s): ",
	       TEST_STRING, TEST_FILE);

	retval = mce_write_string_to_file(TEST_FILE, TEST_STRING);

	if ((status = test_expect((retval == TRUE))))
		goto EXIT;

#if 0
	/* Test #25:
	 * Check whether we triggered the I/O-monitor or not
	 *
	 * Expected output: TRUE
	 */
	printf("Checking if the I/O-monitor triggered: ");

	if ((status = test_expect((file_updated == TRUE))))
		goto EXIT;

	/** Reset update indication */
	file_updated = FALSE;
#endif

	/* Unregister the I/O monitor */
	mce_unregister_io_monitor(iomon_id);

#if 0
	/* Test #26:
	 * Write a string to a file that we have unregistered
	 * the I/O monitor for
	 *
	 * Expected output: TRUE
	 */
	printf("Attempting to write a string (%s) "
	       "to a no longer monitored testfile (%s): ",
	       TEST_STRING, TEST_FILE);

	retval = mce_write_string_to_file(TEST_FILE, TEST_STRING);

	if ((status = test_expect((retval == TRUE))))
		goto EXIT;
#endif

	/* Test #27:
	 * Check whether we triggered the I/O-monitor or not
	 *
	 * Expected output: FALSE
	 */
	printf("Checking if the I/O-monitor triggered: ");

	if ((status = test_expect((file_updated == FALSE))))
		goto EXIT;

// XXX: missing tests: the same set of tests but for filenames rather
//                     than file descriptors
EXIT:
	return status;
}

/**
 * Main
 *
 * @param argc Number of command line arguments
 * @param argv Array with command line arguments
 * @return 0 on success, the first failed test-case on failure
 */
int main(int argc, char **argv)
{
	int retval;

	(void)argc;
	(void)argv;

	/* Setup logging */
	mce_log_open(PRG_NAME, LOG_USER, MCE_LOG_STDERR);

	/* Set maximum verbosity */
	mce_log_set_verbosity(LL_DEBUG);

	retval = run_tests();

	printf("\n");

	if (retval == 0) {
		printf("All tests completed successfully");
	} else {
		printf("First failing test: #%d", retval);
	}

	printf("\n");

	/* Close the logfile */
	mce_log_close();

	return retval;
}
