/**
 * @file test_mce-conf.c
 * Module test for mce-conf
 * <p>
 * Copyright © 2007 Nokia Corporation.  All rights reserved.
 * <p>
 * @author David Weinehall <david.weinehall@nokia.com>
 */
#include <glib.h>

#include <stdio.h>			/* printf() */
#include <string.h>			/* strcmp() */

#include "mce-log.h"			/* mce_log_open(), mce_log_close(),
					 * mce_log_set_verbosity(), LL_*
					 */
#include "mce-conf.h"			/* mce_conf_get_bool(),
					 * mce_conf_get_int(),
					 * mce_conf_get_int_list(),
					 * mce_conf_get_string(),
					 * mce_conf_get_string_list(),
					 * mce_conf_read_conf_file(),
					 * mce_conf_free_conf_file()
					 */

#define MCE_CONF_TEST_FILE			"build-tests/test.ini"
#define MCE_CONF_TEST_GROUP			"test"
#define MCE_CONF_BOOL_TEST_FALSE		"BoolTestFalse"
#define MCE_CONF_BOOL_TEST_TRUE			"BoolTestTrue"
#define MCE_CONF_BOOL_TEST_INVALID		"BoolTestInvalid"
#define MCE_CONF_INT_TEST			"IntTest"
#define MCE_CONF_INT_TEST_INVALID		"IntTestInvalid"
#define MCE_CONF_INT_LIST_TEST			"IntListTest"
#define MCE_CONF_INT_LIST_TEST_INVALID		"IntListTestInvalid"
#define MCE_CONF_STRING_TEST			"StringTest"
#define MCE_CONF_STRING_TEST_INVALID		"StringTestInvalid"
#define MCE_CONF_STRING_LIST_TEST		"StringListTest"
#define MCE_CONF_STRING_LIST_TEST_INVALID	"StringListTestInvalid"

#define MCE_CONF_NONEXISTING			"NONEXISTING"

#define PRG_NAME				"test_mce-conf"

/**
 * Display expect information and keep count of test cases
 *
 * @param test The result of a test
 * @return 0 on success, non-zero on failure
 */
static gint test_expect(gint test)
{
	static gint testcase = 1;
	gint retval = 0;

	if (!test) {
		printf("[NOT OK]\n");
		retval = testcase;
	} else {
		printf("[OK]\n");
		testcase++;
	}

	return retval;
}

/**
 * Run module tests
 *
 * @return 0 on success, the first failed test on failure
 */
static int run_tests(void)
{
	gpointer keyfile;
	gboolean booltmp;
	gint retval;
	gint inttmp;
	gint *intlisttmp;
	gchar *stringtmp;
	gchar **stringlisttmp;
	gsize length;

	/* Initialisation */

	/* Test #1:
	 * Initialise the mce-conf component with a non-existing file
	 *
	 * Expected output: NULL
	 */
	printf("Attempting to read non-existing keyfile: ");

	keyfile = mce_conf_read_conf_file(MCE_CONF_NONEXISTING);

	if ((retval = test_expect((keyfile == NULL))))
		goto EXIT;

	/* Test #2:
	 * Initialise the mce-conf component
	 *
	 * Expected output: A keyfile pointer
	 */
	printf("Attempting to read existing keyfile (%s): ",
	       MCE_CONF_TEST_FILE);

	keyfile = mce_conf_read_conf_file(MCE_CONF_TEST_FILE);

	if ((retval = test_expect((keyfile != NULL))))
		goto EXIT;


	/* Booleans */

	/* Test #3:
	 * Get boolean from non-existing group; default == FALSE
	 *
	 * Expected output: FALSE
	 */
	printf("Attempting to get boolean from non-existing group "
	       "(default: FALSE): ");

	booltmp = mce_conf_get_bool(MCE_CONF_NONEXISTING,
				    MCE_CONF_BOOL_TEST_TRUE,
				    FALSE, keyfile);

	if ((retval = test_expect((booltmp == FALSE))))
		goto EXIT;

	/* Test #4:
	 * Get boolean from non-existing group; default == TRUE
	 *
	 * Expected output: TRUE
	 */
	printf("Attempting to get boolean from non-existing group "
	       "(default: TRUE): ");

	booltmp = mce_conf_get_bool(MCE_CONF_NONEXISTING,
				    MCE_CONF_BOOL_TEST_FALSE,
				    TRUE, keyfile);

	if ((retval = test_expect((booltmp == TRUE))))
		goto EXIT;

	/* Test #5:
	 * Get non-existing boolean; default == FALSE
	 *
	 * Expected output: FALSE
	 */
	printf("Attempting to get non-existing boolean "
	       "(default: FALSE): ");

	booltmp = mce_conf_get_bool(MCE_CONF_TEST_GROUP,
				    MCE_CONF_NONEXISTING,
				    FALSE, keyfile);

	if ((retval = test_expect((booltmp == FALSE))))
		goto EXIT;

	/* Test #6:
	 * Get non-existing boolean; default == TRUE
	 *
	 * Expected output: TRUE
	 */
	printf("Attempting to get non-existing boolean "
	       "(default: TRUE): ");

	booltmp = mce_conf_get_bool(MCE_CONF_TEST_GROUP,
				    MCE_CONF_NONEXISTING,
				    TRUE, keyfile);

	if ((retval = test_expect((booltmp == TRUE))))
		goto EXIT;

	/* Test #7:
	 * Get existing boolean (FALSE); default == TRUE
	 *
	 * Expected output: FALSE
	 */
	printf("Attempting to get existing boolean (FALSE) "
	       "(default: TRUE): ");

	booltmp = mce_conf_get_bool(MCE_CONF_TEST_GROUP,
				    MCE_CONF_BOOL_TEST_FALSE,
				    TRUE, keyfile);

	if ((retval = test_expect((booltmp == FALSE))))
		goto EXIT;

	/* Test #8:
	 * Get existing boolean (TRUE); default == FALSE
	 *
	 * Expected output: TRUE
	 */
	printf("Attempting to get existing boolean (TRUE) "
	       "(default: FALSE): ");

	booltmp = mce_conf_get_bool(MCE_CONF_TEST_GROUP,
				    MCE_CONF_BOOL_TEST_TRUE,
				    FALSE, keyfile);

	if ((retval = test_expect((booltmp == TRUE))))
		goto EXIT;


	/* Test #9:
	 * Get invalid boolean; default == FALSE
	 *
	 * Expected output: FALSE
	 */
	printf("Attempting to get invalid boolean "
	       "(default: FALSE): ");

	booltmp = mce_conf_get_bool(MCE_CONF_TEST_GROUP,
				    MCE_CONF_BOOL_TEST_INVALID,
				    FALSE, keyfile);

	if ((retval = test_expect((booltmp == FALSE))))
		goto EXIT;

	/* Test #10:
	 * Get invalid boolean; default == TRUE
	 *
	 * Expected output: TRUE
	 */
	printf("Attempting to get invalid boolean "
	       "(default: TRUE): ");

	booltmp = mce_conf_get_bool(MCE_CONF_TEST_GROUP,
				    MCE_CONF_BOOL_TEST_INVALID,
				    TRUE, keyfile);

	if ((retval = test_expect((booltmp == TRUE))))
		goto EXIT;


	/* Integers */

	/* Test #11:
	 * Get integer from non-existing group; default == 64
	 *
	 * Expected output: 64
	 */
	printf("Attempting to get integer from non-existing group "
	       "(default: 64): ");

	inttmp = mce_conf_get_int(MCE_CONF_NONEXISTING,
				  MCE_CONF_INT_TEST,
				  64, keyfile);

	if ((retval = test_expect((inttmp == 64))))
		goto EXIT;

	/* Test #12:
	 * Get non-existing integer; default == 128
	 *
	 * Expected output: 128
	 */
	printf("Attempting to get non-existing integer "
	       "(default: 128): ");

	inttmp = mce_conf_get_int(MCE_CONF_TEST_GROUP,
				  MCE_CONF_NONEXISTING,
				  128, keyfile);

	if ((retval = test_expect((inttmp == 128))))
		goto EXIT;

	/* Test #13:
	 * Get existing integer (42); default == 6581
	 *
	 * Expected output: 42
	 */
	printf("Attempting to get existing integer (42) "
	       "(default: 6581): ");

	inttmp = mce_conf_get_int(MCE_CONF_TEST_GROUP,
				  MCE_CONF_INT_TEST,
				  6581, keyfile);

	if ((retval = test_expect((inttmp == 42))))
		goto EXIT;

	/* Test #14:
	 * Get invalid integer; default == 6581
	 *
	 * Expected output: 6581
	 */
	printf("Attempting to get invalid integer "
	       "(default: 6581): ");

	inttmp = mce_conf_get_int(MCE_CONF_TEST_GROUP,
				  MCE_CONF_INT_TEST_INVALID,
				  6581, keyfile);

	if ((retval = test_expect((inttmp == 6581))))
		goto EXIT;


	/* Integer lists */

	/* Test #15:
	 * Get integer list from non-existing group
	 *
	 * Expected output: NULL && length == 0
	 */
	printf("Attempting to get integer list from non-existing group: ");

	/* Poison length */
	length = 666;

	intlisttmp = mce_conf_get_int_list(MCE_CONF_NONEXISTING,
					   MCE_CONF_INT_TEST,
					   &length, keyfile);

	if ((retval = test_expect(((intlisttmp == NULL) && (length == 0)))))
		goto EXIT;

	/* Test #16:
	 * Get non-existing integer list
	 *
	 * Expected output: NULL && length == 0
	 */
	printf("Attempting to get non-existing integer list: ");

	/* Poison length */
	length = 666;

	intlisttmp = mce_conf_get_int_list(MCE_CONF_TEST_GROUP,
					   MCE_CONF_NONEXISTING,
					   &length, keyfile);

	if ((retval = test_expect(((intlisttmp == NULL) && (length == 0)))))
		goto EXIT;

	/* Test #17:
	 * Get existing integer list (1;1;2;3;5;8)
	 *
	 * Expected output: intlisttmp == "1;1;2;3;5;8" && length == 6
	 */
	printf("Attempting to get existing integer list (1;1;2;3;5;8): ");

	/* Poison length */
	length = 666;

	intlisttmp = mce_conf_get_int_list(MCE_CONF_TEST_GROUP,
					   MCE_CONF_INT_LIST_TEST,
					   &length, keyfile);

	if ((retval = test_expect(((length == 6) &&
				   (intlisttmp[0] == 1) &&
				   (intlisttmp[1] == 1) &&
				   (intlisttmp[2] == 2) &&
				   (intlisttmp[3] == 3) &&
				   (intlisttmp[4] == 5) &&
				   (intlisttmp[5] == 8)))))
		goto EXIT;


	/* Test #18:
	 * Get invalid integer list (1,1,2,3,5,8)
	 *
	 * Expected output: NULL && length == 0
	 */
	printf("Attempting to get invalid integer list (1,1,2,3,5,8): ");

	/* Poison length */
	length = 666;

	intlisttmp = mce_conf_get_int_list(MCE_CONF_TEST_GROUP,
					   MCE_CONF_INT_LIST_TEST_INVALID,
					   &length, keyfile);

	if ((retval = test_expect(((intlisttmp == NULL) && (length == 0)))))
		goto EXIT;


	/* Strings */

	/* Test #19:
	 * Get string from non-existing group; default not set
	 *
	 * Expected output: NULL
	 */
	printf("Attempting to get string from non-existing group "
	       "(default not set): ");

	stringtmp = mce_conf_get_string(MCE_CONF_NONEXISTING,
					MCE_CONF_STRING_TEST,
					NULL, keyfile);

	if ((retval = test_expect((stringtmp == NULL))))
		goto EXIT;

	/* Test #20:
	 * Get string from non-existing group; default "Nokia"
	 *
	 * Expected output: "Nokia"
	 */
	printf("Attempting to get string from non-existing group "
	       "(default: \"Nokia\"): ");

	stringtmp = mce_conf_get_string(MCE_CONF_NONEXISTING,
					MCE_CONF_STRING_TEST,
					"Nokia", keyfile);

	if ((retval = test_expect((!strcmp(stringtmp, "Nokia")))))
		goto EXIT;

	/* Test #21:
	 * Get non-existing string; default not set
	 *
	 * Expected output: NULL
	 */
	printf("Attempting to get non-existing string "
	       "(default not set): ");

	stringtmp = mce_conf_get_string(MCE_CONF_TEST_GROUP,
					MCE_CONF_NONEXISTING,
					NULL, keyfile);

	if ((retval = test_expect((stringtmp == NULL))))
		goto EXIT;

	/* Test #22:
	 * Get non-existing string; default "Nokia"
	 *
	 * Expected output: "Nokia"
	 */
	printf("Attempting to get non-existing string: "
	       "(default: \"Nokia\"): ");

	stringtmp = mce_conf_get_string(MCE_CONF_TEST_GROUP,
					MCE_CONF_NONEXISTING,
					"Nokia", keyfile);

	if ((retval = test_expect((!strcmp(stringtmp, "Nokia")))))
		goto EXIT;

	/* Test #23:
	 * Get existing string ("foo"); default not set
	 *
	 * Expected output: "foo"
	 */
	printf("Attempting to get existing string (\"foo\") "
	       "(default not set): ");

	stringtmp = mce_conf_get_string(MCE_CONF_TEST_GROUP,
					MCE_CONF_STRING_TEST,
					NULL, keyfile);

	if ((retval = test_expect((!strcmp(stringtmp, "foo")))))
		goto EXIT;

	/* Test #24:
	 * Get existing string ("foo"); default "Nokia"
	 *
	 * Expected output: "foo"
	 */
	printf("Attempting to get existing string (\"foo\") "
	       "(default: \"Nokia\"): ");

	stringtmp = mce_conf_get_string(MCE_CONF_TEST_GROUP,
					MCE_CONF_STRING_TEST,
					"Nokia", keyfile);

	if ((retval = test_expect((!strcmp(stringtmp, "foo")))))
		goto EXIT;

	/* Test #25:
	 * Get invalid string ("<e5>"); default not set
	 *
	 * Expected output: NULL
	 */
	printf("Attempting to get invalid string (\"<e5>\") "
	       "(default not set): ");

	stringtmp = mce_conf_get_string(MCE_CONF_TEST_GROUP,
					MCE_CONF_STRING_TEST_INVALID,
					NULL, keyfile);

	if ((retval = test_expect((stringtmp == NULL))))
		goto EXIT;

	/* Test #26:
	 * Get invalid string ("<e5>"); default "Nokia"
	 *
	 * Expected output: "Nokia"
	 */
	printf("Attempting to get invalid string (\"<e5>\") "
	       "(default: \"Nokia\"): ");

	stringtmp = mce_conf_get_string(MCE_CONF_TEST_GROUP,
					MCE_CONF_STRING_TEST_INVALID,
					"Nokia", keyfile);

	if ((retval = test_expect((!strcmp(stringtmp, "Nokia")))))
		goto EXIT;


	/* String lists */

	/* Test #27:
	 * Get string list from non-existing group
	 *
	 * Expected output: NULL && length == 0
	 */
	printf("Attempting to get string list from non-existing group: ");

	/* Poison length */
	length = 666;

	stringlisttmp = mce_conf_get_string_list(MCE_CONF_NONEXISTING,
						 MCE_CONF_STRING_LIST_TEST,
						 &length, keyfile);

	if ((retval = test_expect(((stringlisttmp == NULL) && (length == 0)))))
		goto EXIT;

	/* Test #28:
	 * Get non-existing string list
	 *
	 * Expected output: NULL && length == 0
	 */
	printf("Attempting to get non-existing string list: ");

	/* Poison length */
	length = 666;

	stringlisttmp = mce_conf_get_string_list(MCE_CONF_TEST_GROUP,
						 MCE_CONF_NONEXISTING,
						 &length, keyfile);

	if ((retval = test_expect(((stringlisttmp == NULL) && (length == 0)))))
		goto EXIT;

	/* Test #29:
	 * Get existing string list ("bar;baz")
	 *
	 * Expected output: "bar;baz"
	 */
	printf("Attempting to get existing string list (\"bar;baz\"): ");

	/* Poison length */
	length = 666;

	stringlisttmp = mce_conf_get_string_list(MCE_CONF_TEST_GROUP,
						 MCE_CONF_STRING_LIST_TEST,
						 &length, keyfile);

	if ((retval = test_expect(((length == 2) &&
				   (!strcmp(stringlisttmp[0], "bar")) &&
				   (!strcmp(stringlisttmp[1], "baz"))))))
		goto EXIT;

	/* Test #30:
	 * Get invalid string list ("<e5>")
	 *
	 * Expected output: NULL && length == 0
	 */
	printf("Attempting to get invalid string list (\"<e5>\"): ");

	/* Poison length */
	length = 666;

	stringlisttmp = mce_conf_get_string_list(MCE_CONF_TEST_GROUP,
						 MCE_CONF_STRING_LIST_TEST_INVALID,
						 &length, keyfile);

	if ((retval = test_expect(((stringlisttmp == 0) && (length == 0)))))
		goto EXIT;

EXIT:
	return retval;
}

/**
 * Main
 *
 * @param argc Number of command line arguments
 * @param argv Array with command line arguments
 * @return 0 on success, the first failed test-case on failure
 */
int main(int argc, char **argv)
{
	int retval;

	(void)argc;
	(void)argv;

	/* Setup logging */
	mce_log_open(PRG_NAME, LOG_USER, MCE_LOG_STDERR);

	/* Set maximum verbosity */
	mce_log_set_verbosity(LL_DEBUG);

	retval = run_tests();

	printf("\n");

	if (retval == 0) {
		printf("All tests completed successfully");
	} else {
		printf("First failing test: #%d", retval);
	}

	printf("\n");

	/* Close the logfile */
	mce_log_close();

	return retval;
}
