/**
 * @file test_input_event.c
 * Module test for input_event
 * <p>
 * Copyright © 2007 Nokia Corporation.  All rights reserved.
 * <p>
 * @author David Weinehall <david.weinehall@nokia.com>
 */
#include <glib.h>

#include <stdio.h>			/* printf() */
#include <string.h>			/* strcmp() */

#include "mce-log.h"			/* mce_log_open(), mce_log_close(),
					 * mce_log_set_verbosity(), LL_*
					 */
#include "input_event.h"		/* get_event_fd() */

#define INPUT_EVENT_NONEXISTING			"NONEXISTING"
#define INPUT_EVENT_PC_KEYBOARD			"AT Translated Set 2 keyboard"

#define PRG_NAME				"test_input_event"

/**
 * Display expect information and keep count of test cases
 *
 * @param test The result of a test
 * @return 0 on success, non-zero on failure
 */
static gint test_expect(gint test)
{
	static gint testcase = 1;
	gint retval = 0;

	if (!test) {
		printf("[NOT OK]\n");
		retval = testcase;
	} else {
		printf("[OK]\n");
		testcase++;
	}

	return retval;
}

/**
 * Run module tests
 *
 * @return 0 on success, the first failed test on failure
 */
static int run_tests(void)
{
	int eventfd = -1;
	gint retval;

	/* Test #1:
	 * Find a non-existing event driver
	 *
	 * Expected output: -1
	 */
	printf("Attempting to find non-existing input event driver: ");

	eventfd = get_event_fd(INPUT_EVENT_NONEXISTING);

	if ((retval = test_expect((eventfd == -1))))
		goto EXIT;

	/* Test #2:
	 * Find an existing event driver
	 *
	 * Expected output: a valid file descriptor
	 */
	printf("Attempting to find %s input event driver: ",
	       INPUT_EVENT_PC_KEYBOARD);

	eventfd = get_event_fd(INPUT_EVENT_PC_KEYBOARD);

	if ((retval = test_expect((eventfd != -1))))
		goto EXIT;

EXIT:
	return retval;
}

/**
 * Main
 *
 * @param argc Number of command line arguments
 * @param argv Array with command line arguments
 * @return 0 on success, the first failed test-case on failure
 */
int main(int argc, char **argv)
{
	int retval;

	(void)argc;
	(void)argv;

	/* Setup logging */
	mce_log_open(PRG_NAME, LOG_USER, MCE_LOG_STDERR);

	/* Set maximum verbosity */
	mce_log_set_verbosity(LL_DEBUG);

	retval = run_tests();

	printf("\n");

	if (retval == 0) {
		printf("All tests completed successfully");
	} else {
		printf("First failing test: #%d", retval);
	}

	printf("\n");

	/* Close the logfile */
	mce_log_close();

	return retval;
}
