/**
 * Copyright (C) 2007 Nokia. All rights reserved.
 */

#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <unistd.h>
#include <getopt.h>
#include <string.h>
#include <sys/stat.h>
#include <metadata_interface.h>
#include <assert.h>

#include "crawler-interface.h"
#include "crawler-debug.h"
#include "crawler-utils.h"
#include "crawler-volume-monitor.h"


/***********************************************************
 * PRIVATE FUNCTIONS
 */

/**
 * Usage helper function.
 * Prints out the proper usage of metadata crawler command line options etc.
 *
 * @param prog Name of the program
 */
static void
usage (const gchar *prog)
{
  fprintf(stderr, "Usage: %s [OPTIONS]\n\n"
          "-F\t\t\tForce the daemon to run in foreground\n"
          "-d\t\t\tRun the deamon in debug mode\n"
          "-c <custom paths>\tA colon separated list of custom paths\n"
          "-n\t\t\tDon't crawl through all paths when the program is started\n",
          prog);
}


/***********************************************************
 * PUBLIC FUNCTIONS
 */

gint
main (gint argc, gchar **argv)
{
  gint c = 0;
  static AppData app_data;
  gchar *paths = NULL;
  gchar *session_bus_address = NULL;
  MTVal val;
  GMainLoop *loop = NULL;
  GMainContext *context = NULL;
  mmc_t *mmc = NULL;

  if (!g_thread_supported()) 
    {
      g_thread_init(NULL);
    }

  /* This is a horrible kludge. When crawler is started at device bootup,
   * this environment variable is not necessarile set.
   * libgnomevfs needs it though so we set it here
   */
  session_bus_address = getenv("DBUS_SESSION_BUS_ADDRESS");
  if (session_bus_address == NULL || strlen(session_bus_address) == 0)
    {
      setenv("DBUS_SESSION_BUS_ADDRESS",
             "unix:path=/tmp/session_bus_socket",
             1);
    }

  CRAWLER_DEBUG_INFO ("Init app data");
  init_app_data(&app_data);

  opterr = 0;
  /* get command line arguments */
  while ((c = getopt(argc, argv, "cdnF")) != EOF)
    {
      switch (c)
        {
          /* custom paths */
        case 'c':              
          if (optind < argc)
            paths = argv[optind];
          break;

          /* debug mode */
        case 'd':              
          app_data.debug_flag = 1;
          break;

          /* no crawling in the beginning */
        case 'n':              
          app_data.no_crawl_flag = 1;
          break;

          /* run in foreground */
        case 'F':              
          app_data.no_daemon_flag = 1;
          break;

        case '?':
        default:
          usage(argv[0]);
          return 0;
        }
    }

  argc -= optind;
  argv += optind;

  loop = g_main_loop_new(NULL, FALSE);
  context = g_main_loop_get_context(loop);
  app_data.osso = osso_initialize("metalayer_crawler", "0.0", FALSE, context);

  if (!app_data.no_daemon_flag)
    {
      if (!daemonize())
        {
          clean_app_data(&app_data);
          exit(EXIT_FAILURE);
        }
    }

  CRAWLER_DEBUG_INFO("Init inotify");

  if (crawler_init() == -1)
    {
      return -1;
    }

  CRAWLER_DEBUG_INFO("Parse paths");

  /* Add dirs */
  if (!parse_paths(paths == NULL ? DEFAULT_PATHS : paths, &app_data))
    {
      CRAWLER_DEBUG_ERR
        ("Error in parsing paths. Please check the validity of the paths!");
      return -1;
    }

  /* Assume that memory card(s) have no files present. The value
   * of Present key will be set to 1 by the engine for files which
   * are present
   */
  mmc = app_data.mmcs;
  while (mmc->path != NULL)
    {
      meta_path_absent(mmc->path);
      mmc++;
    }
  val.i_value = 1;
  meta_delete_by_key(MT_KEY_UPNP, val, MT_INTEGER);

  volume_monitor_init(&app_data);
  CRAWLER_DEBUG_INFO("Watch MMC");
  gconf_write(&app_data, GCONF_STATE_CRAWLING);
  /* If MMC is already mounted, we should watch it too */
  watch_mmc(&app_data);

  if (!app_data.no_crawl_flag)
    {
      crawl_all(&app_data);
    }
  gconf_write(&app_data, GCONF_STATE_IDLE);

  CRAWLER_DEBUG_INFO("Inotify queue");

  process_inotify_events(&app_data);
  g_main_run(loop);

  volume_monitor_shutdown(&app_data);
  close_listener(&app_data);
  clean_app_data(&app_data);

  return 0;
}
