/**
 * Copyright (C) 2007 Nokia. All rights reserved.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <libgnomevfs/gnome-vfs.h>
#include <libgnomevfs/gnome-vfs-volume-monitor.h>
#include <libgnomevfs/gnome-vfs-volume.h>
#include <metadata_interface.h>

#include <glib.h>   /* Event loop, types */

#include "crawler-interface.h"
#include "crawler-debug.h"
#include "crawler-utils.h"

#include "crawler-volume-monitor.h"

/**
 * Volume mounted handler
 *
 * @param vfsvolumemonitor volume monitor
 * @param volume the mounted volume
 * @param user_data common application data
 */
static void
volume_mounted_handler (GnomeVFSVolumeMonitor *vfsvolumemonitor,
                        GnomeVFSVolume *volume,
                        gpointer user_data)
{
  gchar *device = NULL;
  AppData *app_data = NULL;
  mmc_t *mmc = NULL;

  if (volume == NULL || user_data == NULL)
    {
      return;
    }

  app_data = (AppData *) user_data;

  device = gnome_vfs_volume_get_device_path(volume);

  if (device == NULL)
    {
      return;
    }

  /* Watch the directory that was just mounted */
  mmc = app_data->mmcs;
  while (mmc->path != NULL)
    {
      if (g_ascii_strcasecmp(device, mmc->dev_path) == 0)
	{
	  crawl_watch_dir(app_data, mmc->path);
	} 
      mmc++;
    }

  CRAWLER_DEBUG_INFO("Device mounted: %s", device);
  g_free(device);
}

/**
 * Volume unmounted handler
 *
 * @param vfsvolumemonitor volume monitor
 * @param volume the unmounted volume
 * @param user_data common application data
 */
static void
volume_unmounted_handler (GnomeVFSVolumeMonitor *vfsvolumemonitor,
                          GnomeVFSVolume *volume,
                          gpointer user_data)
{
  gchar *device = NULL;
  AppData *app_data = NULL;
  mmc_t *mmc = NULL;

  if (volume == NULL || user_data == NULL)
    {
      return;
    }

  app_data = (AppData *) user_data;

  device = gnome_vfs_volume_get_device_path(volume);

  if (device == NULL)
    {
      return;
    }

  mmc = app_data->mmcs;
  while (mmc->path != NULL)
    {
      if (g_ascii_strcasecmp(device, mmc->dev_path) == 0)
	{
	  gint wd;
	  
	  gconf_write(app_data, GCONF_STATE_CRAWLING);
	  wd = inotify_data_get_wd(mmc->path, app_data);
	  inotify_rm_watch_dir(wd, app_data);
	  meta_path_absent(mmc->path);
	  gconf_write(app_data, GCONF_STATE_IDLE);
	  CRAWLER_DEBUG("MMC watch removed");
	}
      mmc++;
    }

  CRAWLER_DEBUG_INFO("Device unmounted: %s", device);
  g_free(device);
}

gboolean
volume_monitor_init (AppData *app_data)
{
  if (app_data == NULL)
    {
      return FALSE;
    }

  g_type_init();

  if (!gnome_vfs_init())
    {
      return FALSE; 
    }

  app_data->volume_monitor = gnome_vfs_get_volume_monitor();

  if (app_data->volume_monitor == NULL)
    {
      return FALSE;
    }

  g_signal_connect(G_OBJECT(app_data->volume_monitor), "volume-mounted",
                   G_CALLBACK(volume_mounted_handler), app_data);

  g_signal_connect(G_OBJECT(app_data->volume_monitor), "volume-unmounted",
                   G_CALLBACK(volume_unmounted_handler), app_data);

  return TRUE;
}

gboolean
volume_monitor_shutdown (AppData *app_data)
{
  gnome_vfs_shutdown();

  return TRUE;
}
