/**
 * Copyright (C) 2007 Nokia. All rights reserved.
 */

/** @file crawler_utils.h 
 * \brief Crawler utilities function definitions
 */

#ifndef CRAWLER_UTILS_H
#define CRAWLER_UTILS_H

#include <stdio.h>

#include "crawler-interface.h"

/**
 * Prepend inotify data to the inotify watch list
 *
 * @param path path to add
 * @param wd watch descriptor 
 * @param app_data common application data
 * @return TRUE on success, FALSE otherwise150
 */
gboolean inotify_data_prepend (const gchar *path, gint wd, AppData *app_data);

/**
 * Deletes inotify data from the list that corresponds
 * to wd
 *
 * @param wd watch descriptor to remove
 * @param app_data common application data
 */
gboolean inotify_data_delete (gint wd, AppData *app_data);

/**
 * Searches for a watch descriptor corresponding to a path
 *
 * @param path path to look for
 * @param app_data common application data
 * @return watch descriptor corresponding to the path or -1 of not found
 */
gint inotify_data_get_wd (const gchar *path, AppData *app_data);

/**
 * Searches for a path corresponding to a watch descriptor
 *
 * @param wd watch descriptor to look for
 * @param app_data common application data
 * @return path corresponding to the watch descriptor or NULL if not found
 */
const gchar *inotify_data_get_path (gint wd, AppData *app_data);

/**
 * Parses a path string of format:
 * /path/to/first/directory:/path/to/second/directory:...:/path/to/n/directory
 * and outputs an array of those paths.
 * 
 * @param str path string to parse
 * @param app_data common application data
 * @return TRUE on success, FALSE on failure
 */
gboolean parse_paths (gchar *str, AppData *app_data);

/**
 * Frees an array of paths, but not the array itself
 *
 * @param paths an array of paths to free
 */
void free_paths (gchar **paths);

/**
 * Daemonize the current process
 *
 * @return TRUE on success, FALSE on failure
 */
gboolean daemonize (void);

/**
 * Initializes the AppData struct
 *
 * @param app_data common application data
 */
void init_app_data (AppData *app_data);

/**
 * Removes all INotify watches and frees
 * the memory used by AppData
 *
 * @param app_data common application data
 */
void clean_app_data (AppData *app_data);

/**
 * Frees a NULL terminated array of strings
 *
 * @param vs an array of strings
 */
void vfree (gchar **vs);

/**
 * Changes the GConf key of Crawler state according to the
 * current state
 *
 * @param app_data common application data
 * @param state state to change the GConf key to
 */
void gconf_write (AppData *app_data, const gchar *state);

#endif
