/**
 * Copyright (C) 2007 Nokia. All rights reserved.
 */

/** @file crawler-utils.c 
 * \brief Crawler utilities
 */

#include <unistd.h>
#include <stdlib.h>
#include <assert.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <string.h>

#include "crawler-debug.h"
#include "crawler-interface.h"

#include "crawler-utils.h"

#define HAVE_770

static mmc_t *init_mmcs_array (void);

gboolean
inotify_data_prepend (const gchar *path, gint wd, AppData *app_data)
{
  InotifyData *prev = NULL;

  assert(path != NULL);
  assert(app_data != NULL);

  prev = (InotifyData *) g_malloc(sizeof(InotifyData));

  if (prev == NULL)
    {
      CRAWLER_DEBUG_ERR("Memory allocation failed");
      return FALSE;
    }

  prev->path = g_strdup(path);
  prev->wd = wd;
  prev->next = app_data->inotify_data;
  app_data->inotify_data = prev;

  return TRUE;
}

gboolean
inotify_data_delete (gint wd, AppData *app_data)
{
  InotifyData **p = NULL;
  InotifyData *prev = NULL;

  assert(app_data != NULL);

  p = &app_data->inotify_data;

  while ((*p) != NULL)
    {
      if ((*p)->wd == wd)
        {
          prev = *p;
          *p = (*p)->next;
          free(prev->path);
          prev->path = NULL;
          prev->wd = -1;
          free(prev);
          return TRUE;
        }
      p = &((*p)->next);
    }

  return FALSE;
}

gint
inotify_data_get_wd (const gchar *path, AppData *app_data)
{
  InotifyData *p = NULL;

  assert(app_data != NULL);
  assert(path != NULL);

  p = app_data->inotify_data;

  while (p != NULL)
    {
      if (!strcmp(p->path, path))
        {
          return p->wd;
        }
      p = p->next;
    }
  return -1;
}

const gchar *
inotify_data_get_path (gint wd, AppData *app_data)
{
  InotifyData *p = NULL;

  assert(app_data != NULL);

  p = app_data->inotify_data;

  while (p != NULL)
    {
      if (p->wd == wd)
        {
          return (const gchar *) p->path;
        }
      p = p->next;
    }
  return NULL;
}

gboolean
parse_paths (gchar *str, AppData *app_data)
{
  gchar *c_paths = NULL;
  gchar *path = NULL;
  gchar *p = NULL;

  assert(str != NULL);
  assert(app_data != NULL);

  c_paths = g_strdup(str);
  p = strtok(c_paths, ":");

  while (p != NULL)
    {
      /* Append slash */
      if (p[strlen(p) - 1] != '/')
        {
          path = g_strconcat(p, "/", NULL);
        }
      else
        {
          path = g_strdup(p);
        }

      if (inotify_add_watch_dir(path, app_data) < 0)
        {
          CRAWLER_DEBUG_ERR("Failed to add: %s", path);
          free(path);
        }
      p = strtok(NULL, ":\0");
    }
  g_free(c_paths);

  return TRUE;
}

void
clean_app_data (AppData *app_data)
{
  InotifyData *prev = NULL;

  if (app_data == NULL)
    {
      return;
    }

  while (app_data->inotify_data != NULL)
    {
      prev = app_data->inotify_data;
      app_data->inotify_data = app_data->inotify_data->next;
      free(prev->path);
      prev->path = NULL;
      prev->wd = -1;
      free(prev);
    }

  app_data->inotify_data = NULL;
  g_object_unref(app_data->gconf_client);
}

void
vfree (gchar **vs)
{
  gint i = 0;

  if (vs == NULL)
    {
      return;
    }

  for (i = 0; vs[i] != NULL; i++)
    {
      g_free (vs[i]);
      vs[i] = NULL;
    }
  g_free(vs);
}

gboolean
daemonize (void)
{
  gint i = 0;
  pid_t pid;
  pid_t sid;

  umask(0);

  /* The parent exits, the child continues */
  pid = fork();

  if (pid < 0)
    {
      CRAWLER_DEBUG_ERR("Fork failed");
      return FALSE;
    }
  else if (pid > 0)
    {
      exit(0);
    }

  /* Create a session ID */
  sid = setsid();

  if (sid < 0)
    {
      CRAWLER_DEBUG_ERR("Could not create a session ID");
      return FALSE;
    }

  /* Change the working directory to root */
  if ((chdir("/")) < 0)
    {
      CRAWLER_DEBUG_ERR("Could not change the working directory");
      return FALSE;
    }

  /* Close the standard file descriptors */
  close(STDIN_FILENO);
  close(STDOUT_FILENO);
  close(STDERR_FILENO);

  i = open("/dev/null", O_RDWR);
  dup(i);
  dup(i);

  return TRUE;
}

void
init_app_data (AppData *app_data)
{
  if (app_data == NULL)
    {
      return;
    }

  g_type_init();

  app_data->inotify_data = NULL;
  app_data->debug_flag = 0;
  app_data->no_daemon_flag = 0;
  app_data->no_crawl_flag = 0;
  app_data->gconf_client = gconf_client_get_default();

  /* available memory cards */
  app_data->mmcs = init_mmcs_array();
}

void
gconf_write (AppData *app_data, const gchar *state)
{
  GError *err = NULL;

  if (state == NULL || app_data == NULL || app_data->gconf_client == NULL)
    {
      return;
    }
  
  gconf_client_set_string(app_data->gconf_client, GCONF_STATE, state, &err);

  if (err != NULL)
    {
      CRAWLER_DEBUG_ERR("gconf_client_set_string() failed: %s",
                        err->message);
      g_error_free(err);
      err = NULL;
    }
}

/*
 * Separated function to abstract the array of available memory cards.
 * So, we can change this implementation (reading /proc or similar).
 * Now, they are just defined manually the mmc's
 * FIXME: We should make this more general and use /proc to avoid a
 * hardcoded number of mmcs.
 */
static mmc_t *
init_mmcs_array (void)
{
  static mmc_t mmcs[] =
    {
      {
	MMC1_PATH,
	MMC1_DEV_PATH,
	FALSE,
	GCONF_MMC_COVER_OPEN
      },
#ifndef HAVE_770
      {
	MMC2_PATH,
	MMC2_DEV_PATH,
	FALSE,
	GCONF_INTERNAL_MMC_COVER_OPEN
      },
#endif
      /* keep at end when adding more mmc's */
      {NULL, NULL, FALSE, NULL}
    };

  return mmcs;
}
