/**
 * Copyright (C) 2007 Nokia. All rights reserved.
 */

/** @file crawler_queue.h
 *  \brief Metadata Crawler event queue prototypes
 */

#ifndef CRAWLER_QUEUE_H
#define CRAWLER_QUEUE_H

#include <glib.h>

/* opaque, in crawler-queue.c */
struct queue_struct;    
typedef struct queue_struct *queue_t;

/**
 * Checks if the queue is empty
 * 
 * @param q queue to check
 * @return 1 if the queue is empty, 0 otherwise
 */
gint queue_empty (queue_t q);

/**
 * Checks if the queue has reached its max capacity
 *
 * @param q queue to check
 * @return 1 if the max capacity has been reached, 0 otherwise
 */
gint queue_full (queue_t q);

/**
 * Creates a queue with a capacity of num_elements
 *
 *  @param num_elements the capacity of the queue
 *  @return queue
 */
queue_t queue_create (gint num_elements);

/**
 * Frees the elements of the queue and the queue itself
 *
 * @param q queue to free
 */
void queue_destroy (queue_t q);

/**
 * Sets the parameters of the queue to values indicating
 * that the queue is empty
 *
 * @param q queue to set the parameters for
 */
void queue_make_empty (queue_t q);

/**
 * Adds an element to the queue
 *
 * @param d element to add
 * @param q queue to add the element
 */
void queue_enqueue (gpointer d, queue_t q);

/**
 * Returns the first element in the queue
 *
 * @param q queue to get the first element for
 * @return first element of the queue
 */
gpointer queue_front (queue_t q);

/**
 * Decreases the size of the queue by one. Note that
 * the queue_front() should be called first so that 
 * the element can be freed
 *
 * @param q queue to decrease the size for
 */
void queue_dequeue (queue_t q);

#endif
