/**
 * Copyright (C) 2007 Nokia. All rights reserved.
 */

#include <stdlib.h>
#include <assert.h>
#include <glib.h>

#include "crawler-queue.h"
#include "crawler-debug.h"

struct queue_struct
{
  /* indicates maximum amount of elements in queue */
  gint max_size;  
  /* points to the first element of current queue */
  gint front;   
  /* points to the last element of current queue */
  gint rear;   
  /* indicates the amount of elements currently in queue */
  gint size;   
  /* points to the actual queue array */
  gpointer *array; 
};

gint
queue_empty (queue_t q)
{
  return q->size == 0;
}

gint
queue_full (queue_t q)
{
  return q->size == q->max_size;
}

queue_t
queue_create (gint num_elements)
{
  queue_t q;

  q = g_malloc0(sizeof(struct queue_struct));

  if (q == NULL)
    {
      exit(-1);
    }

  q->array = malloc(sizeof(gpointer) * num_elements);

  if (q->array == NULL)
    {
      exit(-1);
    }

  q->max_size = num_elements;

  queue_make_empty(q);

  return q;
}

void
queue_destroy (queue_t q)
{
  if (q == NULL)
    {
      return;
    }

  if (q->array)
    {
      g_free(q->array);
    }

  g_free(q);
}

void
queue_make_empty (queue_t q)
{
  q->size = 0;
  q->front = 1;
  q->rear = 0;
}

static gint
next_position (gint v, queue_t q)
{
  assert(q != NULL);
  return (v+1) % q->max_size;
}

void
queue_enqueue (gpointer d, queue_t q)
{
  assert(q != NULL);

  if (queue_full(q))
    {
      CRAWLER_DEBUG_ERR("Queue FULL. Unable to add new entry!");
      return;
    }

  q->size++;
  q->rear = next_position(q->rear, q);
  q->array[q->rear] = d;
}

gpointer
queue_front (queue_t q)
{
  assert(q != NULL);

  if (!queue_empty(q))
    {
      return q->array[q->front];
    }

  return NULL;
}

void
queue_dequeue (queue_t q)
{
  assert(q != NULL);

  if (!queue_empty(q))
    {
      q->size--;
      q->front = next_position(q->front, q);
    }
}
