/**
 * Copyright (C) 2007 Nokia. All rights reserved.
 */

/** @file crawler_interface.h
 *  \brief Metadata crawler's interface
 */ 

#ifndef CRAWLER_INTERFACE_H
#define CRAWLER_INTERFACE_H

#include <gconf/gconf.h>
#include <gconf/gconf-client.h>

#include <libgnomevfs/gnome-vfs-volume-monitor.h>
#include <libgnomevfs/gnome-vfs-volume.h>

#include <libosso.h>

#include "crawler-queue.h"

#define HAVE_770

#define DEV_PATH "/dev/"

#ifdef HAVE_770
#define MMC1_COVER "mmcblk0p1"
#else
#define MMC1_COVER "mmcblk1p1"
#endif

#define MMC1_DEV_PATH DEV_PATH MMC1_COVER
#define MMC1_PATH "/media/mmc1/"

#ifdef HAVE_770
#define MMC2_COVER "mmcblk1p1"
#else
#define MMC2_COVER "mmcblk0p1"
#endif

#define MMC2_DEV_PATH DEV_PATH MMC2_COVER
#define MMC2_PATH "/media/mmc2/"

#define EVENT_BUF_SIZE 16384

#ifdef HAVE_770
#define DEFAULT_PATHS "/home/user/MyDocs/.sounds/:" \
                      "/home/user/MyDocs/.videos/:" \
                      "/home/user/MyDocs/.documents/:" \
                      "/home/user/MyDocs/.images/:" \
                      "/home/user/MyDocs/:" \
                      "/media/mmc1/"
#else
#define DEFAULT_PATHS "/home/user/MyDocs/.sounds/:" \
                      "/home/user/MyDocs/.videos/:" \
                      "/home/user/MyDocs/.documents/:" \
                      "/home/user/MyDocs/.images/:" \
                      "/home/user/MyDocs/"
#endif

#define GCONF_STATE "/apps/osso/metalayer-crawler/state"
#define GCONF_STATE_IDLE "IDLE"
#define GCONF_STATE_CRAWLING "CRAWLING"
#define GCONF_MMC_COVER_OPEN "/system/osso/af/mmc-cover-open"
#define GCONF_INTERNAL_MMC_COVER_OPEN "/system/osso/af/internal-mmc-cover-open"
#define USER_MYDOCS_DIR "/home/user/MyDocs"

/**
 * If there are more than ERROR_THRESHOLD read errors
 * when crawling through a directory, crawling for that
 * directory level will be aborted.
 */
#define ERROR_THRESHOLD 3

/**
 * Contains fields needed to work with the memory cards.
 */
typedef struct _mmc
{
  /**< Mount point */
  gchar *path;
  /**< Device path */
  gchar *dev_path;
  /**< When crawling it */
  gboolean skip_crawling;
  /**< GConf key for checking cover */
  gchar *gconf_cover_open;
} mmc_t;

/**
 * Contains the path and watch descriptor
 * of one inotify watch
 */
typedef struct _InotifyData
{
  /**< Watched path                     */
  gchar *path;               
  /**< Watch descriptor of the path     */
  gint wd;                   
  /**< The next item in the linked list */
  struct _InotifyData *next; 
} InotifyData;

/**
 * Contains all kinds of data that is needed
 * all around the application
 */
typedef struct
{
  /**< A linked list of Inotify paths and descriptors */
  InotifyData *inotify_data; 
  /**< Client for GConf */
  GConfClient *gconf_client; 
  /**< Volume monitor for MMC mounts and umounts */
  GnomeVFSVolumeMonitor *volume_monitor; 
  /**< inotify file descriptor listener */
  GIOChannel *channel;       
  /**< OSSO context */
  osso_context_t *osso;      
  /**< The current event queue */
  queue_t queue;             
  /**< source for channel */
  gint source;                
  /**< TRUE if the program is run in foreground */
  gint no_daemon_flag;        
  /**< TRUE if in debug mode */
  gint debug_flag;            
  /**< TRUE if there should be no crawling in the beginning */
  gint no_crawl_flag;         

  /**< Array of available mmc's */
  mmc_t *mmcs;
} AppData;

/**
 * Initializes the inotify monitor
 *
 * @return 0 on success, -1 on failure
 */
gint crawler_init (void);

/**
 * Adds inotify monitor for the given dir. Also goes recursively
 * through subdirectories and adds watches to each subdirectory.
 *
 * @param path directory to monitor
 * @param app_data common application data
 * 
 * @return 0 on success, -1 on error
 */
gint inotify_add_watch_dir (const gchar *path, AppData *app_data);

/**
 * Removes an inotify watch
 *
 * @param wd watch to remove
 * @param app_data common application data
 */
void inotify_rm_watch_dir (gint wd, AppData *app_data);

/**
 * Processes inotify events. Blocks until processing
 * is done
 *
 * @param app_data common application data
 * @return 0 on success, -1 on error
 */
gint process_inotify_events (AppData *app_data);

/**
 * Crawl through all directories that are currently
 * being watched
 *
 * @param app_data common application data
 */
void crawl_all (AppData *app_data);

/**
 *  Starts watching the MMC if it is mounted
 *
 * @param app_data common application data
 * @return 0 on success, -1 on error or if MMC is not mounted
 */
gint watch_mmc (AppData *app_data);

/**
 * Closes the inotify listeners
 *
 * @param app_data common application data
 * @return 0 on success, -1 on error);
 */
gint close_listener (AppData *app_data);

/**
 * Crawls through a directory and watches it
 *
 * @param app_data common application data
 * @param path the path to crawl
 */
void crawl_watch_dir (AppData *app_data, gchar *path);

#endif /* CRAWLER_INTERFACE */
