/**
 * Copyright (C) 2007 Nokia. All rights reserved.
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <dirent.h>
#include <errno.h>
#include <assert.h>
#include <sys/syscall.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <sys/inotify.h>                /* we were using a hacked version */
#include <unistd.h>
#include <libgen.h>                     /* dirname(), basename() */
#include <glib.h>                       /* For glib main loop */
#include <gconf/gconf-client.h>
#include <metadata_interface.h>
#include <osso-mem.h>

#include "crawler-debug.h"
#include "crawler-queue.h"
#include "crawler-interface.h"
#include "crawler-utils.h"

/* #include "inotify.h"            ** this shall be omitted later ** 
#include "inotify-syscalls.h" */

#define SEARCH_MAX_LENGTH 512
#define EVENTS_IN_UNIVERSE_MAX 1000
#define DIRECTORY_RECURSION_LIMIT 32
#define PATHS_CHUNK_SIZE (getpagesize() / sizeof(gchar *))

gint inotify_monitor_fd = -1;

static gboolean crawl(gchar *path,
                      gchar ***files,
                      gint *i,
                      gboolean add_null,
                      guint depth,
		      AppData *app_data);

/***********************************************************
 * PRIVATE FUNCTIONS
 */

/* Directories starting with a period must not be indexed. The exception
 * is that all directories that are under /home/user/MyDocs must be
 * crawled every time
 *
 * @param path the directory to check
 * @return TRUE if directory may be crawled, FALSE if not
*/
static gboolean 
_is_directory_crawlable (const gchar *path)
{
  gchar *bname = NULL; /* For leaf directory */
  gboolean ret = TRUE;

  bname = g_path_get_basename(path);
   
  CRAWLER_DEBUG_INFO("Directory: %s; leaf: %s", path, bname);
    
  /* bname now contains the name of directory */
  if (strstr(path, USER_MYDOCS_DIR))
    {
      goto out; /* Ew! */
    }
  
  if (bname[0] == '.')
    {
      ret = FALSE;
    }
 out:
  g_free(bname);
  return ret;
}

/* Filenames starting with a period must never be indexed. Period.
 *
 * @param fname name of the file, without leading path
 * @return TRUE if file is visible, FALSE otherwise
 */
static gboolean 
_is_file_visible (const gchar *fname)
{
  CRAWLER_DEBUG_INFO("Checking filename %s for visibility", fname);
  
  if (fname[0] == '.')
    {
      return FALSE;
    }
  else
    {
      return TRUE;
    }
}

/**
 * Adds a null to the end of the file list
 *
 * @param files a list of files
 * @param i the number of files in the list
 */
static void
append_null (gchar ***files, gint *i)
{
  if (files == NULL || i == NULL)
    {
      return;
    }

  *files = (gchar **) realloc(*files, sizeof(gchar *) * (*i + 1));

  if (*files == NULL)
    {
      CRAWLER_DEBUG_INFO ("Memory reallocation failed in append_null for %u files", i+1);
      return;
    }

  (*files)[*i] = NULL;
  (*i)++;
}

/**
 * Strips items beginning with 'tag' from 'list'.
 *
 * @param list the list to check
 * @param n the number of items in the list
 * @param tag the tag to look for
 */
static void 
strip_items (gchar **list, gint n, const gchar *tag)
{
  gint i = 0;
  gint debug = 0;
    
  if (list == NULL || n == 0 || tag == NULL)
    {
      return;
    }
  
  CRAWLER_DEBUG_INFO("Stripping items with tag: %s", tag);
  
  for(i = 0; i < n; i++) 
    {
      if (list[i] == NULL) 
        {
          break;
        } 
      else if(list[i][0] == '\0') 
        {
          continue;      
        } 
      else 
        {
          if(strstr(list[i], tag) != NULL) 
            {
              list[i] = g_strdup("");
              debug++;
            }
        }
    }
  
    CRAWLER_DEBUG_INFO("%d items stripped", debug);
}

static gboolean
check_mmc_cover (mmc_t *mmc)
{
  GConfClient *gc_client = NULL;
  gboolean mmc_cover_open = FALSE;

  gc_client = gconf_client_get_default();
  if(gc_client == NULL)
    {
      return FALSE;
    }
  
  mmc_cover_open = gconf_client_get_bool(gc_client,
					 mmc->gconf_cover_open,
					 NULL);
  
  g_object_unref(gc_client);
  
  return mmc_cover_open;
}

/**
 * Checks if either of the MMC covers is closed. If not, the file
 * list needs to be cleaned of files that reside on the MMC which is
 * being removed.
 *
 * @param files the file list to clean up
 */
static void 
check_crawl_list (gchar **files, gint n, gpointer user_data)
{
  AppData *app_data = (AppData *) (user_data);
  mmc_t *mmc = NULL;
  gboolean cover_open = FALSE;

  if (app_data == NULL)
    {
      CRAWLER_DEBUG_INFO("AppData is NULL");
      return;
    }

  CRAWLER_DEBUG_INFO("Checking for cover state...");

  mmc = app_data->mmcs;
  while (mmc->path != NULL)
    {
      cover_open = check_mmc_cover (mmc);
      CRAWLER_DEBUG_INFO("COVER: %s %d", mmc->path, cover_open);
      if (cover_open == TRUE)
	{
	  strip_items(files, n, mmc->path);
	}
      mmc++;
    }
}

/**
 * Check if crawling of path 'path' should be stopped
 * due to open MMC cover. Also cleans the file list of
 * invalid entries.
 *
 * @param files the file list
 * @param n the number of files in the file list
 * @param the path to check
 * @param the application data
 * @return TRUE if 'path' and the path of the mmc that
 * is being unmounted match.
 */
static gboolean 
check_crawl_stop (gchar **files, gint n, gchar *path, AppData *app_data)
{
  gboolean stop_crawling = FALSE;
  mmc_t *mmc = NULL;

  if (path == NULL)
    {
      return FALSE;
    }
    
  CRAWLER_DEBUG("Checking stop condition");
  
  /*
   * If some mmc path matches,let's check
   * if the corresponding MMC cover is open
   */

  mmc = app_data->mmcs;
  while (mmc->path != NULL)
    {
      if (strstr (path, mmc->path))
	{
	  if ((stop_crawling = check_mmc_cover (mmc)))
	    {
	      strip_items (files, n, mmc->path);
	      meta_path_absent(mmc->path);
	    }
	}
      mmc++;
    }
  
  /* Check the memory conditions */
  stop_crawling |= (bool)osso_mem_in_lowmem_state();
  
  CRAWLER_DEBUG("Check crawl stop: path: %s stop_crawling: %d", 
		path, stop_crawling);
  
  return stop_crawling;
}

/**
 * Checks if  a failure to read 'path' was due to unmounted MMC
 *
 * @param path the path to check
 * @param app_data the application data
 * @return TRUE if failure was due to unmounted MMC, FALSE otherwise
 */ 
static gboolean
check_read_failure (const gchar *path, AppData *app_data)
{
  struct stat s;
  mmc_t *mmc = NULL;

  if(path == NULL)
    {
      return FALSE;
    }

  mmc = app_data->mmcs;
  while (mmc->path != NULL)
    {
      if (strstr (path, mmc->path)) 
	{
	  if (lstat(mmc->dev_path, &s) < 0) 
	    {
	      return TRUE;
	    }
	} 
      mmc++;
    }
  
  return FALSE;    
}

/**
 * Crawls through a directory and creates a list of files encountered. Only 
 * files that are regular, non-empty, that have a recognized extension are added
 * to the list; if a file is already on the database and hasn't changed since
 * it was added, the "present" column on its row is set and the file is not
 * added to the list.  
 *
 * @param path the path to crawl
 * @param files a pointer to the array to add the files
 * @param i the number of files added
 * @param add_null if NULL should be added to the end of the list
 * @param depth how many levels remain to stop the recursive crawling
 * @param app_data the application data
 * @return returns FALSE when recursion should immediately be stopped
 */
static gboolean
crawl (gchar *path,
       gchar ***files,
       gint *i,
       gboolean add_null,
       guint depth,
       AppData *app_data)
{
  DIR *dir = NULL;
  struct stat st;
  struct dirent *entry = NULL;
  gboolean continue_recursion = TRUE;
  GSList *subfolders = NULL;
  GSList *first_subfolder = NULL;
  gchar *subfolder_path = NULL;
  bool modified_or_new = FALSE;
  bool is_present = FALSE;

  if (path == NULL || i == NULL || files == NULL)
    {
      return TRUE;
    }
  
  CRAWLER_DEBUG("Entered crawl() function with path: %s i: %d", path, *i);

  if (check_crawl_stop(*files, *i, path, app_data))
    {    
      CRAWLER_DEBUG("Flag is up, stopping now, before entering the directory");
      return FALSE;
    }

  if ((*files == NULL))
    {
      *files = (gchar **) g_malloc0 (sizeof(gchar *) * PATHS_CHUNK_SIZE);
    }

  dir = opendir(path);

  if (dir == NULL)
    {
      CRAWLER_DEBUG("opendir() failed");

      /* The read failure might be due to unmounted MMC. In this case
       * recursion must be stopped. In other kinds of errors, the recursion
       * should continue. */
      return !check_read_failure(path, app_data);

    }

  while ((entry = readdir(dir)) != NULL &&
      continue_recursion) 
    {
      gchar npath[SEARCH_MAX_LENGTH + 1] = "";

      if (path[strlen(path) - 1] != '/')
        {
          snprintf(npath, SEARCH_MAX_LENGTH, "%s/%s", path,
                   entry->d_name);
        }
      else
        {
          snprintf(npath, SEARCH_MAX_LENGTH, "%s%s", path,
                   entry->d_name);
        }

      if ((strcmp(entry->d_name, ".") == 0) ||
          (strcmp(entry->d_name, "..") == 0) || 
          !_is_directory_crawlable(path))
        {
          continue;
        }

      CRAWLER_DEBUG("Examining directory: %s", npath);
      /* Recursively go through directories */
      if (stat(npath, &st) == -1)
        {
          continue;
        }
      else if (S_ISDIR(st.st_mode))
        {
          if (depth > 0)
            {
              subfolders = g_slist_prepend (subfolders, g_strdup(npath));
            }
          else
            {
              /* this is not a serious error, file exploration can continue */
              continue_recursion = TRUE;
            }
          continue;
        }

      if (!_is_file_visible(entry->d_name))
        {
          continue;
        }

      /* A file found, add it to the list if it's regular and recognized */
      if ( S_ISREG (st.st_mode) &&
	   meta_has_recognized_extension(npath))
	{
	  if (meta_is_modified_or_new (npath, &modified_or_new, &is_present))
	    {
	      if (modified_or_new)
		{
		  (*i)++;
		  if (((*i) % PATHS_CHUNK_SIZE == 0))
		    {
		      *files = (gchar **) g_realloc (*files, 
		                                     sizeof(gchar *) * ((*i) + PATHS_CHUNK_SIZE));
		      /* we know that at this point *i is a multiple of 
		       * PATHS_CHUNK_SIZE because (*i) % PATHS_CHUNK_SIZE == 0 
		       * and also *i != 0 */
                      if (*files == NULL)
                        {
                          CRAWLER_DEBUG_INFO ("Memory reallocation failed in crawl for %u files", i+1);
                          return FALSE;
                        }
		    }
		  CRAWLER_DEBUG("Adding media: %s", npath);
		  (*files)[*i-1] = g_strdup(npath);
		}
	      else if (!is_present)
		{
		  meta_set_present (npath);
		}
	      /* This checks if the MMC cover has been opened while we're
	       * still generating the list of files. This is needed for
	       * those situations where an application may generate tens
	       * of thousands of files on removable media.
	       */
	      if ((*i % CHECK_INTERVAL) == 0 && check_crawl_stop(*files, *i, path, app_data))
		{    
		  CRAWLER_DEBUG("Flag is up, stopping now...");
		  continue_recursion = FALSE;
		  break;
		}
            }
        }
    }

  CRAWLER_DEBUG("Exiting crawl() function: %d returning: %d", *i, continue_recursion);

  if (closedir(dir) == -1)
    {
      perror("closedir");
      CRAWLER_DEBUG("Error at closedir dir in crawl");
      return FALSE;
    } 
  
  /* Now crawl subdirectories. */
  while(continue_recursion && subfolders != NULL)
    {
      first_subfolder = subfolders;
      subfolder_path = (gchar*)(first_subfolder->data);
      continue_recursion = crawl (subfolder_path, files, i, false, depth - 1, app_data);
      subfolders = g_slist_remove_link (subfolders, first_subfolder);
      g_slist_free (first_subfolder);
    }
  g_slist_free(subfolders);
  
  if (add_null == TRUE)
    {
      *files = (gchar **) g_realloc (*files, sizeof(gchar *) * (*i + 1));
      if (*files == NULL)
        {
          CRAWLER_DEBUG_INFO ("Memory reallocation failed in crawl for %u files", i+1);
          return FALSE;
        }
      (*files)[*i] = NULL;
      (*i)++;
    }

  return continue_recursion;
}

/**
 * Determinates what happened 
 *
 * @param inotify_event struct 
 * @param app_data common application data
 * @return If a new file was found, a newly allocated string containing 
 *         the path is returned, otherwise NULL
 * @note Caller must 'free()' the returned pointer        
 */
static gchar *
handle_event (struct inotify_event *event, AppData *app_data)
{
  gchar *path = NULL;
  const gchar *tmp = NULL;
  gint i = 0;

  assert(event != NULL);
  assert(app_data != NULL);

  CRAWLER_DEBUG("Handling an event: 0x%x for path: %s wd: %d",
                event->mask, event->len > 0 ? event->name : "",
                event->wd);

  if (event->len > 0)
    {
      /* Look for the path corresponding to the file */
      if ((tmp = inotify_data_get_path(event->wd, app_data)) != NULL)
        {
          /* We are not interested in any other event from /dev
           * but IN_CREATE/IN_DELETE (for MMC)
           */
          if ((strcmp(tmp, DEV_PATH) == 0))
            {
              return NULL;
            }

          path = g_strconcat(tmp, event->name, NULL);
        }
    }

  switch (event->mask & 0x00FFFFFF)
    {
      /* File was closed */
    case IN_CLOSE_WRITE:
      CRAWLER_DEBUG("IN_CLOSE_WRITE event");
      return path;
      break;

      /* File/dir was moved to this folder */
    case IN_MOVED_TO:
      CRAWLER_DEBUG("IN_MOVED_TO event");
      if (event->mask & IN_ISDIR)
        {
          gchar **paths = NULL;
          i = 0;

          CRAWLER_DEBUG("Directory moved");

          if (inotify_add_watch_dir(path, app_data) < 0)
            {
              CRAWLER_DEBUG_ERR("Adding a watch for %s failed", path);
            }
          gconf_write(app_data, GCONF_STATE_CRAWLING);
          crawl(path, &paths, &i, TRUE, DIRECTORY_RECURSION_LIMIT, app_data);

          if (!meta_add_files_with_check(paths, i, check_crawl_list, NULL))
            {
              CRAWLER_DEBUG_ERR("Failed to add the files to Metadata Storage");
            }
          gconf_write(app_data, GCONF_STATE_IDLE);
          /* Free the paths */
          vfree(paths);
        }
      else
        {
          return path;
        }
      break;
      /* File/dir was moved from here to another folder */
    case IN_MOVED_FROM:
      CRAWLER_DEBUG("IN_MOVED_FROM event");
      gconf_write(app_data, GCONF_STATE_CRAWLING);
      if (event->mask & IN_ISDIR)
        {
          meta_path_absent(path);
        }
      else
        {
          meta_file_absent(path);
        }
      gconf_write(app_data, GCONF_STATE_IDLE);
      break;

      /* File/dir was deleted */
    case IN_DELETE:
      CRAWLER_DEBUG("IN_DELETE event");

      if (path == NULL)
        {
          break;
        }

      if ((event->mask & 0xFF000000) != IN_ISDIR)
        {
          gconf_write(app_data, GCONF_STATE_CRAWLING);
          meta_file_absent(path);
          gconf_write(app_data, GCONF_STATE_IDLE);
        }
      break;

      /* If the watched directory gets unmounted or deleted, this event
       * is received (amongst others)
       */
    case IN_DELETE_SELF:
      CRAWLER_DEBUG("IN_DELETE_SELF event");
      gconf_write(app_data, GCONF_STATE_CRAWLING);
      meta_path_absent(inotify_data_get_path(event->wd, app_data));
      gconf_write(app_data, GCONF_STATE_IDLE);
      inotify_rm_watch_dir(event->wd, app_data);
      break;

      /* File/dir was created */
    case IN_CREATE:
      if ((event->mask & 0xFF000000) == IN_ISDIR && path != NULL)
        {
          CRAWLER_DEBUG("Directory created");
          crawl_watch_dir (app_data, path);
        }
      CRAWLER_DEBUG("IN_CREATE event");
      break;

    case IN_UNMOUNT:
      CRAWLER_DEBUG("IN_UNMOUNT event");
      /* This is done in crawler-volume-monitor.c */
      /*meta_path_absent(inotify_data_get_path(event->wd, app_data));*/
      /* Individual inotify watches need to be removed here */
      inotify_rm_watch_dir(event->wd, app_data);
      break;

      /* Some not requested message received */
    default:
      CRAWLER_DEBUG("Not requested message: 0x%x wd: %d\n", event->mask,
                    event->wd);
      break;
    }

  free(path);
  return NULL;
}

static void
handle_events (queue_t q, AppData *app_data)
{
  gchar **paths = NULL;
  gchar *ptr = NULL;
  gint i = 0;
  struct inotify_event *event;

  assert(app_data != NULL);

  while (!queue_empty(q))
    {
      event = queue_front(q);
      queue_dequeue(q);

      /* Make a list of all files to add to Metadata Storage */
      if ((ptr = handle_event(event, app_data)) != NULL)
        {
          paths = (gchar **) realloc(paths, sizeof(gchar *) * (i + 1));
          assert(paths != NULL);
          paths[i++] = ptr;
        }
      free(event);
    }

  /* Add NULL */
  if (paths != NULL)
    {
      paths = (gchar **) realloc(paths, sizeof(gchar *) * (i + 1));

      assert(paths != NULL);
      paths[i++] = NULL;

      if (!meta_add_files_with_check(paths, i, check_crawl_list, app_data))
        {
          CRAWLER_DEBUG_ERR
            ("Failed to add the files to Metadata Storage");
        }
      
      /* Free the paths */
      vfree(paths);
    }
}


static gboolean
read_events (GIOChannel *source, GIOCondition cond, gpointer user_data)
{
  gchar buffer[EVENT_BUF_SIZE];
  size_t buffer_i;
  struct inotify_event *pevent, *event;
  ssize_t r;
  size_t event_size;
  gint count = 0;
  queue_t q;
  AppData *app_data = NULL;

  app_data = (AppData *) user_data;

  CRAWLER_DEBUG("Event received");

  if (app_data == NULL)
    {
      return FALSE;
    }

  q = app_data->queue;

  if (q == NULL)
    {
      return FALSE;
    }

  r = read(inotify_monitor_fd, buffer, EVENT_BUF_SIZE);

  if (r <= 0)
    {
      return r;
    }

  buffer_i = 0;

  while (buffer_i < r)
    {
      /* Parse events and queue them ! */
      pevent = (struct inotify_event *) &buffer[buffer_i];
      event_size = sizeof(struct inotify_event) + pevent->len;
      event = malloc(event_size);
      assert(event != NULL);
      memmove(event, pevent, event_size);
      queue_enqueue(event, q);
      buffer_i += event_size;
      count++;
    }
  CRAWLER_DEBUG("Handling events");

  if (!queue_empty(q))
    {
      handle_events(q, app_data);
    }
  CRAWLER_DEBUG("Handling events done");

  return TRUE;
}

/***********************************************************
 * PUBLIC FUNCTIONS
 */

gint
process_inotify_events (AppData *app_data)
{
  queue_t q;

  assert(app_data != NULL);

  q = queue_create(EVENTS_IN_UNIVERSE_MAX);

  app_data->channel = g_io_channel_unix_new(inotify_monitor_fd);
  app_data->queue = q;

  if (app_data->channel == NULL)
    {
      return -1;
    }

  app_data->source = g_io_add_watch(app_data->channel, G_IO_IN, read_events,
                                    app_data);

  CRAWLER_DEBUG("Installed GIOChannel: %d", app_data->source);

  return 0;
}

gint
crawler_init (void)
{
  if (inotify_monitor_fd != -1)
    {
      CRAWLER_DEBUG("Inotify monitor has been already initialized!\n");
      return -1;
    }

  inotify_monitor_fd = inotify_init();

  if (inotify_monitor_fd < 0)
    {
      CRAWLER_DEBUG("Could not initialize inotify: %s!\n",
                    strerror(errno));
      return -1;
    }

  return 0;
}


static gint
inotify_add_watch_dir_impl (const gchar *path, AppData *app_data, gint count)
{
  guint mask = IN_CLOSE_WRITE | IN_MOVE | IN_CREATE |
    IN_DELETE | IN_DELETE_SELF | IN_UNMOUNT; /* | IN_MODIFY */
  gint wd;
  gchar *tmp = NULL;
  DIR *dir = NULL;
  struct stat st;
  struct dirent *entry = NULL;
  gint errors = 0;

  if (count-- < 0) 
    {
      CRAWLER_DEBUG_ERR("%s is recursive directory", path);
      return -1;
    }
  
  if (!path || (strlen(path) == 0))
    {
      return -1;
    }

  if (!_is_directory_crawlable(path))
    {
      return -1;
    }

  if (stat(path, &st) == -1)
    {
      return -1;
    }
  else if (!S_ISDIR(st.st_mode))
    {
      CRAWLER_DEBUG_ERR("%s is not a directory", path);
      return -1;
    }

  /* check the permissions of the given directory */
  if ((access(path, F_OK) == 0) && (access(path, R_OK) == 0))
    {
      /* add directory to watch */
      wd = inotify_add_watch(inotify_monitor_fd, path, mask);

      if (wd < 0)
        {
          CRAWLER_DEBUG("inotify watch on %s has failed\n", path);
          return -1;
        }
      if (path[strlen(path) - 1] != '/')
        {
          tmp = g_strconcat(path, "/", NULL);
          inotify_data_prepend(tmp, wd, app_data);
          CRAWLER_DEBUG("Watching directory %s wd: %d\n", tmp, wd);
          free(tmp);
        }
      else
        {
          inotify_data_prepend(path, wd, app_data);
          CRAWLER_DEBUG("Watching directory %s wd: %d\n", path, wd);
        }
    }

  /* /dev wont be further processed */
  if ((strcmp(path, DEV_PATH) == 0))
    {
      return 0;
    }

  dir = opendir(path);

  if (dir == NULL)
    {
      CRAWLER_DEBUG_ERR("Opendir failed for path: %s", path);
      return -1;
    }

  while ((entry = readdir(dir)) != NULL)
    {
      gchar npath[SEARCH_MAX_LENGTH + 1] = "";

      if (path[strlen(path) - 1] != '/')
        {
          snprintf(npath, SEARCH_MAX_LENGTH, "%s/%s", path,
                   entry->d_name);
        }
      else
        {
          snprintf(npath, SEARCH_MAX_LENGTH, "%s%s", path,
                   entry->d_name);
        }

      if ((strcmp(entry->d_name, ".") == 0) ||
          (strcmp(entry->d_name, "..") == 0))
        {
          continue;
        }

      if (stat(npath, &st) == -1)
        {
          int errno_tmp = 0;
          errno_tmp = errno;
          
          CRAWLER_DEBUG_ERR("stat() failed: %s", strerror(errno_tmp)); 
          if(errors++ < ERROR_THRESHOLD)
            {
              continue;
            }
          else
            {
              break;
            }
        }
      else if (S_ISDIR(st.st_mode))
        {
          if (inotify_add_watch_dir_impl(npath, app_data, count) < 0)
            {
              /* closedir(dir); */
              continue;
              /* return -1; this stops searching the subfolders if one of them
               * happens to be a hidden one */
            }
          continue;
        }
    }

  closedir(dir);
  return 0;
}

gint
inotify_add_watch_dir(const char *path, AppData * app_data)
  {
    return inotify_add_watch_dir_impl(path, app_data, DIRECTORY_RECURSION_LIMIT);
  }

void
inotify_rm_watch_dir (gint wd, AppData *app_data)
{
  if (wd < 0 || inotify_monitor_fd < 0)
    {
      return;
    }

  inotify_rm_watch(inotify_monitor_fd, wd);

  inotify_data_delete(wd, app_data);
}

void
crawl_all (AppData *app_data)
{
  gint i = 0;
  gchar **paths = NULL;
  InotifyData *next = NULL;
  mmc_t *mmc = app_data->mmcs;

  if (app_data == NULL)
    {
      CRAWLER_DEBUG("crawl_all: app_data is NULL");
      return;
    }
  
  CRAWLER_DEBUG("crawl_all() started");

  next = app_data->inotify_data;

  while (next != NULL)
    {
      if (next->path != NULL &&
          strcmp(next->path, DEV_PATH)) 
        {
	  /* If MMC has been removed, don't crawl it */
	  while (mmc->path != NULL)
	    {
	      if (mmc->skip_crawling &&
		  strstr(next->path, mmc->path))
		{
		  CRAWLER_DEBUG("Skipping: %s", next->path);
		  next = next->next;
		  break;
		}
	      mmc++;
	    }

          /* Returns FALSE if MMC cover was opened while crawling. Do not process further
           * paths related to that MMC */
          if (!crawl(next->path, &paths, &i, FALSE, 0, app_data)) 
            {
                CRAWLER_DEBUG("crawl_all() failed: MMC cover open");

		mmc = app_data->mmcs;
		while (mmc->path != NULL)
		  {
		    if (strstr(next->path, mmc->path))
		      {
			mmc->skip_crawling = TRUE;
			CRAWLER_DEBUG("Skipping %s: %d", mmc->path, mmc->skip_crawling);
		      }
		    mmc++;
		  }
	    }
        }
 
      next = next->next;
    }

  append_null(&paths, &i);

  if (!meta_add_files_with_check(paths, i, check_crawl_list, app_data))
    {
      CRAWLER_DEBUG_ERR("Failed to add the files to Metadata Storage");
    }

  CRAWLER_DEBUG("crawl_all() finished");

  /* restore to FALSE skip_crawl values */
  mmc = app_data->mmcs;
  while (mmc->path != NULL)
    {
      mmc->skip_crawling = FALSE;
      mmc++;
    }
  
  /* Free the paths */
  vfree(paths);
}

gint
watch_mmc (AppData *app_data)
{
  struct stat s;
  mmc_t *mmc = NULL;
  gint return_value = 0;
  
  if (app_data == NULL)
    {
      return -1;
    }

  mmc = app_data->mmcs;
  while (mmc->path != NULL)
    {
      if (lstat(mmc->dev_path, &s) < 0)
	{
	  CRAWLER_DEBUG("%s not mounted", mmc->path);
	}
      else
	{
	  CRAWLER_DEBUG("%s mounted", mmc->path);
	  if (inotify_add_watch_dir(mmc->path, app_data) < 0)
	    {
	      CRAWLER_DEBUG("Adding %s watch failed or maximum directory depth reached", mmc->path);
	      /* no return, fall through to next MMC, but indicate that there was a problem */
	      return_value = -1;
	    }
	}
      mmc++;
    }

  return return_value;
}

gint
close_listener (AppData *app_data)
{
  if (app_data == NULL)
    {
      return -1;
    }

  if(app_data->source > 0)
    {
      g_source_remove(app_data->source);
      app_data->source = 0;
    }

  if (app_data->channel)
    {
      g_io_channel_shutdown(app_data->channel, TRUE, NULL);
      g_io_channel_unref(app_data->channel);
      app_data->channel = NULL;
    }

  return 0;
}

void
crawl_watch_dir (AppData *app_data, gchar *path)
{
  gchar **paths = NULL;
  gint i = 0;
  gboolean success = FALSE;

  if (app_data == NULL || path == NULL)
    {
      return;
    }

  gconf_write(app_data, GCONF_STATE_CRAWLING);
  
  /* Add a watch */
  if (inotify_add_watch_dir(path, app_data) < 0) 
    {
      CRAWLER_DEBUG_ERR("Adding a watch for %s failed", path);
    }
  
  /* A recursive crawl: create list of all files on MMC */
  success = crawl(path, &paths, &i, TRUE, DIRECTORY_RECURSION_LIMIT, app_data);

  if(success) 
    {
      if (!meta_add_files_with_check(paths, i, check_crawl_list, app_data)) 
        {
          CRAWLER_DEBUG_ERR("Failed to add the files to Metadata Storage");
        } 
    }

  gconf_write(app_data, GCONF_STATE_IDLE);

  vfree(paths);
}
