/*
 * This file is part of libosso-rtcom
 *
 * Copyright (C) 2006 Nokia. All rights reserved.
 *
 * Contact: naba.kumar@nokia.com
 */

#ifndef RTCOM_CP_INTERFACE_H_
#define RTCOM_CP_INTERFACE_H_

/************************************************************
 * System includes
 ************************************************************/

/* GLIB utilities */
#include <glib.h>

/* D-BUS definitions */
#ifndef DBUS_API_SUBJECT_TO_CHANGE
#define DBUS_API_SUBJECT_TO_CHANGE
#endif
#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-lowlevel.h>


/************************************************************
 * Type definitions
 ************************************************************/

G_BEGIN_DECLS

/* GObject type definition macros */
#define RTCOM_TYPE_CP_INTERFACE            (rtcom_cp_interface_get_type())
#define RTCOM_CP_INTERFACE(obj)            (G_TYPE_CHECK_INSTANCE_CAST((obj), \
                                             RTCOM_TYPE_CP_INTERFACE, \
                                             RtcomCPInterface))
#define RTCOM_CP_INTERFACE_CLASS(klass)    (G_TYPE_CHECK_CLASS_CAST((klass), \
                                             RTCOM_TYPE_CP_INTERFACE, \
                                             RtcomCPInterfaceClass))
#define IS_RTCOM_CP_INTERFACE(obj)         (G_TYPE_CHECK_INSTANCE_TYPE((obj), \
                                             RTCOM_TYPE_CP_INTERFACE))
#define IS_RTCOM_CP_INTERFACE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE((klass), \
                                             RTCOM_TYPE_CP_INTERFACE))
#define RTCOM_CP_INTERFACE_GET_CLASS(obj)  (G_TYPE_INSTANCE_GET_CLASS((obj), \
                                             RTCOM_TYPE_CP_INTERFACE, \
                                             RtcomCPInterfaceClass))

/*
 * The actual "class" definitions
 */
typedef struct _RtcomCPInterface      RtcomCPInterface;
typedef struct _RtcomCPInterfaceClass RtcomCPInterfaceClass;

struct _RtcomCPInterface {
    GObject      parent;
};

struct _RtcomCPInterfaceClass {
    GObjectClass parent_class;
};

/************************************************************
 * Public function prototypes
 ************************************************************/

GType       rtcom_cp_interface_get_type(void);
GObject    *rtcom_cp_interface_new (void);

/**
 * rtcom_cp_interface_private_message:
 *
 * @plugin:       The #RtcomCPInterface instance
 * @account_id:   A string representation of the account used to
 *                connect to this particular service where the
 *                message came from
 * @user:         The user id of the contacting person e.g. foo@bar.org
 * @operation_id: Returned operation ID saved into the TN Contact plugin.
 *
 * Informs task navigator that there's a new private message from a user
 * that we currently are not in a chat with.
 *
 * Return value: %TRUE if the message was sent. %FALSE if it was not.
 *               Emits user interaction in a signal "operation-reply"
 **/
gboolean 
rtcom_cp_interface_private_message (
    RtcomCPInterface *plugin,
    const gchar account_id[],
    const gchar user[],
    guint *operation_id);

/**
 * rtcom_cp_interface_authorization_request:
 *
 * @plugin:       The #RtcomCPInterface instance
 * @account_id:   A string representation of the account used to
 *                connect to this particular service where the
 *                message came from
 * @user:         The translated user if of the contacting person, might
 *                be service side alias, real name or something.
 * @address:      The address/user id of the @user
 * @message:      The message that was sent, might indicate why someone
 *                wants to authenticate us.
 * @operation_id: Returned operation ID saved into the TN Contact plugin.
 *
 * Informs task navigator that there's a new authorization request sent. This
 * should be called when user interaction is required to verify if we want
 * allow the other person to see our presence.
 *
 * Return value: %TRUE if the message was sent. %FALSE if it was not.
 *               Emits user interaction in a signal "operation-reply"
 **/
gboolean
rtcom_cp_interface_authorization_request (
    RtcomCPInterface *plugin,
    const gchar account_id[],
    const gchar user[],
    const gchar address[],
    const gchar message[],
    guint *operation_id);

/**
 * rtcom_cp_interface_authorization_response:
 *
 * @plugin:       The #RtcomCPInterface instance
 * @account_id:   A string representation of the account used to
 *                connect to this particular service where the
 *                message came from
 * @user:         The user id of the user who has authorized, or rejected us
 * @message:      The message that was sent, might indicate why someone
 *                did or did not authenticate us.
 * @operation_id: Returned operation ID saved into the TN Contact plugin.
 *
 * Informs the Task Navigator that someone as either authorized us or
 * rejected us. This is an information note that is sort of fire and forget,
 * no user interaction is expected. It's just a notification.
 *
 * Return value: %TRUE if the message was sent. %FALSE if it was not.
 **/
gboolean
rtcom_cp_interface_authorization_response (
    RtcomCPInterface *plugin,
    const gchar account_id[],
    const gchar user[],
    const gchar message[],
    const gboolean accepted,
    guint *operation_id);

/**
 * rtcom_cp_interface_group_invitation:
 *
 * @plugin:       The #RtcomCPInterface instance
 * @account_id:   A string representation of the account used to
 *                connect to this particular service where the
 *                message came from
 * @user:         The user id in string format of the user who is inviting us
 * @room:         The room id in string format we're invited to.
 * @operation_id: Returned operation ID saved into the TN Contact plugin.
 *
 * Informs the Contact Plugin that there's a new group invitation for
 * the user.
 *
 * Return value: %TRUE if the message was sent. %FALSE if it was not.
 *               Emits user interaction in a signal "operation-reply"
 **/
gboolean
rtcom_cp_interface_group_invitation (
    RtcomCPInterface *plugin,
    const gchar account_id[],
    const gchar user[],
    const gchar room[],
    guint *operation_id);

/**
 * rtcom_cp_interface_missed_call
 *
 * @plugin:       The #RtcomCPInterface instance
 * @account_id:   A string representation of the account used to
 *                connect to this particular service where the
 *                message came from
 * @user:         The user id in string format of the user who is inviting us
 * @time_stamp:   Time when the call was missed
 * @address:      Caller's (@user) address
 * @operation_id: Returned operation ID saved into the TN Contact plugin.
 *
 * Informs the TN's Contact Plugin that someone tried to call us, but we
 * could not handle it.
 *
 * Return value: %TRUE if the message was sent. %FALSE if it was not.
 **/
gboolean
rtcom_cp_interface_missed_call (
    RtcomCPInterface *plugin,
    const gchar account_id[],
    const gchar user[],
    const guint time_stamp,
    const gchar address[],
    guint *operation_id);

/**
 * rtcom_cp_interface_cancel_request
 *
 * @plugin:        The #RtcomCPInterface instance
 * @operation_ids: A #GArray of operation id's liked to be invalidated
 *
 * Tells the Contact Plugin that some operations need to be invalidated.
 * These are usually operations that require presence to be online for
 * an answer.
 *
 * Return value: %TRUE if the message was sent. %FALSE if it was not.
 **/
gboolean
rtcom_cp_interface_cancel_requests (
    RtcomCPInterface *plugin,
    GArray *operation_ids);

G_END_DECLS

#endif /* RTCOM_CP_INTERFACE_H_ */
