/*
 * This file is part of mission control
 *
 * Copyright (C) 2006 Nokia Corporation. All rights reserved.
 *
 * Contact: Andrei Laperie <andrei.laperie@nokia.com>
 */

#ifndef MISSION_CONTROL_LIB_H
#define MISSION_CONTROL_LIB_H

#ifndef DBUS_API_SUBJECT_TO_CHANGE
#define DBUS_API_SUBJECT_TO_CHANGE
#endif

#include "dbus/dbus-glib.h"
#include "dbus/dbus-glib-lowlevel.h"
#include "dbus/dbus.h"
#include <libtelepathy/tp-constants.h>
#include <libtelepathy/tp-conn.h>
#include <libtelepathy/tp-chan.h>
#include <libosso-rtcom/rtcom-account.h>
#include <sys/types.h>
#include <unistd.h>

#define MISSION_CONTROL_SERVICE "com.nokia.chavo.mission_control"
#define MISSION_CONTROL_IFACE "com.nokia.chavo.mission_control"
#define MISSION_CONTROL_PATH "/com/nokia/chavo/mission_control"

#define MISSIONCONTROL_TYPE       (mission_control_get_type ())

#define MISSIONCONTROL(obj)       (G_TYPE_CHECK_INSTANCE_CAST \
		                    ((obj), MISSIONCONTROL_TYPE, \
		                    MissionControl))

#define MISSIONCONTROL_CLASS(klass)       (G_TYPE_CHECK_CLASS_CAST \
		                            ((klass), MISSIONCONTROL_TYPE, \
				            MissionControlClass))

#define IS_MISSIONCONTROL(obj)    (G_TYPE_CHECK_INSTANCE_TYPE \
		                    ((obj), MISSIONCONTROL_TYPE))

#define IS_MISSIONCONTROL_CLASS(klass)    (G_TYPE_CHECK_CLASS_TYPE \
		                            ((klass), MISSIONCONTROL_TYPE))

#define MISSIONCONTROL_GET_CLASS(obj)     (G_TYPE_INSTANCE_GET_CLASS \
		                            ((obj), MISSIONCONTROL_TYPE, \
				            MissionControlClass))


typedef struct _missioncontrol MissionControl;
typedef struct _missioncontrolclass MissionControlClass;

typedef enum {
        MC_DISCONNECTED_ERROR,
        MC_INVALID_HANDLE_ERROR,
	MC_NO_MATCHING_CONNECTION_ERROR,
	MC_INVALID_ACCOUNT_ERROR, 
	MC_PRESENCE_FAILURE_ERROR,
	MC_NO_ACCOUNTS_ERROR,
	MC_NETWORK_ERROR,
	MC_CONTACT_DOES_NOT_SUPPORT_VOICE_ERROR,
        MC_LOWMEM_ERROR,
	MC_CHANNEL_REQUEST_GENERIC_ERROR,
        MC_CHANNEL_BANNED_ERROR,
        MC_CHANNEL_FULL_ERROR,
        MC_CHANNEL_INVITE_ONLY_ERROR,
	MC_LAST_ERROR
} MCError;

typedef enum {
  OSSO_RTCOM_PRESENCE_UNSET,
  OSSO_RTCOM_PRESENCE_OFFLINE,
  OSSO_RTCOM_PRESENCE_AVAILABLE,
  OSSO_RTCOM_PRESENCE_AWAY,
  OSSO_RTCOM_PRESENCE_EXTENDED_AWAY,
  OSSO_RTCOM_PRESENCE_HIDDEN,
  OSSO_RTCOM_PRESENCE_DO_NOT_DISTURB,
  LAST_OSSO_RTCOM_PRESENCE
} OssoRtcomPresence;

struct _missioncontrol
{
  DBusGProxy parent;

  pid_t client_pid;
  guint operation_id; /* A simple counter for execution order tracking */
  DBusGProxyCall *last_request; /* ? Would we want this to be a queue? */
  gboolean first_run;
};


struct _missioncontrolclass
{
  DBusGProxyClass parent_class;
}; 


GType mission_control_get_type(void);


/* Creates a new Mission Control client library object.
 * 
 * @param connection: The D-BUS connection for this object
 * @return A new mc (Mission Control) library object, or NULL if unsuccesful
 */

MissionControl *mission_control_new(DBusGConnection *connection);


/*
 * Sets presence for the accounts.
 * 
 * ? Would we also want to support the presence string identifier ?
 * @param presence: Integer specifying the presence status code
 * @param message: Optional presence associated message
 */
void mission_control_set_presence(MissionControl *self,
                                  OssoRtcomPresence presence,
				  const gchar *message);

/*
 * Gets the currently requested presence status.
 *
 * ? Would we also want to support the presence string identifier ?
 * @return The currently requested presence status
 */
OssoRtcomPresence
mission_control_get_presence(MissionControl *self);

/*
 * Gets the actual presence status.
 *
 * ? Would we also want to support the presence string identifier ?
 * @return The actual presence status
 */
OssoRtcomPresence
mission_control_get_presence_actual(MissionControl *self);

/*
 * Requests creation of a new channel, or join to an existing channel.
 * 
 * @param account: The account which will join a new channel or request
 * joining to an existing channel
 * @param channel_type: The channel type (see tp-chan.h for supported types)
 * @param handle: The handle we want to initiate the communication with
 * @param handle_type: The type of the handle we are initiating the
 *  communication with. See #TelepathyConnectionHandleType
 */

void mission_control_request_channel(MissionControl *self,
				     RtcomAccount *account,
				     const gchar *type,
                                     guint handle,
                                     TelepathyConnectionHandleType handle_type);

/* Requests creation of a new channel, or join to an existing
 * channel. Differs from the plain #mission_control_request_channel by
 * taking handles as strings, which will be resolved to integers by
 * MC.
 *
 * @param account: The account which will join a new channel or
 * request joining to an existing channel
 * @param channel_type: The channel type (see tp-chan.h for supported types)
 * @param handle: The handle we want to initiate the communication with
 * @param handle_type: The type of the handle we are initiating the
 * communication with. See #TelepathyConnectionHandleType
 */

void mission_control_request_channel_with_string_handle(MissionControl *self,
                                                        RtcomAccount *account,
                                                        const gchar *type,
                                                        const gchar *handle,
                                                        TelepathyConnectionHandleType Handle_type);
                                                 
/* Connect all accounts using default presence,
 * or HIDDEN if default presence is OFFLINE.
 * If accounts are already connected do nothing.
 */
void mission_control_connect_all_with_default_presence(MissionControl *self);

/*
 * Request a status code describing the status of the connection that the
 * provided account currently uses.
 *
 * @param account: The account whose connection status is inspected
 * @return A status code describing the status of the specified connection
 * eg. OFFLINE, CONNECTING, CONNECTED
 */
guint mission_control_get_connection_status(MissionControl *self,
					   RtcomAccount *account);


/* 
 * Request an array of strings representing the account names that have
 * an active connection.
 * 
 * @return A list of RtcomAccounts corresponding to the online
 * connections
 */
GSList *mission_control_get_online_connections(MissionControl *self);


/* Gets a connection object for the specified account name.
 * 
 * @param account: The account the connection is created for.
 * @return An existing TpConn object, NULL if the account is not connected
 */
TpConn *mission_control_get_connection(MissionControl *self,
				       RtcomAccount *account);



/* Cancels the current pending request, if any. Currently only works
   with mission_control_request_channel_with_string_handle, but
   support should be easy to expand. */

void mission_control_cancel_last_request(MissionControl *self);

/*
 * Gets the account corresponding to the connection object.
 * Note that as a result the caller owns a reference to the account object.
 *
 * @param connection: connection object to get the account for
 * @return The matching account object, NULL on error
 *
 */
RtcomAccount *mission_control_get_account_for_connection(MissionControl *self,
                                                         TpConn *connection);

#endif
