/*
 * This file is part of libosso-rtcom
 *
 * Copyright (C) 2006 Nokia. All rights reserved.
 *
 * Contact: naba.kumar@nokia.com
 */

#ifndef __RTCOM_ACCOUNT_H__
#define __RTCOM_ACCOUNT_H__

#include <glib.h>
#include <glib-object.h>

#include "rtcom-profile.h"

G_BEGIN_DECLS

#define RTCOM_TYPE_ACCOUNT rtcom_account_get_type()

#define RTCOM_ACCOUNT(obj) \
  (G_TYPE_CHECK_INSTANCE_CAST ((obj), \
  RTCOM_TYPE_ACCOUNT, RtcomAccount))

#define RTCOM_ACCOUNT_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_CAST ((klass), \
  RTCOM_TYPE_ACCOUNT, RtcomAccountwClass))

#define RTCOM_IS_ACCOUNT(obj) \
  (G_TYPE_CHECK_INSTANCE_TYPE ((obj), \
  RTCOM_TYPE_ACCOUNT))

#define RTCOM_IS_ACCOUNT_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_TYPE ((klass), \
  RTCOM_TYPE_ACCOUNT))

#define RTCOM_ACCOUNT_GET_CLASS(obj) \
  (G_TYPE_INSTANCE_GET_CLASS ((obj), \
  RTCOM_TYPE_ACCOUNT, RtcomAccountwClass))

typedef struct {
    GObject parent;
    gpointer priv;
} RtcomAccount;

typedef struct {
    GObjectClass parent_class;
} RtcomAccountClass;

GType rtcom_account_get_type (void);

RtcomAccount* rtcom_account_new (void);

typedef enum
{
    RTCOM_ACCOUNT_SETTING_ABSENT = 0,
    RTCOM_ACCOUNT_SETTING_FROM_ACCOUNT,
    RTCOM_ACCOUNT_SETTING_FROM_PROFILE,
    RTCOM_ACCOUNT_SETTING_FROM_PROXY,
} RtcomAccountSettingState;

/* returns NULL if unique name does not exist */
RtcomAccount *rtcom_account_lookup (const gchar *unique_name);
RtcomAccount *rtcom_account_lookup_with_profile (RtcomProfile *profile,
                                                 const gchar *account);
RtcomAccount *rtcom_account_lookup_with_vcard_field (const gchar *vcard_field,
                                                     const gchar *account);
void rtcom_account_free (RtcomAccount* account);
void rtcom_account_clear_cache (void);

/* returns newly-created account, enabled by default */
RtcomAccount *rtcom_account_create (RtcomProfile *profile);

/* this function only deletes the account from database,
 * the account struct itself must be freed separately */
gboolean rtcom_account_delete (RtcomAccount *account);

/* lists returned by these functions should be freed with
 * rtcom_accounts_list_free */
GList *rtcom_accounts_list (void);
GList *rtcom_accounts_list_by_enabled (gboolean enabled);
GList *rtcom_accounts_list_by_profile (RtcomProfile *profile);
GList *rtcom_accounts_list_by_vcard_field (const gchar *vcard_field);
void rtcom_accounts_list_free (GList *list);

/* filter a list of accounts according to whether a function returns TRUE,
 * freeing the list and any accounts which are filtered out, and returning a
 * new list which must be freed with rtcom_accounts_list_free. */
typedef gboolean (*RtcomAccountFilter) (RtcomAccount *account, gpointer data);
GList *rtcom_accounts_filter (GList *accounts,
                              RtcomAccountFilter filter,
                              gpointer data);

/* a unique identifier string for this account */
gchar *rtcom_account_get_unique_name (RtcomAccount *account);

/* get profile */
RtcomProfile *rtcom_account_get_profile (RtcomAccount *account);

/* human-readable name */
gchar *rtcom_account_get_display_name (RtcomAccount *account);
gboolean rtcom_account_set_display_name (RtcomAccount *account,
                                         const gchar *name);

/* whether account is enabled or disabled */
gboolean rtcom_account_is_enabled (RtcomAccount *account);
gboolean rtcom_account_set_enabled (RtcomAccount *account,
                                    const gboolean enabled);

/* the following methods retrieve a parameter from the account or the
 * default from the profile if the account does not set the value */
RtcomAccountSettingState rtcom_account_get_param_boolean (RtcomAccount *account,
                                                          const gchar *name,
                                                          gboolean *value);
RtcomAccountSettingState rtcom_account_get_param_int (RtcomAccount *account,
                                                      const gchar *name,
                                                      gint *value);
RtcomAccountSettingState rtcom_account_get_param_string (RtcomAccount *account,
                                                         const gchar *name,
                                                         gchar **value);

/* for every parameter (both optional and mandatory) defined by the
 * protocol, get a hash table of the params from the account or
 * the default profile. each setting is stored in a GValue. */
GHashTable *rtcom_account_get_params (RtcomAccount *account);

/* Set functions. Returns true if success, else false is returned */
gboolean rtcom_account_set_param_boolean (RtcomAccount *account,
                                          const gchar *name,
                                          gboolean value);
gboolean rtcom_account_set_param_int (RtcomAccount *account,
                                      const gchar *name,
                                      gint value);
gboolean rtcom_account_set_param_string (RtcomAccount *account,
                                         const gchar *name,
                                         const gchar *value);

gboolean rtcom_account_unset_param (RtcomAccount *account, const gchar *name);

/* returns TRUE if the account information, along with the profile defaults
 * has all mandatory fields (declared by the protocol) set */
gboolean rtcom_account_is_complete (RtcomAccount *account);

/* returns TRUE if there is at least one account with a profile that supports
 * being invisible */
gboolean rtcom_account_exists_supporting_invisible ();
  
G_END_DECLS

#endif /* __RTCOM_ACCOUNT_H__ */
