#include <linux/module.h>
#include <linux/delay.h>

#include <asm/arch/hardware.h>
#include <asm/arch/gpio.h>
#include <asm/arch/board.h>
#include <asm/arch/omapfb.h>
#include <asm/arch/lcd_lph8923.h>

#include <linux/spi/spi.h>
#include <asm/arch/mcspi.h>

#include "../../cbus/tahvo.h"

/* #define OMAPFB_DBG 1 */

#include "debug.h"

#define LPH8923_MODULE_NAME		"lcd_lph8923"

#define LPH8923_VER_NON_BUGGY		3

static struct {
	int		enabled;
	int		version;

	u8		display_id[3];
	unsigned int	saved_bklight_level;
	unsigned long	hw_guard_end;		/* next value of jiffies
						   when we can issue the
						   next sleep in/out command */
	unsigned long	hw_guard_wait;		/* max guard time in jiffies */
	struct omapfb_device *fbdev;
	struct spi_device *spi;
} lph8923;

static struct lcd_panel lph8923_panel;

#define LPH8923_CMD_READ_DISP_ID	0x04
#define LPH8923_CMD_READ_RED		0x06
#define LPH8923_CMD_READ_GREEN		0x07
#define LPH8923_CMD_READ_BLUE		0x08
#define LPH8923_CMD_READ_DISP_STATUS	0x09
#define LPH8923_CMD_SLEEP_IN		0x10
#define LPH8923_CMD_SLEEP_OUT		0x11
#define LPH8923_CMD_DISP_OFF		0x28
#define LPH8923_CMD_DISP_ON		0x29

static struct lcd_panel lph8923_panel;

#define LPH8923_SPEED_HZ	12000000

static int lph8923_spi_probe(struct spi_device *spi)
{
	DBGENTER(1);

	DBGPRINT(1, "spi %p\n", spi);
	lph8923.spi = spi;

	omapfb_register_panel(&lph8923_panel);

	return 0;
}

static int lph8923_spi_remove(struct spi_device *spi)
{
	DBGENTER(1);

	lph8923.spi = NULL;

	return 0;
}

static void lph8923_shutdown(struct spi_device *spi)
{
	struct lph8923_platform_data *pdata;

	pdata = spi->dev.platform_data;
	pdata->shutdown(pdata);
}

static struct spi_driver lph8923_spi_driver = {
	.driver = {
		.name	= LPH8923_MODULE_NAME,
		.bus	= &spi_bus_type,
		.owner	= THIS_MODULE,
	},
	.probe	= lph8923_spi_probe,
	.remove	= __devexit_p(lph8923_spi_remove),
	.shutdown = lph8923_shutdown,
};

static int lph8923_drv_init(void)
{
	spi_register_driver(&lph8923_spi_driver);

	return 0;
}
module_init(lph8923_drv_init);

static void lph8923_drv_cleanup(void)
{
	spi_unregister_driver(&lph8923_spi_driver);
}
module_exit(lph8923_drv_cleanup);

static void lph8923_read(int cmd, u8 *buf, int len)
{
	struct spi_message	m;
	struct spi_transfer	t[2];
	u16			w;

	BUG_ON(lph8923.spi == NULL);

	spi_message_init(&m);
	m.spi = lph8923.spi;

	memset(t, 0, sizeof(t));
	if (len > 1) {
		cmd <<= 1;
		t[0].bits_per_word = 10;
	} else
		t[0].bits_per_word = 9;

	w		= cmd;
	t[0].tx_buf	= &w;
	t[0].len	= 2;

	spi_message_add_tail(&t[0], &m);

	if (len) {
		t[1].rx_buf	= buf;
		t[1].len	= len;

		spi_message_add_tail(&t[1], &m);
	}

	spi_sync(m.spi, &m);
}

static void lph8923_write(int cmd, const u8 *buf, int len)
{
	struct spi_message	m;
	struct spi_transfer	t[2];
	u16			w;

	BUG_ON(lph8923.spi == NULL);

	spi_message_init(&m);
	m.spi = lph8923.spi;

	memset(t, 0, sizeof(t));

	t[0].bits_per_word	= 9;
	w			= cmd;
	t[0].tx_buf		= &w;
	t[0].len		= 2;

	spi_message_add_tail(&t[0], &m);

	if (len) {
		t[1].bits_per_word	= 9;
		t[1].tx_buf		= buf;
		t[1].len		= len;
	}

	spi_sync(m.spi, &m);
}

static inline void lph8923_cmd(int cmd)
{
	lph8923_write(cmd, NULL, 0);
}

static inline void lph8923_set_16bit_mode(void)
{
	u16 par = 0x0150;

	lph8923_write(0x3a, (u8 *)&par, 2);
}

static void lph8923_send_init_string(void)
{
	u16 initpar[] = { 0x0102, 0x0100, 0x0100 };

	lph8923_write(0xc2, (u8 *)initpar, sizeof(initpar));
	lph8923_set_16bit_mode();
}

static void hw_guard_start(int guard_msec)
{
	lph8923.hw_guard_wait = msecs_to_jiffies(guard_msec);
	lph8923.hw_guard_end = jiffies + lph8923.hw_guard_wait;
}

static void hw_guard_wait(void)
{
	unsigned long wait = lph8923.hw_guard_end - jiffies;

	if ((long)wait > 0 && wait <= lph8923.hw_guard_wait) {
		set_current_state(TASK_UNINTERRUPTIBLE);
		schedule_timeout(wait);
	}
}

static void lph8923_set_sleep_mode(int on)
{
	int cmd, sleep_time = 5;

	if (on)
		cmd = LPH8923_CMD_SLEEP_IN;
	else
		cmd = LPH8923_CMD_SLEEP_OUT;
	hw_guard_wait();
	lph8923_cmd(cmd);
	hw_guard_start(120);
	/* When we enable the panel, it seems we _have_ to sleep
	 * 120 ms before sending the init string */
	if (!on)
		sleep_time = 120;
	msleep(sleep_time);
}

static void lph8923_set_display_state(int enabled)
{
	int cmd = enabled ? LPH8923_CMD_DISP_ON : LPH8923_CMD_DISP_OFF;

	lph8923_cmd(cmd);
}

static void lph8923_detect(void)
{
	struct lph8923_platform_data *pdata;

	lph8923_read(LPH8923_CMD_READ_DISP_ID, lph8923.display_id, 3);
	printk(KERN_INFO "Moscow display id: %02x%02x%02x\n",
		lph8923.display_id[0], lph8923.display_id[1],
		lph8923.display_id[2]);

	lph8923.version = LPH8923_VER_NON_BUGGY;

	pdata = lph8923.spi->dev.platform_data;
	lph8923_panel.data_lines = pdata->data_lines;
}

static int lph8923_enabled(void)
{
	u32 disp_status;
	int enabled;

	lph8923_read(LPH8923_CMD_READ_DISP_STATUS, (u8 *)&disp_status, 4);
	disp_status = __be32_to_cpu(disp_status);
	enabled = (disp_status & (1 << 17)) && (disp_status & (1 << 10));
	DBGPRINT(1, ": panel %senabled by bootloader (status 0x%04x)\n",
		enabled ? "" : "not ", disp_status);
	return enabled;
}

static int lph8923_panel_init(struct omapfb_device *fbdev)
{
	lph8923.fbdev = fbdev;

	lph8923_detect();
	lph8923.enabled = lph8923_enabled();

	return 0;
}

static void lph8923_panel_cleanup(void)
{
}

static int lph8923_panel_set_bklight_level(unsigned int level)
{
	if (level > tahvo_get_max_backlight_level())
		return -EINVAL;
	if (!lph8923.enabled) {
		lph8923.saved_bklight_level = level;
		return 0;
	}
	tahvo_set_backlight_level(level);

	return 0;
}

static unsigned int lph8923_panel_get_bklight_level(void)
{
	return tahvo_get_backlight_level();
}

static unsigned int lph8923_panel_get_bklight_max(void)
{
	return tahvo_get_max_backlight_level();
}

static int lph8923_panel_enable(void)
{
	if (lph8923.enabled)
		return 0;

	lph8923_set_sleep_mode(0);
	lph8923.enabled = 1;
	lph8923_send_init_string();
	lph8923_set_display_state(1);
	lph8923_panel_set_bklight_level(lph8923.saved_bklight_level);

	return 0;
}

static void lph8923_panel_disable(void)
{
	if (!lph8923.enabled)
		return;
	lph8923.saved_bklight_level = lph8923_panel_get_bklight_level();
	lph8923_panel_set_bklight_level(0);
	lph8923_set_display_state(0);
	lph8923_set_sleep_mode(1);
	lph8923.enabled = 0;
}

static unsigned long lph8923_panel_get_caps(void)
{
	return OMAPFB_CAPS_SET_BACKLIGHT;
}

static u16 read_first_pixel(void)
{
	u8 b;
	u16 pixel;

	lph8923_read(LPH8923_CMD_READ_RED, &b, 1);
	pixel = (b >> 1) << 11;
	lph8923_read(LPH8923_CMD_READ_GREEN, &b, 1);
	pixel |= b << 5;
	lph8923_read(LPH8923_CMD_READ_BLUE, &b, 1);
	pixel |= (b >> 1);

	return pixel;
}

static int lph8923_panel_test(int test_num)
{
	static const u16 test_values[4] = {
		0x0000, 0xffff, 0xaaaa, 0x5555,
	};
	int i;

	if (test_num != LCD_LPH8923_TEST_RGB_LINES)
		return LCD_LPH8923_TEST_INVALID;

	for (i = 0; i < ARRAY_SIZE(test_values); i++) {
		int delay;
		unsigned long tmo;

		omapfb_write_first_pixel(lph8923.fbdev, test_values[i]);
		tmo = jiffies + msecs_to_jiffies(100);
		delay = msecs_to_jiffies(25);
		while (1) {
			u16 pixel;

			set_current_state(TASK_UNINTERRUPTIBLE);
			schedule_timeout(delay);
			pixel = read_first_pixel();
			if (pixel == test_values[i])
				break;
			if (time_after(jiffies, tmo)) {
				printk(KERN_ERR "Moscow RGB I/F test failed: "
				       "expecting %04x, got %04x\n",
				       test_values[i], pixel);
				return LCD_LPH8923_TEST_FAILED;
			}
			delay = msecs_to_jiffies(10);
		}
	}

	return 0;
}

static struct lcd_panel lph8923_panel = {
	.name		= "lph8923",
	.config		= OMAP_LCDC_PANEL_TFT,

	.bpp		= 16,
	.data_lines	= 16,
	.x_res		= 800,
	.y_res		= 480,
	.pixel_clock	= 21940,
	.hsw		= 50,
	.hfp		= 20,
	.hbp		= 15,
	.vsw		= 2,
	.vfp		= 1,
	.vbp		= 3,

	.init		= lph8923_panel_init,
	.cleanup	= lph8923_panel_cleanup,
	.enable		= lph8923_panel_enable,
	.disable	= lph8923_panel_disable,
	.get_caps	= lph8923_panel_get_caps,
	.set_bklight_level= lph8923_panel_set_bklight_level,
	.get_bklight_level= lph8923_panel_get_bklight_level,
	.get_bklight_max= lph8923_panel_get_bklight_max,
	.run_test	= lph8923_panel_test,
};
