/**
 * @file libgalago/galago-dbus.h D-BUS Support API
 * @ingroup Utility
 *
 * @Copyright (C) 2004-2006 Christian Hammond.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#ifndef _GALAGO_DBUS_H_
#define _GALAGO_DBUS_H_

#ifndef _DOXYGEN_
#define DBUS_API_SUBJECT_TO_CHANGE
#endif

#include <dbus/dbus.h>
#include <libgalago/galago-object.h>
#include <libgalago/galago-value.h>

#define GALAGO_DBUS_SERVICE "org.freedesktop.Galago"

#define GALAGO_DBUS_ERROR_INVALID_ATTRIBUTE "org.freedesktop.Galago.Error.InvalidAttribute"
#define GALAGO_DBUS_ERROR_OBJECT_NOT_FOUND "org.freedesktop.Galago.Error.ObjectNotFound"

G_BEGIN_DECLS

/**************************************************************************/
/** @name Utility functions                                             */
/**************************************************************************/
/*@{*/

/**
 * Appends a string to a message, or an empty string if the passed string
 * is NULL.
 *
 * @param iter The message iterator.
 * @param str  The string, or NULL.
 */
void galago_dbus_message_iter_append_string_or_nil(DBusMessageIter *iter,
												   const char *str);

/**
 * Returns a string that the iterator points to.
 *
 * If the returned string is an empty string (""), this returns NULL.
 *
 * @param iter The iterator.
 *
 * @return The string or NULL.
 */
const char *galago_dbus_message_iter_get_string_or_nil(DBusMessageIter *iter);

/**
 * Appends an object to a D-BUS message.
 *
 * @param iter   The D-BUS message iterator.
 * @param object The object to append.
 */
void galago_dbus_message_iter_append_object(DBusMessageIter *iter,
											const GalagoObject *object);

/**
 * Appends a list of objects to a D-BUS message.
 *
 * @param iter The D-BUS message iterator.
 * @param type The type of the objects inside.
 * @param list The list of objects to append.
 */
void galago_dbus_message_iter_append_object_list(DBusMessageIter *iter,
												 GType type, GList *list);

/**
 * Returns an object of the specified type from a D-BUS message.
 *
 * @param iter The D-BUS message iterator.
 * @param type The GType.
 *
 * @return The object.
 */
void *galago_dbus_message_iter_get_object(DBusMessageIter *iter, GType type);

/**
 * Returns a list of objects of the specified type from a D-BUS message.
 *
 * The returned list must be destroyed.
 *
 * @param iter The D-BUS message iterator.
 * @param type The GType.
 *
 * @return The list.
 */
GList *galago_dbus_message_iter_get_object_list(DBusMessageIter *iter,
												GType type);

/**
 * Creates a D-BUS message to the specified object.
 *
 * @param object   The object.
 * @param name     The message name.
 * @param reply    TRUE if a reply is expected.
 * @param ret_iter The returned message iterator, if not NULL.
 *
 * @return The D-BUS message.
 */
DBusMessage *galago_dbus_message_new_method_call(const GalagoObject *object,
												 const char *name,
												 gboolean reply,
												 DBusMessageIter *ret_iter);

/**
 * Creates a D-BUS message to the specified object with a va_list of
 * parameters.
 *
 * @param object   The object.
 * @param name     The message name.
 * @param reply    TRUE if a reply is expected.
 * @param args     The @c va_list of parameters.
 *
 * @return The D-BUS message.
 */
DBusMessage *galago_dbus_message_new_method_call_vargs(
	const GalagoObject *object, const char *name,
	gboolean reply, va_list args);

/**
 * Creates a D-BUS message to the specified object with a list of parameters.
 *
 * The parameters passed are in GalagoValue, pointer-to-data form.
 *
 * @param object   The object.
 * @param name     The message name.
 * @param reply    TRUE if a reply is expected.
 * @param ...      The list of parameters.
 *
 * @return The D-BUS message.
 */
DBusMessage *galago_dbus_message_new_method_call_args(
	const GalagoObject *object, const char *name,
	gboolean reply, ...);

/**
 * Sends a new D-BUS message with an object, name, and parameters.
 *
 * The parameters passed are in GalagoValue, pointer-to-data form.
 *
 * @param object   The object.
 * @param name     The message name.
 * @param ...      The list of parameters.
 */
void galago_dbus_send_message(const GalagoObject *object,
							  const char *name, ...);


/**
 * Sends a new D-BUS message with an object, name, and parameters, and
 * waits for a reply with a variable list of return types.
 *
 * The parameters passed are in GalagoValue, pointer-to-data form.
 *
 * @param object       The object.
 * @param name         The message name.
 * @param return_types A list of expected return types, in order.
 * @param ...          The list of parameters.
 *
 * @return The returned data in a list. The list will need to be g_freed,
 *         as may some of its data.
 */
GList *galago_dbus_send_message_with_reply_list(const GalagoObject *object,
												const char *name,
												GList *return_types, ...);

/**
 * Sends a new D-BUS message with an object, name, and parameters, and
 * waits for a reply.
 *
 * The parameters passed are in GalagoValue, pointer-to-data form.
 *
 * @param object      The object.
 * @param name        The message name.
 * @param return_type The expected return type.
 * @param ...         The list of parameters.
 *
 * @return The returned data. This may need to be g_freed.
 */
void *galago_dbus_send_message_with_reply(const GalagoObject *object,
										  const char *name,
										  GalagoValue *return_type, ...);

/**
 * Pushes an object and all its internal data to the server.
 *
 * @internal This is used internally.
 *
 * @param object The object to push.
 */
void galago_dbus_object_push_full(GalagoObject *object);

/*@}*/

G_END_DECLS

#endif /* _GALAGO_DBUS_H_ */
