/**
 * @file libgalago/galago-person.h Galago Person API
 * @ingroup Presence
 *
 * #GalagoPerson represents a single person and their personal information
 * (such as name and address). As people can have multiple IM accounts, each
 * #GalagoPerson can have many #GalagoAccount%s associated with it
 * (created using galago_service_create_account()).
 *
 * @Copyright (C) 2004-2006 Christian Hammond
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#ifndef _GALAGO_PERSON_H_
#define _GALAGO_PERSON_H_

typedef struct _GalagoPerson        GalagoPerson;
typedef struct _GalagoPersonClass   GalagoPersonClass;
typedef struct _GalagoPersonPrivate GalagoPersonPrivate;

#include <libgalago/galago-account.h>
#include <libgalago/galago-image.h>
#include <libgalago/galago-object.h>
#include <libgalago/galago-service.h>

/**
 * A person, which may have multiple accounts.
 */
struct _GalagoPerson
{
	GalagoObject parent_object;

	GalagoPersonPrivate *priv;

#ifndef _DOXYGEN_
	void (*reserved_1)(void);
	void (*reserved_2)(void);
	void (*reserved_3)(void);
	void (*reserved_4)(void);
#endif
};

/**
 * GalagoPerson's class.
 */
struct _GalagoPersonClass
{
	GalagoObjectClass parent_class;

	/* Signals */
	void (*account_added)(GalagoPerson *person, GalagoAccount *account);
	void (*account_removed)(GalagoPerson *person, GalagoAccount *account);
	void (*photo_set)(GalagoPerson *person, GalagoImage *photo);

#ifndef _DOXYGEN_
	void (*reserved_1)(void);
	void (*reserved_2)(void);
	void (*reserved_3)(void);
	void (*reserved_4)(void);
#endif
};

#define GALAGO_TYPE_PERSON (galago_person_get_type())

/** Cast @c obj to a GalagoPerson. */
#define GALAGO_PERSON(obj) \
		(G_TYPE_CHECK_INSTANCE_CAST((obj), GALAGO_TYPE_PERSON, GalagoPerson))

/** Cast @c klass to a GalagoPerson. */
#define GALAGO_PERSON_CLASS(klass) \
		(G_TYPE_CHECK_CLASS_CAST((klass), GALAGO_TYPE_PERSON, \
								 GalagoPersonClass))
/** Check if @c obj is a GalagoPerson. */
#define GALAGO_IS_PERSON(obj) \
		(G_TYPE_CHECK_INSTANCE_TYPE((obj), GALAGO_TYPE_PERSON))

/** Check if @c klass is a GalagoPersonClass. */
#define GALAGO_IS_PERSON_CLASS(klass) \
		(G_TYPE_CHECK_CLASS_TYPE((klass), GALAGO_TYPE_PERSON))

/** Get the class for the GalagoPerson @c obj. */
#define GALAGO_PERSON_GET_CLASS(obj) \
		(G_TYPE_INSTANCE_GET_CLASS((obj), GALAGO_TYPE_PERSON, \
								   GalagoPersonClass))

#define GALAGO_DBUS_PERSON_INTERFACE "org.freedesktop.Galago.Person"

G_BEGIN_DECLS

/**************************************************************************/
/** @name Common Attribute Definitions                                    */
/**************************************************************************/
/*{*/

#define GALAGO_PERSON_ATTR_UID          "uid"
#define GALAGO_PERSON_ATTR_FIRST_NAME   "first-name"
#define GALAGO_PERSON_ATTR_MIDDLE_NAME  "middle-name"
#define GALAGO_PERSON_ATTR_LAST_NAME    "last-name"
#define GALAGO_PERSON_ATTR_ADDRESS      "address"
#define GALAGO_PERSON_ATTR_CITY         "city"
#define GALAGO_PERSON_ATTR_STATE        "state"
#define GALAGO_PERSON_ATTR_COUNTRY      "country"
#define GALAGO_PERSON_ATTR_ZIPCODE      "zipcode"

/*@}*/

/**************************************************************************/
/** @name Person API                                                      */
/**************************************************************************/
/*@{*/

/**
 * Returns GalagoPerson's GType.
 *
 * @return The GType.
 */
GType galago_person_get_type(void);

/**
 * Sets a person to be the "Me" person.
 *
 * This is a one-way thing. You can't make a "Me" person a normal
 * person.
 *
 * @internal Please use galago_get_me() instead of ever using this.
 *
 * @param person The person.
 */
void galago_person_set_me(GalagoPerson *person);

/**
 * Returns whether or not a person is the "Me" person.
 *
 * @param person The person.
 *
 * @return TRUE if the person is the "Me" person, or FALSE.
 */
gboolean galago_person_is_me(const GalagoPerson *person);

/**
 * Returns a person's application-specific unique ID.
 *
 * @param person The person.
 *
 * @return The person's ID.
 */
const char *galago_person_get_id(const GalagoPerson *person);

/**
 * Returns a person's session ID.
 *
 * @param person The person.
 *
 * @return The person's session ID.
 */
const char *galago_person_get_session_id(const GalagoPerson *person);

/**
 * Returns the display name of a person, if any.
 *
 * The returned display name is from a static buffer. If you wish to store
 * the name, you must @c g_strdup it.
 *
 * @param person The person.
 *
 * @return The person's display name, or NULL.
 */
const char *galago_person_get_display_name(const GalagoPerson *person);

/**
 * Sets the person's photo.
 *
 * The person should be a local person.
 *
 * @param person The person.
 * @param photo  The photo to set.
 */
void galago_person_set_photo(GalagoPerson *person, GalagoImage *photo);

/**
 * Returns the person's photo.
 *
 * @param person The person.
 * @param query  TRUE if a remote query should be done if there is no
 *               local photo found, or FALSE.
 *
 * @return The photo, if found, or NULL.
 */
GalagoImage *galago_person_get_photo(const GalagoPerson *person,
									 gboolean query);

/**
 * Returns whether the person has any accounts at all.
 *
 * @param person The person.
 * @param query  TRUE if a query should be done to check the server-side
 *               state of the accounts, or FALSE.
 *
 * @return TRUE if the person has at least one account, or FALSE otherwise.
 */
gboolean galago_person_has_accounts(const GalagoPerson *person,
									gboolean query);

/**
 * Returns a list of accounts in the person.
 *
 * @param person The person.
 * @param query  TRUE if a remote query should be done if there is no
 *               local account found, or FALSE.
 *
 * @return The list of accounts.
 */
GList *galago_person_get_accounts(const GalagoPerson *person, gboolean query);

/**
 * Returns the most available "priority" account.
 *
 * @param person The person.
 *
 * @return The priority account.
 */
GalagoAccount *galago_person_get_priority_account(const GalagoPerson *person);

/**
 * Returns the account with the specified username and service.
 *
 * @param person   The person.
 * @param service  The service.
 * @param username The username.
 * @param query    TRUE if a remote query should be done if there is no
 *                 local account found, or FALSE.
 *
 * @return The account, if found, or NULL.
 */
GalagoAccount *galago_person_get_account(const GalagoPerson *person,
										 const GalagoService *service,
										 const char *username,
										 gboolean query);

/*@}*/

G_END_DECLS

#endif /* _GALAGO_PERSON_H_ */
