/* Certificate Management library
 * 
 * Copyright (C) 2005 Nokia Corporation. All rights reserved.
 * Author: Ed Bartosh <Eduard.Bartosh@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License version 2.1 as published by the Free Software Foundation
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/**
   @file cst.h

   Certificate storage API
*/

/** @defgroup OpenClose Open, close and related */
/** @defgroup Cert Certificate: main */
/** @defgroup CertSearch Certificate: search */
/** @defgroup CertCapab Certificate: S/MIME capabilities */
/** @defgroup Key Private (public) key related */
/** @defgroup CertKeyDefault Certificates and key (assign, default) */
/** @defgroup CRL CRL related */
/** @defgroup ImportExport Import/export functions*/

#ifndef CST_H_
#define CST_H_


#include <openssl/x509.h>
#include <openssl/evp.h>
#include <stdio.h>
#include <time.h>
#include <glib.h>
#include <db/db.h>

#ifdef __cplusplus
extern "C" {
#endif


/** @name Error codes */
/*@{*/    
#define CST_ERROR_OK                    0 
#define CST_ERROR_NOT_FOUND             1
#define CST_ERROR_STRUCTURE_CORRUPT     2
#define CST_ERROR_CREATE_FILE           3
#define CST_ERROR_CERT_EXIST            4
#define CST_ERROR_CRL_EXIST             6
#define CST_ERROR_STORAGE_IS_READONLY   7
#define CST_ERROR_KEY_NOTFOUND          8
#define CST_ERROR_CERT_NOTFOUND         9
#define CST_ERROR_NOT_IMPLEMENTED      10
#define CST_ERROR_NOT_INIT             11
#define CST_ERROR_UNDEF                12
#define CST_ERROR_PARAM_INCORRECT      13
#define CST_ERROR_NOT_OPEN             14
#define CST_ERROR_ASSIGN_INCORRECT     15
#define CST_ERROR_CRL_NOT_VALID        16
#define CST_ERROR_CHAIN_INCOMPLETE     17
#define CST_ERROR_CAPABILITY_NOTFOUND  18    
#define CST_ERROR_INCORRECT_PURPOSE    19

#define CST_ERROR_IO                   20
#define CST_ERROR_NOSPC                21
#define CST_ERROR_DBSTRUCTURE_CORRUPT  22

#define CST_ERROR_LOCK                 23

#define CST_ERROR_PASSWORD_WRONG       24    
#define CST_ERROR_BAD_INTERNAL_FORMAT  25
#define CST_ERROR_EXPORT               CST_ERROR_UNDEF    
#define CST_ERROR_UNDEF_FILE_ERROR     CST_ERROR_UNDEF
#define CST_ERROR_CANCEL               30
    
/*@}*/    

#if 1    
    #define CST_DEBUG_LOCK    
#endif
    
/* Structures */

/** 
    @brief used for certID and keyID
    
    see: cst_const.h
*/    
typedef guint cst_t_seqnum;         

/** 
    @brief used for mcount
    
    see: cst_const.h
*/
typedef guint64 cst_t_mcount;

/** @brief Cert, CRL ang key storage */
    typedef struct CST_st {
        int readonly;               /**< @brief Now not used */
        char *filename;             /**< @brief Name of file */
        unsigned char *password;    /**< @brief Now not used */
        int stub;                   /**< @brief Now not used */

        /** @brief Certificate list (main index) 
          
            Order by Issuer name and Serial number 
        */
        GTree *certs;         

        /** @brief Key list (main index) 
          
            Order by name (Subject name) 
        */
        GTree *keys;

        /* @brief CRL list */
        /* GTree *crls; */

        /** @brief Simple CRL list */
        GSList *simple_crls;

        /* Index structures */
        GTree *idx_cert_name;   /**< Cert index order by Subject name */
        GTree *idx_cert_dns;    /**< Cert index order by Domain name (SSL) */
        GTree *idx_cert_email;  /**< Cert index order by Email (S/MIME) */
        GHashTable *idx_cert_serial; /**< Cert index order by serial */
        GHashTable *idx_cert_fingerprint; /**< Cert index order by fingerprint */

        /* New indexes by integer uid */
        GHashTable *idx_cert_uid; /**< Cert index order by certID */
        GHashTable *idx_key_uid;  /**< Key index order by keyID */

        DB * db; /**< Pointer to Berkley DB structure */

        cst_t_mcount modification_count; /**< How many times the record was
                                          updated */

        GStaticRWLock rwlock; /**< Locking structure */
    } CST;

/** 
    @brief Certificate folders (8 bit) 
    
    see: cst_const.h 
*/
    typedef guint8 cst_t_cert_folder;  
    
/** @name Folders */    
/*@{*/    
/** @brief Certificate authority */    
#define CST_FOLDER_CA       1  
/** @brief Personal */    
#define CST_FOLDER_PERSONAL 2  
/** @brief Other */    
#define CST_FOLDER_OTHER    3    
/** @brief Sites */    
#define CST_FOLDER_SITE     4  
/** @brief Unknown */    
#define CST_FOLDER_UNKNOWN  0  
/*@}*/
    
/** 
    @brief Certificate purposes (32 bit) 
    
    see: cst_const.h
*/
    typedef guint32 cst_t_cert_purpose;
/** @name Purposes */    
/*@{*/    
#define CST_PURPOSE_NONE        0x0000 
#define CST_PURPOSE_CA          0x0001 
#define CST_PURPOSE_SMIME_SGN   0x0002
#define CST_PURPOSE_SMIME_ENC   0x0004
#define CST_PURPOSE_SSL_SERVER  0x0008
#define CST_PURPOSE_SSL_CLIENT  0x0010
#define CST_PURPOSE_SSL_WLAN    0x0020    
#define CST_PURPOSE_CRL_SIGN    0x0040
#define CST_PURPOSE_ALL         0xFFFF
/*@}*/    

/** @name States 

    see function: CST_get_state() */     
/*@{*/    
#define CST_STATE_VALID         0x0001
#define CST_STATE_NOTVALID      0x0002
#define CST_STATE_EXPIRED       0x0004
#define CST_STATE_REVOKED       0x0008
/*@}*/    

/** @name Helper stack defines */
/*@{*/    
#define CST_STACK_OF_X509 STACK_OF(X509)
#define CST_STACK_OF_CRL STACK_OF(X509_CRL)    
#define CST_STACK_OF_ASN1_OBJECT STACK_OF(ASN1_OBJECT)    
/*@}*/    
    
/* Storage config and etc. functions */

/**
    @ingroup OpenClose
    @brief Open default storage.

    If find gconf key "/apps/osso/certman/filename" then
    open file use this value. Else open file "storage.cst". 
    If file not exist then created.

    @param readonly  If storage will be readonly (NOT USED)
    @param password  Password for storage or NULL if password is empty (NOT USED)
    @return NULL if error, see CST_lasError()
    
    Possible error:  CST_ERROR_NOT_FOUND, CST_ERROR_DBSTRUCTURE_CORRUPT, 
    CST_ERROR_IO, CST_ERROR_NOSPC, CST_ERROR_LOCK
*/
    extern CST *CST_open(const int readonly, unsigned char *password);

/**
   @ingroup OpenClose
   @brief Open local storage from file 'filename'. If file not exist then created. 

   @param filename  Filename of local storage
   @param readonly  If storage will be readonly (NOT USED)
   @param password  Password for storage or NULL if password is empty (NOT USED)
   @return NULL if error, see CST_lasError()
   
   Possible error:  CST_ERROR_NOT_FOUND, CST_ERROR_DBSTRUCTURE_CORRUPT
   CST_ERROR_IO, CST_ERROR_NOSPC, CST_ERROR_LOCK
   
   @ref example1
*/
    extern CST *CST_open_file(const char *filename, const int readonly,
                              unsigned char *password);

/**
   @ingroup OpenClose
   @brief Create empty local storage in file 'filename'.

   Only create file, not open it.

   @param filename  Filename of local storage
   @param password  Password for storage or NULL if password is empty (NOT USED)
   @return Error code
   
   Possible error:  CST_ERROR_NOT_FOUND, CST_ERROR_DBSTRUCTURE_CORRUPT

   @deprecated This function is deprecated. Instead use CST_open_file() or CST_open().
*/
    extern int CST_create_file(const char *filename, unsigned char *password);

/**
   @ingroup OpenClose
   Save storage.
    
   @param st  Pointer to storage structure
   @return Error code

   @deprecated This function is deprecated.
   
   Possible error:  CST_ERROR_NOT_FOUND, CST_ERROR_DBSTRUCTURE_CORRUPT
   
   @ref example1
*/
    extern int CST_save(CST * st);


/**
   @ingroup OpenClose
   Free resources used by storage.

   @param   st Pointer to storage structure
   
   @ref example1
*/
    extern void CST_free(CST * st);


/**
   @ingroup OpenClose
   Last error code. If error then return last error code.
   
   @return Last error code
*/
    extern int CST_last_error();


/* Import/export/backup */

/**
   @ingroup OpenClose
   Backup local (file) storage.

   @param st            Pointer to storage structure
   @param filename      Filename for backup
   @param password      Password (NOT USED)
   @return Error code

   Possible error: CST_ERROR_LOCK, 
*/
    extern int CST_backup(CST * st, const char *filename, unsigned char *password);


/* Certificate */

    typedef int (*cst_pkcs12_confirm_cb) (X509 * xcert,
                            cst_t_cert_folder * folder,
                            cst_t_cert_purpose * purpose,
                            unsigned char ** out_password,
                            int is_pair, 
                            int *cancel,
                            void *data);
    
    typedef int (*cst_pkcs12_error_cb) (X509 * xcert, int error, void *data);
    
/**
   @ingroup ImportExport
   Import certificate and private key from the file (PKCS12 format).
   
   @param st         Pointer to storage structure
   @param file       Input file
   @param confirm_cb Callback function to confirm import of certificate or key
   @param error_cb   Callback function to inform about error
   @param password   Password of input file
   @param user_data  User data, that need in both callback functions
   @param error      Error code
   @return Error code
  
   Possible error: CST_ERROR_PASSWORD_WRONG, CST_ERROR_STRUCTURE_CORRUPT, CST_ERROR_CANCEL. 
     Other errors check in error_cb (see error codes of CST_import_cert(), CST_assign(), 
     CST_import_priv_key()).

   Callback function confirm_cb must return TRUE to import and FALSE to skip.
   Callback function confirm_cb use following parameters:
     is_pair - TRUE if need confirm import of private key and corresponding certificate;
     xcert - certificate to import;
     folder - use to set folder, 
        by default equal to CST_FOLDER_CA (is_pair == FALSE) or CST_FOLDER_PERSONAL (is_pair == TRUE),
     purpose - use to set purpose of certificate, 
        by default equal to CST_PURPOSE_NONE;
     out_password - use to set private key password, need only if is_pair equal to TRUE;
     cancel - set TRUE to stop import, by default FALSE;
     data - user data.
     @ref example3

   Callback function error_cb must return FALSE if need stop import and TRUE to skip and continue import.
   Callback function error_cb use following parameters:
     xcert - certificate on which error;
     error - error code;
     data - user data;
*/   
    extern void CST_import_PKCS12(CST * st, 
            FILE * file, 
            cst_pkcs12_confirm_cb confirm_cb,
            cst_pkcs12_error_cb error_cb,
            unsigned char *password,
            void * user_data,
            GError **error);   
    
/**
   @ingroup ImportExport
   Import certificate from the file (PEM format).
   
   @param st        Pointer to storage structure
   @param file      Input file
   @param password  Password for open file or NULL if password is empty
   @return Error code
  
   Possible error:  
   
   CST_ERROR_NOT_FOUND - certificate not found in file or format incorrect
   
   CST_ERROR_DBSTRUCTURE_CORRUPT - db-file structure corrupt
   
   CST_ERROR_IO - I/O error
   
   CST_ERROR_NOSPC - no space left on device
*/
    extern int CST_import_cert(CST * st, FILE * file, unsigned char *password);
    
/**
   @ingroup ImportExport
   Import certificate from the file (DER format).
   
   @param st        Pointer to storage structure
   @param file      Input file
   @return Error code
  
   Possible error: see. CST_import_cert() 
*/   
    extern int CST_import_cert_DER(CST * st, FILE * file);

/**
   @ingroup ImportExport 
   Import certificate from the file and set folder (PEM format).
   If folder incorrect cert placed in default folder (CST_FOLDER_UNKNOWN)
   and error returned.
   
   @param st        Pointer to storage structure
   @param file      Input file
   @param password  Password for open file or NULL if password is empty
   @param folder    Folder code
   @return Error code
  
   Possible error: see. CST_import_cert()
*/
    extern int CST_import_cert_f(CST * st, FILE * file, unsigned char *password, const cst_t_cert_folder folder);
    
/**
   @ingroup ImportExport 
   Import certificate from the file and set folder (DER format).
   If folder incorrect cert placed in default folder (CST_FOLDER_UNKNOWN)
   and error returned.
   
   @param st        Pointer to storage structure
   @param file      Name of input file
   @param folder    Folder code
   @return Error code
  
   Possible error: see. CST_import_cert()
*/
    extern int CST_import_cert_f_DER(CST * st, FILE * file, const cst_t_cert_folder folder);
   
 /**
   @ingroup ImportExport 
   Import certificate from the file and set folder (PEM format).
   If folder incorrect cert placed in default folder (CST_FOLDER_UNKNOWN)
   and error returned.
   
   @param st        Pointer to storage structure
   @param file      Name of input file
   @param folder    Folder code
   @param error     Used to return error code (if NULL then ignored)
   @return ID imported certificate or 0 if error
  
   Possible error: see. CST_import_cert()
*/   
    extern cst_t_seqnum CST_import_cert_adv(CST * st, FILE * file, 
                                     const cst_t_cert_folder folder, 
                                     GError **error);
    
 /**
   @ingroup ImportExport 
   Import all certificates from the file and set folder (DER format).
   If folder incorrect cert placed in default folder (CST_FOLDER_UNKNOWN)
   and error returned.
   
   @param st        Pointer to storage structure
   @param file      Name of input file
   @param folder    Folder code
   @param error     Used to return error code (if NULL then ignored)
   @return ID imported certificate or 0 if error   
  
   Possible error: see. CST_import_cert()
*/   
    extern cst_t_seqnum CST_import_cert_adv_DER(CST * st, FILE * file, 
                                         const cst_t_cert_folder folder, 
                                         GError **error);
    
/**
   @ingroup ImportExport
   Export all certificates to the file (PEM format).
   
   @param st        Pointer to storage structure
   @param file      Output file
   @param folder    Folder which need export
   @return Error code
   
   Possible error:  CST_ERROR_CREATE_FILE
*/
    extern int CST_export_all(CST * st, FILE * file,
                              const cst_t_cert_folder folder);
/**
   @ingroup ImportExport
   Export all certificates to the file (DER format).
   
   @param st        Pointer to storage structure
   @param file      Output file
   @param folder    Folder which need export
   @return Error code
   
   Possible error:  CST_ERROR_CREATE_FILE
*/
    extern int CST_export_all_DER(CST * st, FILE * file,
                              const cst_t_cert_folder folder);
    
/**
   @ingroup ImportExport
   Export selected certificate to the file (PEM format)
   
   @param st        Pointer to storage structure
   @param cert      Same cert (X509 structure)
   @param file      Output file
   @return Error code
   
   Possible error:  CST_ERROR_CREATE_FILE
*/
    extern int CST_export_cert(CST * st, X509 * cert, FILE * file);
    
/**
   @ingroup ImportExport
   Export selected certificate to the file (DER format)
   
   @param st        Pointer to storage structure
   @param cert      Same cert (X509 structure)
   @param file      Output file
   @return Error code
   
   Possible error:  CST_ERROR_CREATE_FILE
*/
    extern int CST_export_cert_DER(CST * st, X509 * cert, FILE * file);   
    
/**
   @ingroup ImportExport
   Export selected certificate (by certID) to the file (PEM format)
   
   @param st        Pointer to storage structure
   @param certID    Certificate ID
   @param file      Output file
   @return Error code
   
   Possible error:  CST_ERROR_CREATE_FILE
*/
    extern int CST_export_cert_by_id(CST * st, const cst_t_seqnum certID, FILE * file);

/**
   @ingroup ImportExport
   Export selected certificate (by certID) to the file (DER format)
   
   @param st        Pointer to storage structure
   @param certID    Certificate ID
   @param file      Output file
   @return Error code
   
   Possible error:  CST_ERROR_CREATE_FILE
*/
    extern int CST_export_cert_by_id_DER(CST * st, const cst_t_seqnum certID, FILE * file);
    
/**
   @ingroup Cert
   Delete certificate
   
   @param st        Pointer to storage structure
   @param certID    Certificate ID
   @return Error code
   
   Possible error:  CST_ERROR_CERT_NOTFOUND
*/
    extern int CST_delete_cert(CST * st, const cst_t_seqnum certID);

/**
   @ingroup Cert
   Append X509 certificate to storage
  
   @param st        Pointer to storage structure
   @param cert      Pointer to X509 structure 
   @return Error code
  
   Possible error:  CST_ERROR_CERT_EXIST, CST_ERROR_DBSTRUCTURE_CORRUPT,
   CST_ERROR_IO, CST_ERROR_NOSPC
*/
    extern int CST_append_X509(CST * st, X509 * cert);
   
/**
   @ingroup Cert
   Append STACK_OF(X509) to storage
  
   @param st        Pointer to storage structure
   @param list      Stack of X509 certificates 
   @return Pointer to GSList with error code for each element of stack.
        Use GPOINTER_TO_INT(i->data) to get result for each code. You need
        execute g_slist_free(result) to free resorces.
        @code
        STACK_OF(X509) * list = ...
        GSList * result = CST_append_sk_X509(st, list);
        GSList * i;
        for (i = result; i != NULL; i = i->next);
        {
            printf("Error code: %i\n", GPOINTER_TO_INT(i->data));
        }
        g_slist_free(list);
        @endcode
  
   Possible error:  CST_ERROR_CERT_EXIST
*/    
    extern GSList * CST_append_sk_X509(CST * st, CST_STACK_OF_X509 * list);

/**
   @ingroup CRL
   Import Certificate Revocation List from file (PEM).

   @param st        Pointer to storage structure
   @param file      Input file (PEM encoded)
   @return Error code
   
   Possible error: see. CST_import_cert()
*/
    extern int CST_import_CRL(CST * st, FILE * file);
    
/**
   @ingroup CRL
   Import Certificate Revocation List from file (DER).

   @param st        Pointer to storage structure
   @param file      Input file (DER encoded)
   @return Error code
   
   Possible error: see. CST_import_cert()
*/   
    extern int CST_import_CRL_DER(CST * st, FILE * file);

/**
    @ingroup CRL
    Return all CRLs from storage
    
    @param st       Pointer to storage structure
    @return List of CRL and NULL if not found
*/
    extern GSList * CST_get_all_crl(CST * st);

/**
    @ingroup CRL
    Remove CRL from storage

    @param st       Pointer to storage structure
    @param crlID    CRL ID
    @return Error code

    Possible error:  CST_ERROR_NOT_FOUND
*/
    extern int CST_delete_crl(CST * st, const cst_t_seqnum crlID);

/**
    @ingroup CRL
    Get CRL by crlID
    
    @param st       Pointer to storage structure
    @param crlID    CRL ID
    @return X509_CRL structure
*/
    extern X509_CRL * CST_get_CRL(CST * st, const cst_t_seqnum crlID);
    


/* Private key */

/**
   @ingroup ImportExport
   Import private key from PEM encoded file 
   If private key with same eaccount exist in storage then key appended.
   
   @param st        Pointer to storage structure
   @param file      Input PEM encoded file
   @param account   Distinguished name for identify imported key
   @param inpass    Password for imported keys
   @param outpass   Password for key in storage
   @return Error code
   
   Possible error: see. CST_import_cert()
*/
    extern int CST_import_priv_key(CST * st, X509_NAME * account,
                                   FILE * file, unsigned char *inpass,
                                   unsigned char *outpass);
/**
   @ingroup ImportExport
   Import private key from DER encoded file 
   If private key with same eaccount exist in storage then key appended.
   
   @param st        Pointer to storage structure
   @param file      Input DER encoded file
   @param account   Distinguished name for identify imported key
   @param outpass   Password for key in storage
   @return Error code
   
   Possible error: see. CST_import_cert()
*/    
    extern int CST_import_priv_key_DER(CST * st, X509_NAME * account,
                                   FILE * file, unsigned char *outpass);

/**
   @ingroup ImportExport
   Import private key from PEM encoded file 
   If private key with same eaccount exist in storage then key appended.
   
   @param st        Pointer to storage structure
   @param file      Input PEM encoded file
   @param account   Distinguished name for identify imported key
   @param inpass    Password for imported keys
   @param outpass   Password for key in storage
   @param error     Used to return error code (if NULL then ignored)
   @return Id of imported key or 0 on error
   
   Possible error: see. CST_import_cert()
*/
    extern cst_t_seqnum CST_import_priv_key_adv(CST * st, X509_NAME * account,
                                   FILE * file, unsigned char *inpass,
                                   unsigned char *outpass,
                                   GError **error);
/**
   @ingroup ImportExport
   Import private key from DER encoded file 
   If private key with same eaccount exist in storage then key appended.
   
   @param st        Pointer to storage structure
   @param file      Input DER encoded file
   @param account   Distinguished name for identify imported key 
   @param outpass   Password for key in storage
   @param error     Used to return error code (if NULL then ignored)
   @return Id of imported key or 0 on error
   
   Possible error: see. CST_import_cert()
*/    
    extern cst_t_seqnum CST_import_priv_key_adv_DER(CST * st, X509_NAME * account,
                                   FILE * file, unsigned char *outpass, 
                                   GError **error);   
/**
   @ingroup ImportExport
   Export private key with given account to file (PEM format)
   
   @param st         Pointer to storage structure
   @param key        Pointer to key
   @param file       Output file
   @param password   Password for exported keys
   @return Error code
   
   Possible error:  CST_ERROR_NOT_FOUND
*/
    extern int CST_export_priv_key(CST * st, EVP_PKEY * key, FILE * file,
                                   unsigned char *password);
    
/**
   @ingroup ImportExport
   Export private key with given account to file (DER format)
   
   @param st         Pointer to storage structure
   @param key        Pointer to key
   @param file       Output file
   @param password   Password for exported keys
   @return Error code
   
   Possible error:  CST_ERROR_NOT_FOUND
*/
    extern int CST_export_priv_key_DER(CST * st, EVP_PKEY * key, FILE * file,
                                   unsigned char *password);
    
/**
   @ingroup Key
   Append private key to storage
  
   @param st        Pointer to storage structure
   @param account   Distinguished name for identify key
   @param key       Pointer to private key 
   @param password  Password for key in storage
   @return Error code
   
   Possible error:  CST_ERROR_DBSTRUCTURE_CORRUPT,
   CST_ERROR_IO, CST_ERROR_NOSPC
*/
    extern int CST_append_priv_key(CST * st, X509_NAME * account,
                                   EVP_PKEY * key, unsigned char *password);


/* Public key */

/**
   @ingroup ImportExport
   Import public key from file 
   If public key with same account exist in storage then key appended.
  
   @param st        Pointer to storage structure
   @param file      Input file
   @param account   Distinguished name Key for identify imported key
   @return Error code
   
   Possible error: see. CST_import_cert()
*/
    extern int CST_import_pub_key(CST * st, X509_NAME * account,
                                  FILE * file);
/**
   @ingroup ImportExport
   Import public key from file 
   If public key with same account exist in storage then key appended.
  
   @param st        Pointer to storage structure
   @param file      Input file
   @param account   Distinguished name Key for identify imported key
   @return Error code
   
   Possible error: see. CST_import_cert()
*/ 
    extern int CST_import_pub_key_DER(CST * st, X509_NAME * account,
                                  FILE * file);

/**
   @ingroup ImportExport
   Import public key from file 
   If public key with same account exist in storage then key appended.
  
   @param st        Pointer to storage structure
   @param file      Input file
   @param account   Distinguished name Key for identify imported key
   @param error     Used to return error code (if NULL then ignored)
   @return Id of imported key or 0 on error
   
   Possible error: see. CST_import_cert()
*/
    extern cst_t_seqnum CST_import_pub_key_adv(CST * st, X509_NAME * account,
                                  FILE * file, GError **error);
/**
   @ingroup ImportExport
   Import public key from file 
   If public key with same account exist in storage then key appended.
  
   @param st        Pointer to storage structure
   @param file      Input file
   @param account   Distinguished name Key for identify imported key
   @param error     Used to return error code (if NULL then ignored)
   @return Id of imported key or 0 on error
   
   Possible error: see. CST_import_cert()
*/ 
    extern cst_t_seqnum CST_import_pub_key_adv_DER(CST * st, X509_NAME * account,
                                  FILE * file, GError **error);   
/**
   @ingroup ImportExport
   Export all public keys with given account to file (PEM format)
  
   @param st        Pointer to storage structure
   @param account   Key for identify key
   @param file      Output file
   @return Error code
  
   Possible error:  CST_ERROR_NOT_FOUND, CST_ERROR_KEY_NOTFOUND
*/
    extern int CST_export_all_pub_key(CST * st, X509_NAME * account,
                                      FILE * file);
    
/**
   @ingroup ImportExport
   Export all public keys with given account to file (DER format)
  
   @param st        Pointer to storage structure
   @param account   Key for identify key
   @param file      Output file
   @return Error code
  
   Possible error:  CST_ERROR_NOT_FOUND, CST_ERROR_KEY_NOTFOUND
*/
    extern int CST_export_all_pub_key_DER(CST * st, X509_NAME * account,
                                      FILE * file);

/**
   @ingroup ImportExport
   Export public key with given account to file (PEM format)
  
   @param st        Pointer to storage structure
   @param key       Pointer to key
   @param file      Output file
   @return Error code
   
   Possible error:  CST_ERROR_NOT_FOUND, CST_ERROR_KEY_NOTFOUND
*/
    extern int CST_export_pub_key(CST * st, EVP_PKEY * key, FILE * file);
    
/**
   @ingroup ImportExport
   Export public key with given account to file (DER format)
  
   @param st        Pointer to storage structure
   @param key       Pointer to key
   @param file      Output file
   @return Error code
   
   Possible error:  CST_ERROR_NOT_FOUND, CST_ERROR_KEY_NOTFOUND
*/
    extern int CST_export_pub_key_DER(CST * st, EVP_PKEY * key, FILE * file);
    
/**
   @ingroup Key
   Append public key to storage
  
   @param st        Pointer to storage structure
   @param account   Distinguished name for identify key
   @param key       Pointer to public key 
   @return Error code
   
   Possible error:  CST_ERROR_DBSTRUCTURE_CORRUPT,
   CST_ERROR_IO, CST_ERROR_NOSPC
*/
    extern int CST_append_pub_key(CST * st, X509_NAME * account,
                                  EVP_PKEY * key);

/**
    @ingroup Key
    Get account assigned with key

    @param st       Pointer to storage structure
    @param keyID    Key ID
    @return account assigned with key on append or import 
*/ 
    extern X509_NAME * CST_get_key_account(CST * st, cst_t_seqnum keyID);
    
/* Delete */

/**
   @ingroup Key
   Delete all public key for account
  
   @param st        Pointer to storage structure
   @param account  Key for identify key
   @return Error code
   
   Possible error:  CST_ERROR_KEY_NOTFOUND - if account not found
*/
    extern int CST_delete_all_pub_key(CST * st, X509_NAME * account);

/**
   @ingroup Key
   Delete all private key for account
  
   @param st        Pointer to storage structure
   @param account  Key for identify key
   @return Error code
   
   Possible error:  CST_ERROR_KEY_NOTFOUND - if account not found
*/
    extern int CST_delete_all_priv_key(CST * st, X509_NAME * account);

/**
   @ingroup Key
   Delete public key form storage
   
   @param st        Pointer to storage structure
   @param keyID     Public key ID
   @return Error code
 
   Possible error:  CST_ERROR_KEY_NOTFOUND 
*/
    extern int CST_delete_pub_key(CST * st, 
                                 const cst_t_seqnum keyID);

/**
   @ingroup Key
   Delete private key form storage
   
   @param st        Pointer to storage structure
   @param keyID     Private key ID
   @return Error code
 
   Possible error:  CST_ERROR_KEY_NOTFOUND 
*/
    extern int CST_delete_priv_key(CST * st, 
                                   const cst_t_seqnum keyID);

/* Search */

/**
   @ingroup CertSearch
   Search certifiacate by subject name
  
   @param st       Pointer to storage structure
   @param subject_name Subject Distinguished name
   @return List of certificates ID for given DN or NULL if not found
    
   Possible error:  CST_ERROR_CERT_NOTFOUND

   @ref example2
*/
    extern GSList * CST_search_by_subj_name(CST * st,
                                            X509_NAME *
                                            subject_name);

/**
   @ingroup CertSearch
   Search certifiacate by email
  
   @param st       Pointer to storage structure
   @param email    Email of subject
   @return Array of certificates for given DN and NULL if not found
    
   Possible error:  CST_ERROR_CERT_NOTFOUND

   @ref example2
*/
    extern GSList * CST_search_by_email(CST * st,
                                        const char *email);

/**
   @ingroup CertSearch
   Search certifiacate by domain name
  
   @param st          Pointer to storage structure
   @param domain_name Domain name of subject
   @return Array of certificates for given domain name and NULL if not found
    
   Possible error:  CST_ERROR_CERT_NOTFOUND

   @ref example2
*/

    extern GSList * CST_search_by_domain_name(CST * st,
                                              const char
                                              *domain_name);

 /**
   @ingroup CertSearch
   Search certifiacate by serial
  
   @param st       Pointer to storage structure
   @param serial   Serial number
   @return Array of certificates for given serial and NULL if not found
    
   Possible error:  CST_ERROR_CERT_NOTFOUND

   @ref example2
*/
    extern GSList * CST_search_by_serial(CST * st, const char *serial);   

/**
   @ingroup CertSearch
   Search certifiacate by email
  
   @param st       Pointer to storage structure
   @param fingerprint Fingerprint (see. CST_get_fingerprint())
   @return Array of certificates for given fingerprint and NULL if not found
    
   Possible error:  CST_ERROR_CERT_NOTFOUND

   @ref example2
*/
    extern GSList * CST_search_by_fingerprint(CST * st, const char *fingerprint);
    
/**
   @ingroup CertSearch
   Search issuer of given certifiacate 
  
   @param st          Pointer to storage structure
   @param cert        Certificate
   @return certID of issuer or NULL 
*/

    extern cst_t_seqnum CST_search_issuer(CST * st, X509 * cert);    
    
/**
   @ingroup Cert
   Get cert chain for given certificate

   @param st        Pointer to storage structure
   @param cert      Certificate for which need chan
   @return Stack of certificates or NULL if not found chain

   Possible error:  CST_ERROR_CERT_NOTFOUND - if cert chain incomplite
*/
    extern CST_STACK_OF_X509 * CST_get_chain(CST * st, X509 * cert);

/**
    @ingroup Cert
    Get cert chain for given certificate ID

    @param st       Pointer to storage structure
    @param certID   Certificate ID
    @return GSList * - is list of certificate ID

    @ref example2
*/    
    extern GSList * CST_get_chain_id_by_id(CST * st, const cst_t_seqnum certID);

/**
    @ingroup Cert
    Get cert chain for given certificate ID

    @param st       Pointer to storage structure
    @param x        X509 Certificate
    @return GSList * - is list of certificate ID   

    @ref example2
*/    
    extern GSList * CST_get_chain_id(CST * st, X509 * x);
    
/**
   @ingroup CertSearch
   Get all trusted certificates for given purpose in selected folder.
   For example, all CA certificates for WLAN.

   @param st        Pointer to storage structure
   @param folder    Folder
   @param purpose   Certificate purpose
   
   @return List of certificates or NULL if not found

   @ref example2
*/
    extern GSList * CST_search_by_folder_and_purpose(CST * st,
                                          const cst_t_cert_folder folder,
                                          const cst_t_cert_purpose purpose);
/**
   @ingroup CertSearch
   Get all certificates by folder
   If folder = CST_FOLDER_CA then return all CA certificates

   @param st        Pointer to storage structure
   @param folder    Folder
   
   @return List of certificates or NULL if not found

   @ref example2
*/
    extern GSList * CST_search_by_folder(CST * st,
                                         const cst_t_cert_folder
                                         folder);

/** 
   @ingroup Key
   Get All private keys for given account
   
   @param st        Pointer to storage structure
   @param account   Account 
   @return Array of keys or NULL if not found chain

   Possible error:  CST_ERROR_KEY_NOTFOUND - if cert chain incomplite

   @ref example2
*/
    extern GSList * CST_priv_key_search_by_name(CST * st,
                               X509_NAME * account);
    
/** 
   @ingroup Key
   Get All public keys for given account
   
   @param st        Pointer to storage structure
   @param account   Account
   @return Array of keys or NULL if not found chain

   Possible error:  CST_ERROR_KEY_NOTFOUND - if cert chain incomplite

   @ref example2
*/
    extern GSList * CST_pub_key_search_by_name(CST * st,
                                               X509_NAME * account);
   
/**
   @ingroup CertSearch
   Get all expired certificates 
   @todo Need implement

   @param st        Pointer to storage structure
   @return Array of certificates or NULL if not found chain

   Possible error:  CST_ERROR_CERT_NOTFOUND - if cert chain incomplite

   @ref example2
*/
    extern GSList * CST_all_expired(CST * st);

/**
   @ingroup CertSearch
   Get all revoked certificates
   @todo Need implement

   @param st        Pointer to storage structure
   @return Stack of certificates or NULL if not found chain

   @ref example2
*/
    extern GSList * CST_all_revoked(CST * st);

/**
   @ingroup CertSearch
   Get all trusted certificates for given purpose 

   @param st        Pointer to storage structure
   @param purpose   Certificate purpose
   
   @return List of certificates or NULL if not found 

   @ref example2
*/
    extern GSList * CST_search_by_purpose(CST * st,
                                          const cst_t_cert_purpose
                                          purpose);


/* Cert */


/**
   @ingroup Cert
   Get issuer distinguished name (issued by)
*/
    extern X509_NAME *CST_get_issued_by_dn(X509 * cert);

/** 
   @ingroup Cert
   Get subject distinguished name (issued to)
*/
    extern X509_NAME *CST_get_subject_dn(X509 * cert);

/**
   @ingroup Cert
   Get valid to
*/
    extern time_t CST_get_valid_from(X509 * cert);

/**
   @ingroup Cert
   Get valid from
*/
    extern time_t CST_get_valid_to(X509 * cert);

/**
    @ingroup Cert
    If certificate expired or not valid yet return TRUE
*/
    extern int CST_is_expired(X509 * cert);
    
/**
   @ingroup Cert
   Get serial number
*/
    extern ASN1_INTEGER *CST_get_serial_number(X509 * cert);
    
/**
   @ingroup Cert
   Get serial number in string
*/    
    extern char *CST_get_serial_number_t(X509 * cert);

/**
   @ingroup Cert
   Get fingerprint
*/
    extern char *CST_get_fingerprint(X509 * cert);
    
/**
   @ingroup Cert
   Get fingerprint MD5
*/
    extern char *CST_get_fingerprint_MD5(X509 * cert);
    
/**
   @ingroup Cert
   Get fingerprint SHA1
*/
    extern char *CST_get_fingerprint_SHA1(X509 * cert);

/**
    @ingroup Cert
    Get email if exist or NULL 
*/
    extern char *CST_get_email(X509 * cert);
    
/**
    @ingroup Cert
    Get domain name if exist or NULL 
*/
    extern char *CST_get_domain_name(X509 * cert);   

/**
    @ingroup Cert
    Get public key algorithm
*/    
    extern char *CST_get_public_key_alg(X509 * cert);
    
/**
    @ingroup Cert
    @brief Check purpose of X509 

    @param x X509 certificate 
    @param purposes Purpose

    @return TRUE if all purposes is ok for certificate
*/    
    extern int CST_check_purpose_x(X509 * x, const cst_t_cert_purpose purposes);
    
/**
    @ingroup Cert
    @brief Check purpose of certificate in storage by his ID

    @param st       Pointer to certificate storage  
    @param certID   Certificate ID
    @param purpose  Purpose

    @return TRUE can set given purpose to certificate
*/    
    extern int CST_check_purpose(CST * st, const cst_t_seqnum certID, 
                                 const cst_t_cert_purpose purpose);
    
/**
    @ingroup Cert
    Check that certificate is root

    @param  cert    X509 certificate
    @return TRUE if is root
*/    
    extern int CST_is_root(X509 * cert);
  
/**
    @ingroup Cert
    Check that certificate (certID) is root

    @param  st      Pointer to storage structure
    @param  certID  Certificate ID
    @return TRUE if certificate is root, FALSE if not root or not found
*/    
    extern int CST_is_root_id(CST * st, const cst_t_seqnum certID);

/**
    @ingroup Cert
    Check that certificate can be CA
    
    @param cert X509 certificate
    @return TRUE if "Basic Constraint" not present or in "Basic Constraint" CA = TRUE
*/    
    extern int CST_is_CA(X509 * cert);    
    
/**
    @ingroup Key
    Convert key to text format
*/    
    extern char *CST_EVP_PKEY_to_text(EVP_PKEY * key);    
    
/**
   @ingroup Cert
   Get revoked state 
*/
    extern int CST_is_revoked(CST * st, X509 * cert);

/**
   @ingroup Cert
   Get stored on network state
   @todo Need implement
*/
    extern int CST_is_network(CST * st, X509 * cert);

/**
   @ingroup Cert
   Get stored on network URL
   @todo Need implement
*/
    extern char *CST_get_network_URL(CST * st, X509 * cert);

    
/**
   @ingroup CertKeyDefault
   Set default certificate by unique ID (UID) (issuer + serial)
   for email of certificate.
   
   @param st        Pointer to storage structure
   @param certID    Certificate ID
   @return UID of default certificate or NULL
*/
    extern int CST_set_default(CST * st, const cst_t_seqnum certID); 
    
/**
   @ingroup CertKeyDefault
   Get default certificate by email

   @param st        Pointer to storage structure
   @param email     Email
   @return Defailt certificate or NULL
*/
    extern X509 *CST_default_cert(CST * st, const char *email);

/**
   @ingroup CertKeyDefault
   Get default certificate ID by email

   @param st        Pointer to storage structure
   @param email     Email
   @return certID for default certificate
*/   
    extern cst_t_seqnum CST_default_cert_id(CST * st, const char *email);
    
/**
   @ingroup CertKeyDefault
   Default or not certificate by UID

   @param st        Pointer to storage structure
   @param certID    Certificate id
   @return TRUE if default 
*/
    extern int CST_is_default(CST * st, const cst_t_seqnum certID);

/**
   @ingroup CertSearch
   Get certificate ID (int) by UID (issuer + serial)

   @param st        Pointer to storage structure
   @param issuer    Distinguished name of issuer
   @param serial    Serial number
   @return Certificate or 0 if not found
*/
    extern cst_t_seqnum CST_search_by_UID(CST * st, X509_NAME * issuer,
                                   ASN1_INTEGER * serial);

/**
    @ingroup CertSearch
    Get certificate ID (int) by openssl X509 certificate

    @param st       Pointer to storage structure
    @param xcert    X509 certificate

    @return Certificate ID or 0 in not found
*/    
    extern cst_t_seqnum CST_search_by_X509(CST * st, X509 * xcert);
    
/**
    @ingroup CertKeyDefault
    Assign private key with certificate

    @param st       Storage structure
    @param certID   Certificate
    @param keyID    Private key
    @param password Password, need for test correct assign
    @return Error code

    Possible error:  CST_ERROR_ASSIGN_INCORRECT
*/
    extern int CST_assign(CST * st, 
                          const cst_t_seqnum certID, 
                          const cst_t_seqnum keyID, 
                          unsigned char *password);

/**
   @ingroup CertKeyDefault
   Get private key by certificate UID

   @param st        Pointer to storage structure
   @param issuer    Distinguished name of issuer
   @param serial    Serial number
   @param password  Password 
   @return Private key or NULL if not found
*/
    extern EVP_PKEY *CST_get_priv_key_by_UID(CST * st,
                                             X509_NAME * issuer,
                                             ASN1_INTEGER * serial,
                                             unsigned char *password);

/**
   @ingroup CertKeyDefault
   Get private key by certificate 

   @param st        Pointer to storage structure
   @param cert      Pointer to cert
   @param password  Password
   @return Private key or NULL if not found
*/
    extern EVP_PKEY *CST_get_priv_key(CST * st, X509 * cert, 
                                      unsigned char *password);

/**
   @ingroup CertKeyDefault
   Get private key by his ID 

   @param st        Pointer to storage structure
   @param certID    certID
   @return keyID or 0 if not found
*/
    extern cst_t_seqnum CST_get_assigned_key(CST * st, const cst_t_seqnum certID);   
    
/**
   @ingroup CertKeyDefault
   Get private key for default certificate

   @param st        Pointer to storage structure
   @param email     Email
   @param password  Password
   @return Private key or NULL
*/
    extern EVP_PKEY * CST_get_priv_key_default(CST * st,
                                    char *email,
                                    unsigned char *password);
    
/**
    @ingroup Cert
    Set certificate folder
*/
    extern int CST_set_folder(CST * st, const cst_t_seqnum certID, 
                              const cst_t_cert_folder f);

/**
    @ingroup Cert
    Get certificate folder
*/
    extern cst_t_cert_folder CST_get_folder(CST * st, 
                                            const cst_t_seqnum certID);

/**
    @ingroup Cert
    Set purose (trust settings)
*/
    extern int CST_set_purpose(CST * st, 
                               const cst_t_seqnum certID,
                               const cst_t_cert_purpose p,
                               const int value);

/**
    @ingroup Cert
    Check purpose (trust)
*/
    extern int CST_is_purpose(CST * st, 
                              const cst_t_seqnum certID,
                              const cst_t_cert_purpose p);

/**
    @ingroup Cert
    Check certificate validity
*/
    extern int CST_is_valid(CST * st, X509 * cert);

/**
    @ingroup Cert
    Check certificate validity. Read certificate from file (PEM format).
*/
    extern int CST_is_valid_f(CST * st, FILE * file, GError **error);
    
/**
    @ingroup Cert
    Check certificate validity. Read certificate from file (DER format).
*/    
    extern int CST_is_valid_f_DER(CST * st, FILE * file, GError **error);
    
/**
    @ingroup Cert
    Check certificate validity
*/   
    extern int CST_is_valid_for(CST * st, 
            X509 * cert, const cst_t_cert_purpose purpose);
    
/* S/MIME Capabilities */
    
/**
    @ingroup CertCapab
    Set S/MIME Capability for given cert
*/
    extern int CST_set_capability(CST * st, 
            const cst_t_seqnum certID,
            ASN1_OBJECT * oid, 
            unsigned char *data, int data_length);


/**
    @ingroup CertCapab
    Get S/MIME Capability for given cert
*/
    extern unsigned char * CST_get_capability_data(CST * st, 
            const cst_t_seqnum certID, 
            ASN1_OBJECT * oid, int *data_length);

/**
    @ingroup CertCapab
    Get S/MIME Capabilities list for given cert
*/
    extern CST_STACK_OF_ASN1_OBJECT * CST_get_capabilities(CST * st, 
            const cst_t_seqnum certID);

/**
    @ingroup CertCapab
    Return TRUE if capability exist
*/
    extern int CST_is_capability(CST * st,
            const cst_t_seqnum certID, ASN1_OBJECT * oid);

/**
    @ingroup CertCapab
    Delete capabilities
*/
    extern int CST_delete_capability(CST * st, 
            const cst_t_seqnum certID, 
            ASN1_OBJECT * oid);

/**
    @ingroup Cert
    @brief Get state of certificate 
    
    see: CST_STATE_REVOKED, CST_STATE_EXPIRED, ...
*/
    extern int CST_get_state(CST * st, X509 * cert);

/**
    @ingroup Cert
    Get X509 by certID
*/
    extern X509 * CST_get_cert(CST * st, const cst_t_seqnum certID);

/**
    @ingroup Key
    Get private EVP_PKEY by keyID
*/
    extern EVP_PKEY * CST_get_key(CST * st, const cst_t_seqnum keyID, 
            unsigned char *password);

/**
    @ingroup Key
    Get public key EVP_PKEY by keyID
*/
    extern EVP_PKEY * CST_get_pub_key(CST * st, const cst_t_seqnum keyID);
    
/**
    @page example1 Example 1. Open and close.
    @code
    CST * storage = CST_open_file("storage.cst", FALSE, NULL);

    X509 * cert = ... // Some cert received from server

    if (CST_is_valid(storage, cert))
    {
        printf("Certificate is valid");
    }
    else
    {
        printf("Certificate expired, have not valid sign or signed by untrusted...");
    }

    CST_free(storage);
    @endcode
*/ 

/**
    @page example2 Example 2. Using search functions, which return GSList.
    @code
    GSList * list = CST_search_by_folder(st, CST_FOLDER_CA);
    GSList * i;
    for (i = list; I != NULL; i = i->next)
    {
      printf(certID = %u\n, GPOINTER_TO_UINT(i->data));
    }
    g_slist_free(list);
    @endcode
*/ 

 /**
    @page example3 Example 3. Using CST_import_PKCS12().
    @code
    // implement callback function
    int confirm_cb(X509 * xcert, 
        cst_t_cert_folder * folder, cst_t_cert_purpose * purpose,
        unsigned char * out_pass, int is_pair, int *cancel, 
        void * user_data)
    {
        if (is_pair)
        {
            *out_pass = "secret";
            *purpose = CST_PURPOSE_SSL_CLIENT;
            *folder = CST_FOLDER_PERSONAL;
            *cancel = FALSE;
            return TRUE;
        }
        else
        {
            *purpose = CST_PURPOSE_CA;
            *folder = CST_FOLDER_CA;
            *cancel = FALSE;
            return TRUE;
        }
    }
    @endcode
*/   
#ifdef __cplusplus
}
#endif
#endif                          /* CST_H_ */
