/* 
 * CyberLinkC AV API for Content Directory Service
 *
 * ccontentdirectory.h
 *
 * Copyright 2005 Nokia Corporation. All rights reserved.
 *
 * This is licensed under BSD-style license with patent exclusion,
 * see file COPYING.
 */

#ifndef CG_UPNP_AV_CCONTENTDIRECTORY_H
#define CG_UPNP_AV_CCONTENTDIRECTORY_H

#include <cybergarage/upnp/cdevice.h>
#include <cybergarage/upnp/cservice.h>
#include <cybergarage/upnp/caction.h>
#include <cybergarage/xml/cxml.h>

/*********************************************
 * Service name
 *********************************************/
#define CG_UPNP_CDS_SERVICE_TYPE "urn:schemas-upnp-org:service:ContentDirectory"

/*********************************************
 * Mandatory actions
 *********************************************/
#define CG_UPNP_CDS_ACTION_GETSEARCHCAPABILITIES "GetSearchCapabilities"
#define CG_UPNP_CDS_ACTION_GETSORTCAPABILITIES "GetSortCapabilities"
#define CG_UPNP_CDS_ACTION_GETSYSTEMUPDATEID "GetSystemUpdateID"
#define CG_UPNP_CDS_ACTION_BROWSE "Browse"

/*********************************************
 * Optional actions
 *********************************************/
#define CG_UPNP_CDS_ACTION_SEARCH "Search"
#define CG_UPNP_CDS_ACTION_CREATEOBJECT "CreateObject"
#define CG_UPNP_CDS_ACTION_DESTROYOBJECT "DestroyObject"
#define CG_UPNP_CDS_ACTION_UPDATEOBJECT "UpdateObject"
#define CG_UPNP_CDS_ACTION_IMPORTRESOURCE "ImportResource"
#define CG_UPNP_CDS_ACTION_EXPORTRESOURCE "ExportResource"
#define CG_UPNP_CDS_ACTION_STOPTRANSFERRESOURCE "StopTransferResource"
#define CG_UPNP_CDS_ACTION_GETTRANSFERPROGRESS "GetTransferResource"
#define CG_UPNP_CDS_ACTION_DELETERESOURCE "DeleteResource"
#define CG_UPNP_CDS_ACTION_CREATEREFERENCE "CreateReference"

/*********************************************
 * Content Directory Service container
 *********************************************/

/**
 * @brief The Content Directory Service structure.
 *
 * Actually, this is not THE CDS service from CyberLink's point of view, but a
 * container for CDS-specific information. This can be stored in the userData
 * pointer of the actual CgUpnpService struct. The actual service can be found
 * by checking whether cg_upnp_service_isname() returns the CDS type string
 * stored in @ref CG_UPNP_CDS_SERVICE_TYPE.
 *
 * Short howto for device side: First, create a regular device and give the
 * device its XML description to parse. The device should now contain services
 * (with CDS). Second, give each service (especially CDS in this case) their
 * own XML description files to parse actions and state tables. Third, set the
 * service's action listener (to be able to respond correctly to actions and
 * pass action invocations to other listening parties like libgupnp)
 * and optionally the query listener (for libgupnp and friends, not really much
 * else, because CyberLink does it all for us).
 */
typedef struct _CgUpnpAVCDS
{
	/** The actual CyberLink service struct i.e. the parent */
	CgUpnpService* upnpService;

	/** Sort capabilities */
	CgString* sortCaps;

	/** Search capabilities */
	CgString* searchCaps;

	/** Update ID */
	CgString* systemUpdateID;

	/** CDS contents */
	CgXmlNodeList* contentNodeList;

	/** Thread that periodically checks for updates in added directories */
	CgThread* fileWatcher;

} CgUpnpAVCDS;

/*****************************************************************************
 * Content Directory Service
 *****************************************************************************/

/**
 * Create a new Content Directory Service sub-structure.
 * You must first have a parsed CgUpnpService* and then set a
 * CgUpnpAVCDS* as the service's user data.
 *
 */
CgUpnpAVCDS* cg_upnp_av_cds_new();

/**
 * Destroy a Content Directory container struct.
 * Does not destroy the CgUpnpService* that it might be a part of.
 *
 * @param cds The CDS container to destroy
 */
void cg_upnp_av_cds_delete(CgUpnpAVCDS* cds);

/**
 * Add a directory to the CDS (device side only). This function is recursive,
 * meaning it adds everything found under the given directory.
 *
 * @param cds The CDS container in question
 * @param dir Directory to add
 */
void cg_upnp_av_cds_add_directory(CgUpnpAVCDS* cds, CgString* dir);

/**
 * Remove a directory from the CDS (device side only). This function
 * is recursive, meaning it removes the whole tree under the given node.
 *
 * @param cds The CDS container in question
 * @param node The node to remove
 */
void cg_upnp_av_cds_remove_directory(CgUpnpAVCDS* cds, CgXmlNode* node);

/*****************************************************************************
 * Callback functions for CyberLinkC service
 *****************************************************************************/

/**
 * The function that gets called by the CyberLinkC stack when an action
 * invoke has been sent to the CDS service
 *
 * @param action The received action
 * @return TRUE if action was processed successfully; otherwise false
 */
BOOL cg_upnp_av_cds_action_received(CgUpnpAction *action);

/*****************************************************************************
 * Directory Watcher functions
 *****************************************************************************/

/**
 * The function that keeps track of the contents in the directories
 * that have been set available in the content directory service.
 *
 * @note Users shouldn't call this function directly.
 *
 * @param thread The CgThread struct that owns this function
 */
void cg_upnp_av_cds_directory_watcher(CgThread* thread);

/*****************************************************************************
 * CyberLinkC service
 *****************************************************************************/

/**
 * Get the CyberLinkC service from a Content Directory struct.
 *
 * @param cds Content Directory in question
 * @return The CgUpnpService* type service
 */
#define cg_upnp_av_cds_get_service(cds) cds->service

/*****************************************************************************
 * Sort Capabilities
 *****************************************************************************/

/**
 * Set the CDS' sort capabilities
 *
 * @param cds The Content Directory Service
 * @param sort_caps A string containing sort capabilities
 */
#define cg_upnp_av_cds_setsortcaps(cds, sort_caps) \
        cg_string_setvalue(cds->sortCaps, sort_caps)

/**
 * Get the CDS' sort capabilities
 *
 * @param cds The Content Directory Service
 * @return A string containing sort capabilities
 */
#define cg_upnp_av_cds_getsortcaps(cds) \
        cg_string_getvalue(cds->sortCaps)

/*****************************************************************************
 * Search Capabilities
 *****************************************************************************/

/**
 * Set the CDS' search capabilities
 *
 * @param cds The Content Directory Service
 * @param search_caps A string containing seach capabilities
 */
#define cg_upnp_av_cds_setsearchcaps(cds, search_caps) \
        cg_string_setvalue(avDevice->sortCaps, search_caps)

/**
 * Get the CDS' search capabilities
 *
 * @param cds The Content Directory Service
 * @return A string containing search capabilities
 */
#define cg_upnp_av_cds_getsearchcaps(cds) \
        cg_string_getvalue(cds->searchCaps)

/*****************************************************************************
 * System Update ID
 *****************************************************************************/

/**
 * Set the CDS' system update ID
 *
 * @param cds The Content Directory Service
 * @param id System update ID
 */
#define cg_upnp_av_cds_setsystemupdateid(cds, id) \
        cg_string_setvalue(cds->systemUpdateID, id)

/**
 * Get the CDS' system update ID
 *
 * @param cds The Content Directory Service
 * @return System update ID
 */
#define cg_upnp_av_cds_getsystemupdateid(cds) \
        cg_string_getvalue(cds->systemUpdateID)

#endif /* CG_UPNP_AV_CCONTENTDIRECTORY_H */
