/*
  osso-ic-oss Internet Connectivity library
  Copyright (C) 2005 Nokia Corporation. All rights reserved.

  This library is free software; you can redistribute it and/or modify it
  under the terms of the GNU Lesser General Public License version 2.1 as
  published by the Free Software Foundation.

  This library is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
  General Public License for more details.

  You should have received a copy of the GNU Lesser General Public License
  along with this library; if not, write to the Free Software Foundation,
  Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
  */

/**
 * @file osso-ic.h
 * @brief Nokia OSSO Internet Connectivity (IC) API definitions.
 * @author Patrik Flykt <patrik.flykt@nokia.com>
 * @author Kalle Valo <kalle.valo@nokia.com>
 *
 *  Interface to IAP related tasks such as establishing an internet
 * connection, tearing it down and getting statistics.
 */

#ifndef OSSO_IC_H
#define OSSO_IC_H

#include <dbus/dbus.h>
#include <libosso.h>

/**
 * @mainpage Internet Connectivity API
 *
 * f. Internet Connectivity API (IC API) is used for starting and stopping
 * Internet connections on Nokia 770. Using the IC API applications can
 * request Internet connections on-demand and notify when the connection
 * isn't needed anymore.
 *
 * @section OVERVIEW Overview
 *
 * IC API uses D-BUS to communicate with Internet Connection daemon (ICd)
 * which takes care of all Internet connections. ICd establishes and tears
 * down Internet connections based on requests from applications. Currently
 * supported connection types are WLAN and Bluetooth Dial Up Networking
 * (DUN). 
 *
 * ICd keeps record of applications using network connections. When the
 * first application requests for a network connection, ICd opens an
 * connection depending on the available networks, preferences and user
 * input. If later other applications request for network access, ICd will
 * try to use the already opened connection. After the last application has
 * requested for disconnection, ICd will tear down the connection.
 *
 * This means that the while applications share the same connection but
 * they do not need to worry about other applications using it at all.
 * Applications are free to request the disconnection at any point of time,
 * despite the fact that another application would be using the connection.
 *
 * Different network connection settings are stored as Internet Access
 * Points (IAP). Each IAP has a unique name and contains all the necessary
 * settings (type, WLAN SSID, proxies, usernames etc.). User can create,
 * edit and delete IAPs using the Connectivity Settings applet from the
 * Control Panel. IC API provides the possibility to request use of
 * specific IAP, but this is rarely needed. Normally user selects IAP from
 * a list during the connection establishment or the preferred IAP can be
 * configured from the Connectivity Settings.
 * 
 * @section USAGE How to use IC API
 *
 * There are two different interfaces, @ref OSSO_IAP_EXTENDED and @ref
 * OSSO_IAP_SOCKET. Either one of the interfaces must be used, <em>they can
 * not be used at the same time</em>. It is recommended to use the
 * extended interface.
 *
 * Two debian packages are needed for developing using IC API, osso-ic-lib
 * and osso-ic-dev. The osso-ic-dev package supports pkg-config for
 * managing compile and link flags. Here's an example how you can use it
 * with autoconf:
 *
 * @verbatim
PKG_CHECK_MODULES(OSSOIC, osso-ic)
AC_SUBST(OSSOIC_CFLAGS)
AC_SUBST(OSSOIC_LIBS)
@endverbatim
 *
 * If autoconf isn't used, pkg-config can be used manually, for example
 * from a Makefile:
 *
 * @verbatim
pkg-config --cflags --libs osso-ic
@endverbatim
 *
 */

#ifdef __cplusplus
extern "C" {
#endif

  /**@defgroup OSSO_IAP_EXTENDED Extended interface
   *
   * Internet Connectivity extended interface has more control for creating
   * and destroying network connections. It is the recommended way to use
   * the IC API. For a simpler, but features lacking interface, see @ref
   * OSSO_IAP_SOCKET.
   *
   * @remarks <em>@ref OSSO_IAP_EXTENDED and @ref OSSO_IAP_SOCKET should
   * not be used together in the same application!</em> They are two
   * different methods to use the IC API, and they are mutually exclusive.
   *
   * The IC Extended interface is event based, with few simple
   * functions for establishing connections and querying statistics. Events
   * are delivered by using a callback function. The basic sequence of a
   * program using IC Extended interface is this:
   *
   * <ol>
   *   <li> registers the callback function with osso_iap_cb()
   *   <li> requests for network establishment using
   *        osso_iap_connect()
   *   <li> receives OSSO_IAP_CONNECTED event through the callback function 
   *        and saves name of the IAP
   *   <li> uses network normally, but with non-blocking sockets
   *   <li> calls osso_iap_disconnect() just before it quits
   *   <li> after receiving OSSO_IAP_DISCONNECTED event for the IAP used 
   *        (through the callback function) it quits
   * </ol>
   *
   * Using the IC Extended interface has few requirements:
   *
   * - The use of blocking network functions isn't possible because IC events
   *   can't then be processed, use non-blocking methods instead. 
   *   Otherwise normal
   *   Linux network functions can be used (getaddrinfo(), socket(), close(),
   *   select() etc.).
   * - osso_socket() and osso_close() must not be used at the same time
   *   with  the extended interface. These functions are explained more in @ref
   *   OSSO_IAP_SOCKET
   * - IC API uses D-BUS, so some kind main loop (eg. GLib's Main Event Loop)
   *   is needed processing D-BUS messages.
   *
   * For a complete example, see <tt>
   * /usr/share/doc/osso-ic-dev/example/iap-connect.c.gz</tt> from osso-ic-dev
   * package.
   */
  /*@{*/

  /* IAP connect reasons */
  /**
   * IAP connect reason which states that connection attempt has been
   * initiated by user. It shall be used only if the connection is a
   * user-triggered event, i.e. link is clicked, send/receive button in email
   * is clicked, etc.
   */
#define OSSO_IAP_REQUESTED_CONNECT      0x00

  /**
   * The IAP connection request which is a result of timed event.
   * Use this if the application itself (automatically) starts the
   * connection, e.g. 
   * an email program checks the mailbox in the background, a browser tries to refresh a
   * page, etc. 
   * 
   * The Internet connection will be established if:
   * - OSSO_IAP_ANY is requested and a connection is on
   * - a specific connection is requested and that specific connection is 
   *   already on
   * - for all other cases the connection attempt is denied.
   */
#define OSSO_IAP_TIMED_CONNECT          0x01

  /* IAP event types */

  /**
   * IAP connection has been established event. The
   * socket can be now created and device has access to network. When this
   * event is received after calling osso_iap_connect(), the name of IAP
   * should be saved.
   */
#define OSSO_IAP_CONNECTED              0x01

  /**
   * IAP connection has been torn down for some reason, when this event is
   * received. All network connections (ie. sockets) should be closed and
   * network should not be used. Use IAP name from the OSSO_IAP_CONNECTED
   * to check that the event is about correct IAP.
   */
#define OSSO_IAP_DISCONNECTED           0x02

  /**
   * There was an error during the connection establishment, when this
   * event is received.
   *
   * The program should treat this similar as OSSO_IAP_DISCONNECTED event,
   * ie. close sockets and not use network anymore.
   */
#define OSSO_IAP_ERROR                  0x03

  /**
   * Statistics event type.
   *
   * The argument received with the event contains struct iap_statistics_t
   * filled with statistics data.
   */
#define OSSO_IAP_STATISTICS             0x04

  /*IAP meta names */

  /**
   * IAP meta name which forces to ask the user which IAP should be
   * used for connection.
   *
   * Even if an IAP connection is already established, this will force the
   * connection to be changed.
   *
   * @remarks <em> Use of this meta name is highly discouraged. </em> It
   * should be used only in very special cases and after thorough
   * consideration.
   */
#define OSSO_IAP_ASK    		"[ASK]"

  /**
   * IAP meta name which is normally used.
   *
   * If there isn't any connection opened, either asks the IAP to be used
   * from user or uses the preferred IAP if one is set. If there is an
   * connection already opened, it will be used.
   */
#define OSSO_IAP_ANY    		"[ANY]"


  /* Error codes */
  /**
   * The request IAP was invalid, for example it didn't exist.
   */
#define OSSO_ERROR_INVALID_IAP          -0x1000

  /**
   * Not used yet.
   */
#define OSSO_ERROR_IAP_FAILED           -0x1001

  /**
   * Not used yet.
   */
#define OSSO_ERROR_IAP_NOT_AVAILABLE    -0x1002

  /**
   * Not used yet.
   */
#define OSSO_ERROR_BIND_FAILED          -0x1003

  /**
   * Structure for containing statistic counters for an IAP.
   */
  struct iap_statistics_t
  {
    /**
     * The time, in seconds, the active IAP has been connected.
     */
    dbus_uint32_t time_active;

    /**
     * Signal strength of the WLAN connection.
     */
    dbus_uint32_t signal_strength;

    /**
     * The number of received packets during the current IAP connection.
     */
    dbus_uint32_t rx_packets;

    /**
     * The number of sent packets during the current IAP connection.
     */
    dbus_uint32_t tx_packets;

    /**
     * The amount of data received during the current IAP connection.
     */
    dbus_uint32_t rx_bytes;

    /**
     * The amount of data transferend during the current IAP connection.
     */
    dbus_uint32_t tx_bytes;
  };

  /**
   * Structure for an IAP related event.
   */
  struct iap_event_t {
    /** Event type */
    gint type;
    /** Name of the IAP related to event */
    const char *iap_name;


    /**
     * Contents of this union depends on type of event received.
     */

    union {

      /**
       * Error code received with OSSO_IAP_ERROR events. Is one of
       * OSSO_ERROR_INVALID_IAP, OSSO_ERROR_IAP_FAILED etc.
       */
      gint error_code;

      /**
       * Statistics data received with OSSO_IAP_STATISTICS events.
       */
      struct iap_statistics_t statistics;
    } u;
  };

  /**
   * Callback function definition for IAP related events.
   *
   * @param event Event which caused the callback to be called.
   * @param arg Parameter passed for callback function (aka. user_data).
   */
  typedef void (*osso_iap_cb_t)(struct iap_event_t *event, void *arg);

  /** 
   * Register a callback function for IAP related events.
   *
   * @param callback Pointer to callback function.
   * @retval OSSO_OK for success
   * @retval OSSO_ERROR for an error
   */
  gint osso_iap_cb(osso_iap_cb_t callback);

  /**
   * Establish an Internet connection. Depending on the Internet settings and
   * the parameters of this function, user might be asked to choose an IAP from
   * a list, use a preferred IAP specified in the settings, use already
   * active IAP or forcefully change the active IAP to something else.
   *
   * Time taken to establish the IAP connection is unspecified. It might take
   * just few seconds or maybe even a minute, all depending on the network
   * and the user. @ref OSSO_IAP_CONNECTED event is sent when the connection is
   * established and can be used. If something goes wrong during connection
   * establishment or user cancels it, OSSO_IAP_ERROR event is sent.
   *
   * @param iap The name of the IAP. Also OSSO_IAP_ANY and
   *	OSSO_IAP_ASK IAP meta names can be used. If uncertain, use
   *    OSSO_IAP_ANY. 
   * @param flags Flags used with the connection, either 
   *    OSSO_IAP_REQUESTED_CONNECT or OSSO_IAP_TIMED_CONNECT.
   * @param arg Parameter passed for callback function (aka. user_data).
   * @retval OSSO_OK for success
   * @retval OSSO_ERROR for an error
   * @note Succesful invocation of this function will always generate
   *	an event. It can be OSSO_IAP_CONNECTED or OSSO_IAP_ERROR.
   * @sa OSSO_IAP_ANY, OSSO_IAP_ASK, OSSO_IAP_REQUESTED_CONNECT,
   *         OSSO_IAP_TIMED_CONNECT
   */
  gint osso_iap_connect(const char *iap, dbus_uint32_t flags, void *arg);

  /**
   * Request disconnection of an IAP. The IAP connection will disconnected
   * only after all programs using the connection have requested for
   * disconnection. Programs should normally call this function just before
   * they quit. Everything depends how the program is meant to be used.
   *
   * @param iap The name of the IAP.
   * @param arg Parameter passed for callback function (aka. user_data).
   * @retval OSSO_OK for success
   * @retval OSSO_ERROR for an error
   * @note Succesful return value indicates that the request has been
   *	sent to Connectivity Module. After the connection has been
   *	torn down actually an OSSO_IAP_DISCONNECTED event will be
   *	sent. OSSO_IAP_ERROR event indicates an error and application should
   *    assume that the connection is disconnected.
   */
  gint osso_iap_disconnect(const char *iap, void *arg);

  /**
   * Request statistics from an IAP.
   *
   * @param iap The name of the IAP whose statistics are asked for.
   * @param arg Parameter passed for callback function (aka. user_data).
   * @retval OSSO_OK for success
   * @retval OSSO_ERROR for an error
   * @note The statistics will be reported in an OSSO_IAP_STATISTICS
   *	event. For asynchronous errors OSSO_IAP_ERROR event will be
   *	sent.
   */
  gint osso_iap_get_statistics(const char *iap, void *arg);

  /**
   * Get a list of configured IAPs.
   * @return A singly linked list of configured IAPs or NULL for an error.
   * @note Caller is responsible to free the returned list. You should
   *	g_free() each string in the list and g_slist_free() the list itself.
   * @note This function uses GConf so make sure you have called g_type_init()
   *	before calling this.
   */
  GSList *osso_iap_get_configured_iaps(void);

  /*@}*/

  /**@defgroup OSSO_IAP_SOCKET Socket interface
   *
   * Internet Connectivity (IC) Socket interface is a minimalist way to use
   * connectivity services. Connection is created and disconnected during
   * socket creation and tear down. If this is not enough, see @ref
   * OSSO_IAP_EXTENDED.
   *
   * @remarks <em>@ref OSSO_IAP_EXTENDED and @ref OSSO_IAP_SOCKET should
   * not be used together in the same application!</em> They are two
   * different methods to use the IC API, and they are mutually exclusive.
   *
   * You can use blocking network functions with the IC Socket interface
   * which might be an advantage compared to the @ref OSSO_IAP_EXTENDED. If
   * the IAP is disconnected  then all sockets created with
   * osso_socket() are closed using shutdown().
   *
   * The socket interface is the easiest way to port existing application
   * to use IC API. Just replace all socket() and close() calls with
   * osso_socket() and osso_close(), respectively.
   *
   * For a complete example, see <tt>
   * /usr/share/doc/osso-ic-dev/example/iap-connect.c</tt> from osso-ic-dev
   * package.
   * 
   */
  /*@{*/

  /**
   * Connects to an IAP, blocks until the connection is established and
   * returns a new socket. Creates the socket using Linux's socket()
   * call and the parameters are same as in socket().
   *
   * @param domain Communication domain, ie. the protocol family.
   * @param type Socket type.
   * @param protocol Protocol to be used.
   * @return On success a file descriptor of the created socket is returned.
   * On failure -1 is returned and errno is set appropriately. If OSSO IAP
   * connection establishment fails, errno is set to EHOSTUNREACH.
   */
  int osso_socket(int domain, int type, int protocol);

  /**
   * Closes the socket and disconnects an IAP connection if there are
   * no more sockets opened with the osso_socket().
   *
   * @param fd File descriptor of the socket to be closed.
   * @return If successful, zero is returned. Otherwise -1 will be returned
   * and errno contains the error.
   */
  int osso_close(int fd);



  /*@}*/

#ifdef __cplusplus
}
#endif

#endif /* OSSO_IC_H */
