/*
A simple test program for GPS BT management API.

Copyright (C) 2006 Nokia Corporation. All rights reserved.

Author: Jukka Rissanen <jukka.rissanen@nokia.com>

Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met:

Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.
Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer in the documentation and/or other materials provided with the distribution.
The name of the author may not be used to endorse or promote products derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <pthread.h>

#include "gpsbt.h"

#define MAX_ERROR_BUF_LEN 255

int term_flag = 0;
int reading_failed = 0;

void terminate(int sig)
{
  printf("signal %d received\n", sig);
  term_flag = 1;
}


static void *handler_thread(void *arg)
{
  int sock = (int)arg, ret;
#define MAX_BUF_LEN 255
  char buf[MAX_BUF_LEN+1];
  time_t t;

  memset(buf, 0, MAX_BUF_LEN+1);

  t = time(0);
  printf("Thread for sock %d started, %s", sock, ctime(&t));

  while (!term_flag) {
    memset(buf, 0, MAX_BUF_LEN); /* this must be done! */
    ret = read(sock, buf, MAX_BUF_LEN);
    if (ret<0) {
      perror("read");
      printf("Socket to gpsd closed, reading thread quit, %s", ctime(&t));
      break;
    }
    if (ret==0) {
      t = time(0);
      printf("EOF from gpsd socket, reading thread quit, %s", ctime(&t));
      break;
    }
    printf("recv(%d)=\"%s\"\n", ret, buf);
  }

  printf("Thread for sock %d stopped\n", sock);
  reading_failed = 1;

  return 0;
}


/* Just send a random command to gpsd. See gpsd(8) for details about
 * available commands. It is possible to either use the gpsd socket
 * directly or through libgps, now it is used through socket but
 * you may want to use libgps so that there is no need to parse socket
 * output by yourself.
 */
void send_cmd(int sock)
{
  struct {
    char *cmd_key;
    char *cmd_desc;
  } cmds[]={
#define C(key,desc) { #key, desc },

    C(a, "altitude")
    C(b, "serial link data")
    C(c, "cycle time")
    C(d, "date")
    C(e, "estimated error")
    C(f, "GPS device name")
 /* C(g, "set information type returned") */
    C(i, "GPS id")
    C(k, "GPS devices")
    C(l, "protocol version")
 /* C(m, "NMEA mode") */
    C(n, "GPS driver mode")
    C(o, "report")
    C(p, "current position")
    C(q, "count of satellites")
 /* C(r, "toggle raw mode") */
    C(s, "NMEA status")
    C(t, "track made good")
    C(u, "rate of climb")
    C(v, "speed in knots")
 /* C(w, "toggle watcher mode") */
    C(x, "return mode")
    C(y, "return various info")
    C(z, "return profiling info")

    { 0, 0 } /* must be last element */
  };

  int idx;
  int max = sizeof(cmds) / (sizeof(char *)*2);

  idx = (int) ((double)max * (rand() / (RAND_MAX + 1.0)));

  if (cmds[idx].cmd_key == 0)
    printf("skipping cmd send to gpsd\n");
  else {
    printf("send %s (%s) to gpsd\n", cmds[idx].cmd_key, cmds[idx].cmd_desc);
    write(sock, cmds[idx].cmd_key, strlen(cmds[idx].cmd_key));
  }
}


int main(int argc, char **argv)
{
  gpsbt_t ctx = {0};  /* clearing the context is important! */
  int st;
  char errbuf[MAX_ERROR_BUF_LEN+1];
  char *bda = NULL;
  int cnt = 0;
  int sock = 0;
  struct sockaddr_in ipa;
  int port = 2947;  /* default gpsd port */
  pthread_t thread;

  if (argc > 1) {
    if (strcmp(argv[1], "-h")==0 || strcmp(argv[1], "--help")==0) {
      printf("Usage: %s [-b <BT address to pair>] [BT address or rfcomm device of GPS device, optional. Example: /dev/rfcomm0 or 00:11:22:33:44:55:66 ]\n", argv[0]);
      exit(0);
    }

    if (strcmp(argv[1], "-b")==0 || strcmp(argv[1], "--bond")==0) {
      /* Initiate the pairing of GPS BT device. */
      bda = argv[2];
      printf("Initiating pairing to %s\n", bda);
      st = gpsbt_init_pairing(bda);
      exit(st);
    }

    bda = argv[1];
    printf("BT address or rfcomm device of GPS device is set to %s\n", bda);
  }

  memset(errbuf, 0, MAX_ERROR_BUF_LEN+1);
  srand(time(0));

  /* wait until we get killed */
  signal(SIGTERM, terminate);
  signal(SIGINT, terminate);

  st = gpsbt_start(bda, 1, 2, 0, errbuf, MAX_ERROR_BUF_LEN, 0, &ctx);
  if (st<0) {
    printf("gpsbt_start(): error: %s, %d [%s]\n", strerror(errno), errno, errbuf);
    goto OUT;
  }

  sleep(1); /* let the gpsd start */

  reading_failed = 1; /* causes start of the reading thread and opens socket
			 to gpsd */

  while(!term_flag) {

    /* Note that gpsd timeouts the connection after one minute if there
     * is no traffic to the client i.e., this process.
     */
    if (reading_failed) {

      if (sock>0)
	close(sock);

      /* Open the socket to gpsd */
      sock = socket(PF_INET, SOCK_STREAM, 0);
      if (sock < 0) {
	printf("socket() error (%s/%d)", strerror(errno), errno);
	st = -1;
	goto OUT;
      }

      memset(&ipa, 0, (size_t)sizeof(struct sockaddr_in));
      ipa.sin_family      = AF_INET;
      ipa.sin_addr.s_addr = htonl(INADDR_LOOPBACK);
      ipa.sin_port = htons(port);

      printf("Connecting to gpsd at %s:%d\n", inet_ntoa(ipa.sin_addr), port);

      st = connect(sock, (struct sockaddr *)&ipa, sizeof(ipa));
      if (st<0) {
	printf("connect() error (%s/%d)", strerror(errno), errno);
	goto OUT;
      }

      st = pthread_create(&thread, NULL, handler_thread, (void *)sock);
      if (st) {
	printf("pthread_create() failed (%s/%d)\n", strerror(errno), errno);
	goto OUT;
      }

      reading_failed = 0;
    }

    if (!(cnt % 10)) {
      printf("gpsd running\n");

      /* as an example send a random message to gpsd */
      send_cmd(sock);
    }

    if (!gpsmgr_is_gpsd_running(&ctx.mgr, NULL, GPSMGR_MODE_JUST_CHECK)) {
      printf("ERROR, gpsd is not running\n");
      break;
    }

    sleep(1);
    cnt++;
  }

  st = gpsbt_stop(&ctx);
  if (st<0) {
    printf("gpsbt_stop(): error: %s, %d\n", strerror(errno), errno);
  }

  close(sock);

 OUT:
  exit(st);
}

