/**
 * @file libgalago/galago-core.h Galago Core API
 * @ingroup Presence
 *
 * @Copyright (C) 2004-2006 Christian Hammond
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#ifndef _GALAGO_CORE_H_
#define _GALAGO_CORE_H_

#include <libgalago/galago-account.h>
#include <libgalago/galago-dbus.h>
#include <libgalago/galago-person.h>
#include <libgalago/galago-service.h>

#define GALAGO_DBUS_CORE_INTERFACE "org.freedesktop.Galago.Core"
#define GALAGO_DBUS_CORE_OBJECT    "/org/freedesktop/Galago/Core"

G_BEGIN_DECLS

/**************************************************************************/
/** @name Core API                                                      */
/**************************************************************************/

#define GALAGO_TYPE_CORE (galago_core_get_type())

/** Cast @c obj to a GalagoCore. */
#define GALAGO_CORE(obj) \
		(G_TYPE_CHECK_INSTANCE_CAST((obj), GALAGO_TYPE_CORE, \
									GalagoCore))

/** Cast @c klass to a GalagoCore. */
#define GALAGO_CORE_CLASS(klass) \
		(G_TYPE_CHECK_CLASS_CAST((klass), GALAGO_TYPE_CORE, \
								 GalagoCoreClass))

/** Check if @c obj is a GalagoCore. */
#define GALAGO_IS_CORE(obj) \
		(G_TYPE_CHECK_INSTANCE_TYPE((obj), GALAGO_TYPE_CORE))

/** Check if @c klass is a GalagoCoreClass. */
#define GALAGO_IS_CORE_CLASS(klass) \
		(G_TYPE_CHECK_CLASS_TYPE((klass), GALAGO_TYPE_CORE))

/** Get the class for the GalagoCore @c obj. */
#define GALAGO_CORE_GET_CLASS(obj) \
		(G_TYPE_INSTANCE_GET_CLASS((obj), GALAGO_TYPE_CORE, \
								   GalagoCoreClass))

typedef struct _GalagoCore        GalagoCore;
typedef struct _GalagoCoreClass   GalagoCoreClass;
typedef struct _GalagoCorePrivate GalagoCorePrivate;

/**
 * Initialization flags
 */
typedef enum
{
	/** Connection will be used for querying data only. */
	GALAGO_INIT_CLIENT      = 0,

	/**
	 * Connection will feed presence data. This takes precedence over
	 * GALAGO_INIT_CLIENT.
	 */
	GALAGO_INIT_FEED        = 1 << 0,

	/** The daemon will not be activated automatically. */
	GALAGO_INIT_NO_ACTIVATION = 1 << 1

} GalagoInitFlags;


struct _GalagoCore
{
	GalagoObject parent_object;

	GalagoCorePrivate *priv;
};

struct _GalagoCoreClass
{
	GalagoObjectClass parent_class;

	/* Signals */
	void (*registered)(GalagoCore *core);
	void (*unregistered)(GalagoCore *core);
	void (*service_added)(GalagoCore *core, GalagoService *service);
	void (*service_removed)(GalagoCore *core, GalagoService *service);
	void (*person_added)(GalagoCore *core, GalagoPerson *person);
	void (*person_removed)(GalagoCore *core, GalagoPerson *person);
	GalagoAccount *(*calc_priority_account)(GalagoCore *core,
											GalagoPerson *person);

#ifndef _DOXYGEN_
	void (*reserved_1)(void);
	void (*reserved_2)(void);
	void (*reserved_3)(void);
	void (*reserved_4)(void);
#endif
};

/**
 * Returns GalagoCore's GType.
 *
 * @return The GType.
 */
GType galago_core_get_type(void);

/**
 * Initializes Galago.
 *
 * @param name  The ID name used for this connection. It's often the
 *              name of the program, without spaces or special characters.
 * @param flags The flags used for this connection.
 *
 * @return TRUE if initialization was successful, or FALSE otherwise.
 */
gboolean galago_init(const char *name, GalagoInitFlags flags);

/**
 * Uninitializes Galago.
 *
 * This will be called automatically on exit, but can be called manually.
 */
void galago_uninit(void);

/**
 * Returns whether or not Galago has been initialized.
 *
 * @return TRUE if Galago has been initialized, or FALSE otherwise.
 */
gboolean galago_is_initted(void);

/**
 * Returns whether or not Galago is connected to the daemon.
 *
 * @return TRUE if Galago is connected, or FALSE otherwise.
 */
gboolean galago_is_connected(void);

/**
 * Returns whether or not Galago is successfully registered to the daemon.
 *
 * @return TRUE if this Galago connection is registered, or FALSE otherwise.
 */
gboolean galago_is_registered(void);

/**
 * Returns whether or not the Galago daemon is active.
 *
 * @return TRUE if the daemon is active, or FALSE.
 */
gboolean galago_is_daemon_active(void);

/**
 * Returns whether or not this instance of Galago is the daemon.
 *
 * @return TRUE if this is the daemon, or FALSE.
 */
gboolean galago_is_daemon(void);

/**
 * Returns Galago's D-BUS connection.
 *
 * @return The D-BUS connection.
 */
DBusConnection *galago_get_dbus_conn(void);

/**
 * Returns the local Galago connection's unique ID.
 *
 * @internal This ID is used in D-BUS object paths.
 *
 * @return This connection's unique ID.
 */
const char *galago_get_uid(void);

/**
 * Returns the D-BUS object path representing this connection.
 *
 * @return This connection's D-BUS object path.
 */
const char *galago_get_client_obj_path(void);

/**
 * Returns whether or not this Galago connection works as a feed.
 *
 * @return TRUE if this connection works as a feed, or FALSE if not.
 */
gboolean galago_is_feed(void);

/**
 * Sets whether or not all services, accounts, etc. should be watched for
 * property updates.
 *
 * This will only add accounts, persons, presences, etc. that have a known
 * parent object. That is, if a signal arrives for an account on a service
 * that the local program does not know about, it will be ignored. If you
 * wish to get information on every single object, call the list query
 * functions first.
 *
 * @param watch_all TRUE if everything should be watched, or FALSE.
 */
void galago_set_watch_all(gboolean watch_all);

/**
 * Returns whether or not all services, accounts, etc. is being watched
 * for property updates.
 *
 * @return TRUE if everything is being watched, or FALSE.
 */
gboolean galago_get_watch_all(void);

/**
 * Creates a service and adds it to the local cache.
 *
 * The name and flags will be computed if the ID is understood internally.
 * Otherwise, they will have to be provided.
 *
 * @param id    The service ID.
 * @param name  The service name.
 * @param flags The flags.
 */
GalagoService *galago_create_service(const char *id, const char *name,
									 GalagoServiceFlags flags);

/**
 * Returns the service with the specified ID.
 *
 * @param id     The service ID.
 * @param origin The desired service's origin.
 * @param query  TRUE if a remote query should be done if there is no
 *               local service found, or FALSE.
 *
 * @return The service, if found, or NULL.
 */
GalagoService *galago_get_service(const char *id, GalagoOrigin origin,
								  gboolean query);

/**
 * Returns a list of known services.
 *
 * This may emit a service-added signal for every object that returns. If
 * your code connects to this signal and calls galago_core_get_services() as
 * a result, you will want to add a lock so that you don't end up with
 * unwanted side-effects.
 *
 * @param origin The desired origin.
 * @param query  TRUE if a remote query should be done if there are no
 *               local services found, or FALSE.
 *
 * @return The list of services, or NULL.
 */
GList *galago_get_services(GalagoOrigin origin, gboolean query);

/**
 * Creates a person and adds it to the local cache.
 *
 * @param id The person ID.
 */
GalagoPerson *galago_create_person(const char *id);

/**
 * Returns the person with the specified UID.
 *
 * @param uid    The person's UID.
 * @param origin The desired person's origin.
 * @param query  TRUE if a remote query should be done if there is no
 *               local person found, or FALSE.
 *
 * @return The person, if found, or NULL.
 */
GalagoPerson *galago_get_person(const char *uid, GalagoOrigin origin,
								gboolean query);

/**
 * Returns a list of known people.
 *
 * This may emit a person-added signal for every object that returns. If
 * your code connects to this signal and calls galago_core_get_people() as
 * a result, you will want to add a lock so that you don't end up with
 * unwanted side-effects.
 *
 * @param origin The desired origin.
 * @param query  TRUE if a remote query should be done if there is no
 *               local people found, or FALSE.
 *
 * @return The people, if found, or NULL.
 */
GList *galago_get_people(GalagoOrigin origin, gboolean query);

/**
 * Returns the "Me" person.
 *
 * If @a origin is GALAGO_LOCAL, this will always return a "Me" person.
 * In the remote case, it will only return one if @a query is TRUE, or
 * if it has already retrieved a remote "Me" person.
 *
 * @param origin The desired person's origin.
 * @param query  TRUE if a remote query should be done if there is no
 *               local person found, or FALSE.
 *
 * @return The "Me" person, if found, or NULL.
 */
GalagoPerson *galago_get_me(GalagoOrigin origin, gboolean query);

/**
 * Returns the core object.
 *
 * This is needed for signals.
 *
 * @return The core object.
 */
GalagoCore *galago_get_core(void);

G_END_DECLS

#endif /* _GALAGO_CORE_H_ */
