/**
 * @file libgalago/galago-presence.h Galago Presence API
 * @ingroup Presence
 *
 * A presence is associated with a #GalagoAccount and contains a list of the
 * account's statuses (#GalagoStatus), the current status, the idle time, and
 * other information.
 *
 * @Copyright (C) 2004-2006 Christian Hammond
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#ifndef _GALAGO_PRESENCE_H_
#define _GALAGO_PRESENCE_H_

#include <sys/time.h>
#include <time.h>

typedef struct _GalagoPresence        GalagoPresence;
typedef struct _GalagoPresenceClass   GalagoPresenceClass;
typedef struct _GalagoPresencePrivate GalagoPresencePrivate;

#include <libgalago/galago-account.h>
#include <libgalago/galago-object.h>
#include <libgalago/galago-person.h>
#include <libgalago/galago-status.h>

/**
 * A presence.
 *
 * A presence is associated with an account and contains a list of the
 * account's statuses, as well as idle times and other information.
 */
struct _GalagoPresence
{
	GalagoObject parent_object;

	GalagoPresencePrivate *priv;

#ifndef _DOXYGEN_
	void (*reserved_1)(void);
	void (*reserved_2)(void);
	void (*reserved_3)(void);
	void (*reserved_4)(void);
#endif
};

/**
 * GalagoPresence's class.
 */
struct _GalagoPresenceClass
{
	GalagoObjectClass parent_class;

	/* Signals */
	void (*idle_changed)(GalagoPresence *presence, gboolean idle,
						 time_t idle_start_time);
	void (*status_added)(GalagoPresence *presence, GalagoStatus *status);
	void (*status_removed)(GalagoPresence *presence, GalagoStatus *status);
	void (*status_updated)(GalagoPresence *presence, GalagoStatus *status,
						   const char *attr_id);
	void (*changed)(GalagoPresence *presence);

#ifndef _DOXYGEN_
	void (*reserved_1)(void);
	void (*reserved_2)(void);
	void (*reserved_3)(void);
	void (*reserved_4)(void);
#endif
};

#define GALAGO_TYPE_PRESENCE (galago_presence_get_type())

/** Cast @c obj to a GalagoPresence. */
#define GALAGO_PRESENCE(obj) \
		(G_TYPE_CHECK_INSTANCE_CAST((obj), GALAGO_TYPE_PRESENCE, \
									GalagoPresence))

/** Cast @c klass to a GalagoPresenceClass. */
#define GALAGO_PRESENCE_CLASS(klass) \
		(G_TYPE_CHECK_CLASS_CAST((klass), GALAGO_TYPE_PRESENCE, \
								 GalagoPresenceClass))

/** Check if @c obj is a GalagoPresence. */
#define GALAGO_IS_PRESENCE(obj) \
		(G_TYPE_CHECK_INSTANCE_TYPE((obj), GALAGO_TYPE_PRESENCE))

/** Check if @c klass is a GalagoPresenceClass. */
#define GALAGO_IS_PRESENCE_CLASS(klass) \
		(G_TYPE_CHECK_CLASS_TYPE((klass), GALAGO_TYPE_PRESENCE))

/** Get the class for the GalagoPresence @c obj. */
#define GALAGO_PRESENCE_GET_CLASS(obj) \
		(G_TYPE_INSTANCE_GET_CLASS((obj), GALAGO_TYPE_PRESENCE, \
								   GalagoPresenceClass))

#define GALAGO_DBUS_PRESENCE_INTERFACE "org.freedesktop.Galago.Presence"

G_BEGIN_DECLS

/**************************************************************************/
/** @name Presence API                                                    */
/**************************************************************************/
/*@{*/

/**
 * Returns GalagoPresence's GType.
 *
 * @return The GType.
 */
GType galago_presence_get_type(void);

/**
 * Sets the idle start time on a presence.
 *
 * @param presence        The presence.
 * @param idle            TRUE if the user is idle, otherwise FALSE.
 * @param idle_start_time The idle start time in seconds.
 */
void galago_presence_set_idle(GalagoPresence *presence, gboolean idle,
							  time_t idle_start_time);

/**
 * Sets a list of statuses in a presence.
 *
 * The presence claims ownership of the list and will free it when
 * destroyed.
 *
 * @param presence The presence.
 * @param statuses The list of statuses.
 */
void galago_presence_set_statuses(GalagoPresence *presence, GList *statuses);

/**
 * Adds a single status to a presence.
 *
 * @param presence The presence.
 * @param status   The status.
 */
void galago_presence_add_status(GalagoPresence *presence,
								GalagoStatus *status);

/**
 * Removes a single status from a presence.
 *
 * The status being removed must not be an exclusive status.
 *
 * @param presence  The presence.
 * @param status_id The ID of the status to remove.
 */
void galago_presence_remove_status(GalagoPresence *presence,
								   const char *status_id);

/**
 * Clears all statuses in a presence.
 *
 * @param presence The presence.
 */
void galago_presence_clear_statuses(GalagoPresence *presence);

/**
 * Returns the account from a presence.
 *
 * @param presence The presence.
 *
 * @return The presence's account.
 */
GalagoAccount *galago_presence_get_account(const GalagoPresence *presence);

/**
 * Returns the idle state from a presence.
 *
 * @param presence The presence.
 *
 * @return The presence's idle state.
 */
gboolean galago_presence_is_idle(const GalagoPresence *presence);

/**
 * Returns the presence's idle time in seconds.
 *
 * @param presence The presence.
 *
 * @return The presence's idle time in seconds.
 */
time_t galago_presence_get_idle_time(const GalagoPresence *presence);

/**
 * Returns the time the presence was marked idle.
 *
 * @param presence The presence.
 *
 * @return The presence's start idle time.
 */
time_t galago_presence_get_idle_start_time(const GalagoPresence *presence);

/**
 * Returns whether or not this is a discarded presence.
 *
 * A discarded presence signifies that all existing presence info for
 * that user is to be discarded.
 *
 * @param presence The presence.
 *
 * @return TRUE if the presence is discarded, or FALSE.
 */
gboolean galago_presence_is_discarded(const GalagoPresence *presence);

/**
 * Returns whether or not a presence is considered available.
 *
 * @param presence The presence.
 *
 * @return TRUE if the presence is available, or FALSE.
 */
gboolean galago_presence_is_available(const GalagoPresence *presence);

/**
 * Returns the list of statuses from a presence.
 *
 * @param presence The presence.
 *
 * @return The list of statuses.
 */
GList *galago_presence_get_statuses(const GalagoPresence *presence);

/**
 * Returns the active exclusive status from a presence.
 *
 * @param presence The presence.
 *
 * @return The active exclusive status, if set, or NULL.
 */
GalagoStatus *galago_presence_get_active_status(const GalagoPresence *presence);

/**
 * Returns whether or not the status with the specified ID in a presence
 * is exclusive.
 *
 * @param presence  The presence.
 * @param status_id The ID of the status.
 *
 * @return TRUE if the status is exclusive, or FALSE.
 */
gboolean galago_presence_is_status_exclusive(const GalagoPresence *presence,
											 const char *status_id);

/**
 * Returns the status from a presence with the specified ID.
 *
 * @param presence  The presence.
 * @param status_id The status ID.
 *
 * @return The status, if found, or NULL.
 */
GalagoStatus *galago_presence_get_status(const GalagoPresence *presence,
										 const char *status_id);

/**
 * Returns whether or not a presence has a status with the specified ID.
 *
 * @param presence  The presence.
 * @param status_id The status ID.
 *
 * @return TRUE if the presence has the status, or FALSE.
 */
gboolean galago_presence_has_status(const GalagoPresence *presence,
									const char *status_id);

/**
 * Returns whether or not a presence has a status with the specified
 * primitive type.
 *
 * @param presence The presence.
 * @param type     The status type.
 *
 * @return TRUE if the presence has the status, or FALSE.
 */
gboolean galago_presence_has_status_type(const GalagoPresence *presence,
										 GalagoStatusType type);

/**
 * Compares two presences for availability.
 *
 * @param presence1 The first presence.
 * @param presence2 The second presence.
 *
 * @return -1 if @a presence1 is less available than @a presence2,
 *          0 if @a presence1 is as available as @a presence2, and
 *          1 if @a presence1 is more available than @a presence2.
 */
int galago_presence_compare(const GalagoPresence *presence1,
							const GalagoPresence *presence2);

/*}@*/

G_END_DECLS

#endif /* _GALAGO_PRESENCE_H_ */
