/*
 * This file is part of maemopad
 *
 * Copyright (C) 2005 Nokia Corporation.
 *
 *
 * This software is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */
                     
#include <ui/interface.h>
#include <ui/callbacks.h>
#include <gtk/gtk.h>
#include <libintl.h>

#include <hildon-widgets/hildon-app.h>
#include <hildon-widgets/hildon-file-chooser-dialog.h>
#include <hildon-widgets/hildon-note.h>
#include <hildon-widgets/hildon-font-selection-dialog.h>

#define MAIN_TEXT MAIN_VIEW_NAME

/* Privates: */
static void create_menu( MainView *main );
static void create_toolbar( MainView *main );
void create_textarea( MainView *main );
PangoFontDescription* create_desc_from_list( PangoAttrList* list );

/* Creates and initialises a main_view */
MainView* interface_main_view_new( AppData *data )
{
    /* Zero memory with g_new0 */
    MainView* result = g_new0( MainView, 1 );
    /* Create GTK vbox which will be our text area */
    GtkWidget *main_vbox = gtk_vbox_new( FALSE, 0 );
    result->clipboard = gtk_clipboard_get (GDK_SELECTION_CLIPBOARD);
    /* Initialize interface data */
    result->file_edited = FALSE;
    /* Store handle to app's data to view's data */
    result->data = data;
    /* Store handle to hildon_appview to app's data */
    result->data->main_view = HILDON_APPVIEW(hildon_appview_new(  _( MAIN_TEXT ) ));
    /* Create all our necessary parts */
    create_textarea( result );
    create_toolbar( result ); 
    create_menu( result );
    /* Put GTK vbox to app's view */
    gtk_container_add( GTK_CONTAINER( result->data->main_view ), main_vbox );
    /* Put scrolledwindow to GTK vbox and show it */
    gtk_box_pack_start( GTK_BOX( main_vbox ), result->scrolledwindow, TRUE, TRUE, 0 );
    gtk_widget_show( main_vbox );
    /* Focus on textarea */
    gtk_widget_grab_focus( GTK_WIDGET(result->textview ));
    /* Initialize our textview's file (no file opened yet) */
    result->file_edited=FALSE;
    result->file_name=NULL;
    /* Return with new mainview handle */
    return result;
}



/* clean up the allocated memory */
void interface_main_view_destroy( MainView *main )
{
    g_free( main );
}



/* File chooser */
gchar* interface_file_chooser(MainView * mainview, GtkFileChooserAction action)
{
    GtkWidget *dialog;
    gchar* filename = NULL;

    dialog = hildon_file_chooser_dialog_new( GTK_WINDOW(mainview->data->app), action);
    gtk_widget_show_all (GTK_WIDGET(dialog));
    if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_OK)
    {
        filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));
    }
    gtk_widget_destroy (dialog);
    return filename;
}

/* Font chooser */
PangoFontDescription* interface_font_chooser( MainView * main )
{
    PangoFontDescription *font = NULL;
    HildonFontSelectionDialog *dialog = NULL;
    PangoAttrList *default_list = NULL;
    PangoFontDescription *default_font = NULL;
    PangoAttribute *default_attr = NULL;

    /* create default font for dialog to default_list */
    default_font = pango_font_description_new();
    pango_font_description_set_family(default_font, "Nokia Sans");
    /* pango_font_description_set_size(default_font, 22); */
    default_attr = pango_attr_font_desc_new(default_font);
    default_attr->start_index = 0;
    default_attr->end_index = -1;
    default_list = pango_attr_list_new();
    pango_attr_list_insert(default_list, default_attr);

    /* create dialog */
    dialog = HILDON_FONT_SELECTION_DIALOG ( hildon_font_selection_dialog_new( NULL, NULL ) );
    hildon_font_selection_dialog_set_font(dialog, default_list);

    gtk_widget_show_all (GTK_WIDGET(dialog));
    if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_OK)
    {
        default_list = hildon_font_selection_dialog_get_font ( HILDON_FONT_SELECTION_DIALOG ( dialog ) );
        font = create_desc_from_list( default_list );
    }
    gtk_widget_destroy(GTK_WIDGET(dialog));

    return font;
}



/* Save changes note */
gint interface_save_changes_note(MainView * main)
{
    HildonNote *hn = NULL;
    gint response = FALSE;

    g_assert(main != NULL && main->data->app != NULL);

    hn = HILDON_NOTE(hildon_note_new_confirmation_add_buttons
                     (GTK_WINDOW(main->data->app),
                      _("maemopad_save_changes_made"),
                      _("maemopad_yes"), CONFRESP_YES,
                      _("maemopad_no"), CONFRESP_NO,
                      NULL, NULL));
    response = gtk_dialog_run(GTK_DIALOG(hn));
    gtk_widget_destroy(GTK_WIDGET(hn));
    return response;
}



/* Error note */
void interface_error( MaemopadError me, MainView *main )
{
    GtkWidget *dialog = NULL;
    gchar *msg = NULL;

    switch (me) {

    case MAEMOPAD_ERROR_INVALID_URI:
        msg = g_strdup( _("maemopad_invalid_uri") );
        break;

    case MAEMOPAD_ERROR_SAVE_FAILED:
        msg = g_strdup( _("maemopad_save_failed") );
        break;

    case MAEMOPAD_ERROR_OPEN_FAILED:
        msg = g_strdup( _("maemopad_open_failed") );
        break;

    case MAEMOPAD_NO_ERROR:
        return;

    default:
        g_assert_not_reached();
    }

    dialog = hildon_note_new_information( GTK_WINDOW( main->data->app ), msg );

    gtk_dialog_run( GTK_DIALOG( dialog ) );
    gtk_widget_destroy( dialog );
    g_free( msg );
}



/* Privates */



/* Create the menu items needed for the drop down menu */
static void create_menu( MainView *main )
{
    /* Create needed handles */
    GtkMenu *main_menu;
    GtkWidget *file_menu, *edit_menu;
    GtkWidget *separator = NULL;
    GtkWidget *close = NULL;

    /* Get the menu of our view */
    main_menu = hildon_appview_get_menu( main->data->main_view );
    /* Create new menus for submenus in our drop down menu */
    file_menu = gtk_menu_new ();
    edit_menu = gtk_menu_new ();

    /* Create the menu items */
    main->file_item = gtk_menu_item_new_with_label ( _("File") );
    main->new_item = gtk_menu_item_new_with_label ( _("New") );
    main->open_item = gtk_menu_item_new_with_label ( _("Open") );
    main->save_item = gtk_menu_item_new_with_label ( _("Save") );
    main->saveas_item = gtk_menu_item_new_with_label ( _("Save As...") );
    main->edit_item = gtk_menu_item_new_with_label ( _("Edit") );
    main->cut_item = gtk_menu_item_new_with_label ( _("Cut") );
    main->copy_item = gtk_menu_item_new_with_label ( _("Copy") );
    main->paste_item = gtk_menu_item_new_with_label ( _("Paste") );
    main->font_item = gtk_menu_item_new_with_label ( _("Font") );
    main->fullscreen_item = gtk_menu_item_new_with_label ( _("Full Screen") );
    separator = gtk_separator_menu_item_new();
    close = gtk_menu_item_new_with_label( _("Close") );

    /* Add menu items to right menus */
    gtk_menu_append( main_menu, main->file_item );
    gtk_menu_append( file_menu, main->new_item );
    gtk_menu_append( file_menu, main->open_item );
    gtk_menu_append( file_menu, main->save_item );
    gtk_menu_append( file_menu, main->saveas_item );
    gtk_menu_append( main_menu, main->edit_item );
    gtk_menu_append( edit_menu, main->cut_item );
    gtk_menu_append( edit_menu, main->copy_item );
    gtk_menu_append( edit_menu, main->paste_item );
    gtk_menu_append( main_menu, main->font_item );
    gtk_menu_append( main_menu, main->fullscreen_item );
    gtk_menu_append( main_menu, separator );
    gtk_menu_append( main_menu, close );

    /* Add submenus to the right items */
    gtk_menu_item_set_submenu( GTK_MENU_ITEM (main->file_item), file_menu );
    gtk_menu_item_set_submenu( GTK_MENU_ITEM (main->edit_item), edit_menu );

    /* Attach the callback functions to the activate signal */
    g_signal_connect( G_OBJECT( main->new_item ), "activate", 
		      G_CALLBACK ( callback_file_new), main );
    g_signal_connect( G_OBJECT( main->open_item ), "activate", 
		      G_CALLBACK ( callback_file_open), main );
    g_signal_connect( G_OBJECT( main->save_item ), "activate", 
              G_CALLBACK ( callback_file_save), main );
    g_signal_connect( G_OBJECT( main->saveas_item ), "activate", 
		      G_CALLBACK ( callback_file_saveas), main );
    g_signal_connect( G_OBJECT( main->cut_item ), "activate", 
		      G_CALLBACK ( callback_edit_cut), main );
    g_signal_connect( G_OBJECT( main->copy_item ), "activate", 
		      G_CALLBACK( callback_edit_copy ), main );
    g_signal_connect( G_OBJECT( main->paste_item ), "activate", 
		      G_CALLBACK( callback_edit_paste ), main );
    g_signal_connect( G_OBJECT( main->font_item ), "activate", 
		      G_CALLBACK ( callback_font ), main );
    g_signal_connect( G_OBJECT( main->fullscreen_item ), "activate", 
		      G_CALLBACK ( callback_fullscreen ), main );
    g_signal_connect( G_OBJECT( close ), "activate", gtk_main_quit, NULL );

    /* We need to show menu items */
    gtk_widget_show_all( GTK_WIDGET( main_menu ) );
}



/* Create toolbar to mainview */
static void create_toolbar ( MainView *main )
{
    /* Create new GTK toolbar */
    main->toolbar = gtk_toolbar_new ();

 	/* Set toolbar properties */
    gtk_toolbar_set_orientation( GTK_TOOLBAR(main->toolbar), GTK_ORIENTATION_HORIZONTAL);
    gtk_toolbar_set_style( GTK_TOOLBAR(main->toolbar), GTK_TOOLBAR_BOTH_HORIZ);

    /* Make menus and buttons to toolbar: */
    /* Create toolitems using defined items from stock */
    main->new_tb = gtk_tool_button_new_from_stock(GTK_STOCK_NEW);
    main->open_tb = gtk_tool_button_new_from_stock(GTK_STOCK_OPEN);
    main->save_tb = gtk_tool_button_new_from_stock(GTK_STOCK_SAVE);
    main->saveas_tb = gtk_tool_button_new_from_stock(GTK_STOCK_SAVE_AS);
    main->separator_tb1 = gtk_separator_tool_item_new();
    main->cut_tb = gtk_tool_button_new_from_stock(GTK_STOCK_CUT);
    main->copy_tb = gtk_tool_button_new_from_stock(GTK_STOCK_COPY);
    main->paste_tb = gtk_tool_button_new_from_stock(GTK_STOCK_PASTE);
    main->separator_tb2 = gtk_separator_tool_item_new();
    main->font_tb = gtk_tool_button_new_from_stock( GTK_STOCK_SELECT_FONT );
    main->fullscreen_tb = gtk_tool_button_new_from_stock( GTK_STOCK_ZOOM_FIT );

    /* Insert items to toolbar */
    gtk_toolbar_insert ( GTK_TOOLBAR(main->toolbar), main->new_tb, -1);
    gtk_toolbar_insert ( GTK_TOOLBAR(main->toolbar), main->open_tb, -1);
    gtk_toolbar_insert ( GTK_TOOLBAR(main->toolbar), main->save_tb, -1);
    gtk_toolbar_insert ( GTK_TOOLBAR(main->toolbar), main->saveas_tb, -1);
    gtk_toolbar_insert ( GTK_TOOLBAR(main->toolbar), main->separator_tb1, -1);
    gtk_toolbar_insert ( GTK_TOOLBAR(main->toolbar), main->cut_tb, -1);
    gtk_toolbar_insert ( GTK_TOOLBAR(main->toolbar), main->copy_tb, -1);
    gtk_toolbar_insert ( GTK_TOOLBAR(main->toolbar), main->paste_tb, -1);
    gtk_toolbar_insert ( GTK_TOOLBAR(main->toolbar), main->separator_tb2, -1);
    gtk_toolbar_insert ( GTK_TOOLBAR(main->toolbar), main->font_tb, -1);
    gtk_toolbar_insert ( GTK_TOOLBAR(main->toolbar), main->fullscreen_tb, -1);

    /* Connect signals to buttons */
    g_signal_connect(G_OBJECT(main->new_tb), "clicked",
                     G_CALLBACK(callback_file_new), main);
    g_signal_connect(G_OBJECT(main->open_tb), "clicked",
                     G_CALLBACK(callback_file_open), main);
    g_signal_connect(G_OBJECT(main->save_tb), "clicked",
                     G_CALLBACK(callback_file_save), main);
    g_signal_connect(G_OBJECT(main->saveas_tb), "clicked",
                     G_CALLBACK(callback_file_saveas), main);
    g_signal_connect(G_OBJECT(main->cut_tb), "clicked",
                     G_CALLBACK(callback_edit_cut), main);
    g_signal_connect(G_OBJECT(main->copy_tb), "clicked",
                     G_CALLBACK(callback_edit_copy), main);
    g_signal_connect(G_OBJECT(main->paste_tb), "clicked",
                     G_CALLBACK(callback_edit_paste), main);
    g_signal_connect(G_OBJECT(main->font_tb), "clicked",
                     G_CALLBACK(callback_font), main);
   g_signal_connect(G_OBJECT(main->fullscreen_tb), "clicked",
                     G_CALLBACK(callback_fullscreen), main);
	
    /* Attach toolbar to main view */
    hildon_appview_set_toolbar( main->data->main_view, GTK_TOOLBAR(main->toolbar));

    /* Show toolbar */
    gtk_widget_show_all (GTK_WIDGET(main->toolbar));
    gtk_widget_show_all (GTK_WIDGET(main->data->main_view) );
}



/* Create the text area */
void create_textarea( MainView *main )
{
    /* Scrolled window */
    main->scrolledwindow = gtk_scrolled_window_new(NULL, NULL);
    gtk_widget_show(main->scrolledwindow);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(main->scrolledwindow),
                                   GTK_POLICY_AUTOMATIC,
                                   GTK_POLICY_AUTOMATIC);
    /* Text view */
    main->textview = gtk_text_view_new ();

    /* Some text view settings */
    gtk_text_view_set_editable ( GTK_TEXT_VIEW (main->textview), TRUE );
    gtk_text_view_set_left_margin (GTK_TEXT_VIEW (main->textview), 10 );
    gtk_text_view_set_right_margin (GTK_TEXT_VIEW (main->textview), 10 );

    /* Get handle */
    main->buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (main->textview));

    /* Put textview under scrolledwindow and show it*/
    gtk_container_add(GTK_CONTAINER(main->scrolledwindow), main->textview);
    gtk_widget_show(main->textview);

    /* Change default font throughout the widget */
    main->font_desc = pango_font_description_from_string ("Nokia Sans 22");
    gtk_widget_modify_font (main->textview, main->font_desc);

    /* Connect signals */
    g_signal_connect (G_OBJECT (main->buffer), "modified-changed",
                      G_CALLBACK(callback_buffer_modified), main);
    g_signal_connect (G_OBJECT (main->buffer), "changed",
                      G_CALLBACK(callback_buffer_modified), main);
}



/* Gets PangoFontDescription from iterator */
PangoFontDescription* create_desc_from_list( PangoAttrList* list )
{
    PangoFontDescription *font = NULL;
    PangoAttribute *pattr = NULL;
    PangoAttrIterator *iter = NULL;
    GSList *attrs = NULL;
    GSList *curattr = NULL;

    font = pango_font_description_new();
    if (font == NULL) {
        return NULL;
    }
    iter = pango_attr_list_get_iterator( list );
    if ( iter == NULL ) {
        return NULL;
    }
    attrs = pango_attr_iterator_get_attrs( iter );
    if ( attrs == NULL ) {
        return NULL;
    }

    /* Search needed desc from list's attributes */
    for (curattr = attrs; curattr != NULL; curattr = curattr->next) {
        pattr = (PangoAttribute *) curattr->data;

        switch (pattr->klass->type) {
        case PANGO_ATTR_FONT_DESC:
            pango_attr_iterator_get_font(iter, font, NULL, NULL);
            break;
        default:
            break;
        }
    }
    return font;
}
