/*
 * Maemo Games Startup
 * Copyright (c) 2006 INdT.
 * @author Andre Moreira Magalhaes <andre.magalhaes@indt.org.br>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "startup_config.h"

#include <stdio.h>
#include <string.h>
#include <glib/gmem.h>
#include <glib/gmessages.h>
#include <glib/gstrfuncs.h>

struct _StartupConfigPrivate 
{
  gboolean initialized;
  gchar *current_group;
  GHashTable *hash_group;
  GHashTable *current_hash_entry;
};

static void startup_config_cleanup             (StartupConfig *sc);
static void startup_config_hash_free_str_func  (gpointer data);
static void startup_config_hash_free_hash_func (gpointer data);

StartupConfig *
startup_config_new (const gchar *file)
{
  StartupConfig *sc;
  
  g_return_val_if_fail (file != NULL, NULL);
  
  sc = g_new0 (StartupConfig, 1);
  sc->file = g_strdup (file);
  
  sc->priv = g_new0 (StartupConfigPrivate, 1);
  sc->priv->initialized = FALSE;
  sc->priv->current_group = NULL;
  sc->priv->hash_group = NULL;
  sc->priv->current_hash_entry = NULL;

  return sc;
}

void 
startup_config_destroy (StartupConfig *sc)
{
  g_return_if_fail (sc != NULL);

  g_free (sc->file);
  sc->file = NULL;

  startup_config_cleanup (sc);
  g_free (sc->priv);
}

gboolean 
startup_config_parse_file (StartupConfig *sc)
{
  FILE *file;
  gchar line[256];
  gchar *group, *key, *value;
#ifdef DEBUG
  guint current_line = 0;
#endif  
  
  g_return_val_if_fail (sc != NULL, FALSE);
 
  if (sc->priv->initialized)
    startup_config_cleanup (sc);  

  file = fopen (sc->file, "rb");
  if (!file) {
    // error - file not found or permission denied
    return FALSE;
  }
  
  sc->priv->hash_group = g_hash_table_new_full (g_str_hash, 
      g_str_equal, 
      startup_config_hash_free_str_func,
      startup_config_hash_free_hash_func);
      
  while (!feof (file)) {
#ifdef DEBUG
    ++current_line;
#endif  

    fgets (line, 256, file);
    g_strchug (line);
    if (line[0] == '\0' || line[0] == '#')
    {
      continue;
    }

    g_strchomp (line);
    
    // group found
    if (line[0] == '[') {
      if (line[strlen (line) - 1] != ']') {
#ifdef DEBUG      
        g_print ("invalid group '%s' at line '%d'\n", line, current_line);
#endif

        startup_config_cleanup (sc);
        fclose (file);
        return FALSE;
      }

      group = g_new0 (gchar, strlen (line) - 1);
      g_strlcpy (group, line + 1, strlen(line) - 1);
 
      sc->priv->current_hash_entry = g_hash_table_new_full (g_str_hash, 
          g_str_equal, 
          startup_config_hash_free_str_func,
          startup_config_hash_free_str_func);

      g_hash_table_insert (sc->priv->hash_group,
          group, 
          sc->priv->current_hash_entry);
      continue;
    }
    
    value = strchr (line, '=');
    if (!value) { 
#ifdef DEBUG      
      g_print ("invalid entry '%s' at line '%d'\n", line, current_line);
#endif
      
      startup_config_cleanup (sc);
      fclose (file);
      return FALSE;
    }
      
    if (sc->priv->current_hash_entry == NULL) {
      // no group
      sc->priv->current_hash_entry = g_hash_table_new_full (g_str_hash, 
          g_str_equal, 
          startup_config_hash_free_str_func,
          startup_config_hash_free_str_func);
      g_hash_table_insert (sc->priv->hash_group,
          g_strdup(""), 
          sc->priv->current_hash_entry);
    }
   
    key = g_new0 (gchar, value - line + 1);
    g_strlcpy (key, line, value - line + 1);
    
    // TODO test duplicate entry 

    g_hash_table_insert (sc->priv->current_hash_entry, 
        key, 
        g_strdup (value + 1));
  }
 
  sc->priv->current_hash_entry = NULL; 
  sc->priv->initialized = TRUE;
    
  fclose (file);
  return TRUE;
}

void 
startup_config_set_group (StartupConfig *sc, 
                          const gchar   *group)
{
  g_return_if_fail (sc != NULL);
 
  if (sc->priv->current_group)
    g_free (sc->priv->current_group);

  if (group)
    sc->priv->current_group = g_strdup (group);
  else 
    sc->priv->current_group = NULL;
}

G_CONST_RETURN gchar *
startup_config_read_entry (StartupConfig *sc,
                           const gchar   *entry,
                           const gchar   *default_value)
{
  gchar *group;
  gpointer *key, *value;
  
  g_return_val_if_fail (sc != NULL, NULL);
  g_return_val_if_fail (sc->priv->initialized != FALSE, NULL);
  g_return_val_if_fail (entry != NULL, NULL);
 
  if (!sc->priv->current_hash_entry) { // use default group
    if (sc->priv->current_group)
      group = sc->priv->current_group;
    else
      group = "";
      
    if (!g_hash_table_lookup_extended (sc->priv->hash_group, group, (gpointer) &key, (gpointer) &value)) {
#ifdef DEBUG
      g_print ("%s (%d): invalid group '%s'. use startup_config_set_group\n",
          __FUNCTION__, __LINE__, group);
#endif
      return NULL;
    }
    sc->priv->current_hash_entry = (GHashTable *) value;
  }
      
  if (g_hash_table_lookup_extended (sc->priv->current_hash_entry, entry, (gpointer) &key, (gpointer) &value)) {
    return (gchar *) value;
  }

  return default_value;
}

static void 
startup_config_cleanup (StartupConfig *sc)
{
  if (!sc->priv->initialized)
    return;

  sc->priv->initialized = FALSE;
  if (sc->priv->current_group) {
    g_free (sc->priv->current_group);
    sc->priv->current_group = NULL;
  }

  g_hash_table_destroy (sc->priv->hash_group);
  sc->priv->hash_group = NULL;
  sc->priv->current_hash_entry = NULL;
}

static void 
startup_config_hash_free_str_func (gpointer data)
{ 
  g_free (data);
}

static void 
startup_config_hash_free_hash_func (gpointer data)
{
  g_hash_table_destroy ((GHashTable *) data);
}
