/*
 * Maemo Games Startup
 * Copyright (c) 2006 INdT.
 * @author Andre Moreira Magalhaes <andre.magalhaes@indt.org.br>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include "startup_config.h"

#include <stdio.h>
#include <glib.h>
#include <gtk/gtk.h>

#include <libosso.h>
#include <osso-log.h>
#include "startup_app.h"

struct _StartupApp {
  StartupConfig *config;
  osso_context_t *osso;
  guint state;

  startup_app_top_cb *top_cb;
  gpointer top_cb_data;
  startup_app_state_change_cb *state_change_cb;
  gpointer state_change_cb_data;

  const gchar *service;
  const gchar *path;
  const gchar *iface;
};

/* libosso callbacks */
static void startup_app_top_event_cb (const gchar     *args, 
                                      gpointer         data);
static void startup_app_hw_event_cb  (osso_hw_state_t *state, 
                                      gpointer         data);
static gint startup_app_rpc_event_cb (const gchar     *interface, 
                                      const gchar     *method,
                                      GArray          *arguments, 
                                      gpointer         data,
                                      osso_rpc_t      *retval);

#define GAME_RUN_METHOD      "game_run"
#define GAME_CONTINUE_METHOD "game_continue"
#define GAME_RESTART_METHOD  "game_restart"
#define GAME_PAUSE_METHOD    "game_pause"
#define GAME_CLOSE_METHOD    "game_close"
#define GAME_CRASHED_METHOD  "game_crashed"

static const char *method_strings[LAST_MESSAGE] = {
  GAME_RUN_METHOD,
  GAME_CONTINUE_METHOD,
  GAME_RESTART_METHOD,
  GAME_PAUSE_METHOD,
  GAME_CLOSE_METHOD
};

StartupApp *
startup_app_new (void)
{
  StartupApp *app;
  
  app = g_new0 (StartupApp, 1);
  g_assert (app);
  
  app->config = NULL;
  app->osso = NULL;
  app->state = GAME_UNLOADED;
  app->top_cb = NULL;
  app->top_cb_data = NULL;
  app->state_change_cb = NULL;
  app->state_change_cb_data = NULL;
  
  return app;
}

void
startup_app_destroy (StartupApp *app)
{
  g_return_if_fail (app != NULL);

  if (app->osso)
    startup_app_deinitialize (app);
 
  if (app->config) {
    startup_config_destroy (app->config);
    app->config = NULL;
  }
}

gboolean
startup_app_set_config_file (StartupApp  *app,
                             const gchar *file)
{
  g_return_val_if_fail (app != NULL, FALSE);
  g_return_val_if_fail (file != NULL, FALSE);

  if (app->config)
    startup_config_destroy (app->config);
  
  app->config = startup_config_new (file);
  if (!startup_config_parse_file (app->config))
    return FALSE;
  
  startup_config_set_group (app->config, "Startup Entry");
  return TRUE;
}

StartupConfig *
startup_app_get_config (StartupApp  *app)
{
  g_return_val_if_fail (app != NULL, NULL);

  return app->config;
}

gboolean
startup_app_initialize (StartupApp *app)
{
  osso_return_t ret;
  gchar *service, *path, *iface;
  
  g_return_val_if_fail (app != NULL, FALSE);
  g_return_val_if_fail (app->config != NULL, FALSE);
 
  if (app->osso)
    startup_app_deinitialize (app);
 
  /* Init osso */
  osso_log (LOG_INFO, "Initializing osso");
  
  app->osso = osso_initialize (
      startup_config_read_entry (app->config, "Name", NULL),
      startup_config_read_entry (app->config, "Version", NULL),
      TRUE, 
      NULL);
  
  if (app->osso == NULL) {
    osso_log (LOG_ERR, "Osso initialization failed");
    return FALSE;
  }
  
  /* Set topping callback */
  ret = osso_application_set_top_cb (
      app->osso, 
      startup_app_top_event_cb, 
      app);
  if (ret != OSSO_OK)
    osso_log (LOG_ERR, "Could not set topping callback");

  /* Set handling changes in HW states. */
  ret = osso_hw_set_event_cb (
      app->osso,
      NULL,
      startup_app_hw_event_cb,
      app);
  if (ret != OSSO_OK)
    osso_log (LOG_ERR, "Could not set callback for HW monitoring");

  app->service = startup_config_read_entry (app->config, "ServiceName", NULL),
  app->path = startup_config_read_entry (app->config, "PathName", NULL),
  app->iface = startup_config_read_entry (app->config, "InterfaceName", NULL),

  service = g_strdup_printf ("%s.startup", app->service);
  path = g_strdup_printf ("%s/startup", app->path);
  iface = g_strdup_printf ("%s.startup", app->iface);
  
  /* Set handling d-bus messages from session bus */
  ret = osso_rpc_set_cb_f (
      app->osso,
      service,
      path,
      iface,
      startup_app_rpc_event_cb,
      app);

  g_free (service);
  g_free (path);
  g_free (iface);
  
  if (ret != OSSO_OK)
    osso_log (LOG_ERR, "Could not set callback for receiving messages");
  
  return TRUE;
}

void
startup_app_deinitialize (StartupApp *app)
{
  gchar *service, *path, *iface;

  g_return_if_fail (app != NULL);
  g_return_if_fail (app->config != NULL);
  g_return_if_fail (app->osso != NULL);
  
  /* Unset callbacks */
  osso_application_unset_top_cb (
		  app->osso, 
		  startup_app_top_event_cb, 
		  NULL);
    
  osso_hw_unset_event_cb (
		  app->osso,
		  NULL);

  service = g_strdup_printf ("%s.startup", app->service);
  path = g_strdup_printf ("%s/startup", app->path);
  iface = g_strdup_printf ("%s.startup", app->iface);
 
  osso_rpc_unset_cb_f (
      app->osso,
      service,
      path,
      iface,
      startup_app_rpc_event_cb,
      NULL);

  g_free (service);
  g_free (path);
  g_free (iface);
  
  /* Deinit osso */
  osso_deinitialize (app->osso);
  app->osso = NULL;
}

static void 
startup_app_top_event_cb (const gchar *arguments, 
                    gpointer     data)
{
  StartupApp *app;
 
  g_return_if_fail (data != NULL);

  app = STARTUP_APP (data);

  if (app->top_cb)
    app->top_cb (app, app->top_cb_data);
}
 
static gint 
startup_app_rpc_event_cb (const gchar *interface, 
	  	          const gchar *method,
                          GArray      *args, 
                          gpointer     data, 
                          osso_rpc_t  *retval)
{
  StartupApp *app;
  guint oldstate, newstate;
 
  g_return_val_if_fail (data != NULL, OSSO_ERROR);
  g_return_val_if_fail (method != NULL, OSSO_ERROR);
  g_return_val_if_fail (retval != NULL, OSSO_ERROR);

  app = STARTUP_APP (data);
 
  osso_log (LOG_DEBUG, "Got DBUS method: %s\n", method);

  retval->type = DBUS_TYPE_BOOLEAN;
  retval->value.b = TRUE;
  
  /* Handle pause method */
  if (g_ascii_strcasecmp (method, GAME_PAUSE_METHOD) == 0) {
    newstate = GAME_PAUSED;
  }
  else if (g_ascii_strcasecmp (method, GAME_CLOSE_METHOD) == 0 ||
           g_ascii_strcasecmp (method, GAME_CRASHED_METHOD) == 0) {
    newstate = GAME_CLOSED;
  }
  else {
    osso_log (LOG_ERR, "Unknown DBUS method: %s\n", method);
    retval->value.b = FALSE;
    return OSSO_ERROR;
  }

  oldstate = app->state;
  app->state = newstate;
  if (app->state_change_cb)
    app->state_change_cb (app, oldstate, app->state, app->state_change_cb_data);

  return OSSO_OK;
}

/* Depending on the state of hw, do something */
static void 
startup_app_hw_event_cb (osso_hw_state_t *state, 
                         gpointer data)
{
  StartupApp *app;
 
  g_return_if_fail (data != NULL);

  app = STARTUP_APP (data);

  // TODO handle hw state 
}

gboolean 
startup_app_send_game_msg (StartupApp *app,
                           const guint msg)
{
  const gchar *method;
  osso_return_t ret;
  osso_rpc_t retval;
  guint oldstate;

  g_return_val_if_fail (msg < LAST_MESSAGE, FALSE);
  g_return_val_if_fail (app != NULL, FALSE);
  g_return_val_if_fail (app->config != NULL, FALSE);
  g_return_val_if_fail (app->osso != NULL, FALSE);
  
  method = method_strings [msg];
  
  osso_log (LOG_DEBUG, "Sendind msg: %s\n", method);
  osso_log (LOG_DEBUG, "Current state: %d\n", app->state);

  ret = osso_rpc_run (
      app->osso,
      app->service,
      app->path,
      app->iface,
      method,
      &retval,
      0);

  if (ret != OSSO_OK) {
    osso_log (LOG_DEBUG, "Error sendind msg: %s\n", method);
    return FALSE;
  }

  oldstate = app->state;
 
  switch (msg) {
    case GAME_RUN:
    case GAME_CONTINUE:
    case GAME_RESTART:
      app->state = GAME_RUNNING;
      break;
    case GAME_PAUSE:
      app->state = GAME_PAUSED;
      break;
    case GAME_CLOSE:
      app->state = GAME_CLOSED;
      break;
    default:
      break;
  }
 
  osso_log (LOG_DEBUG, "New state: %d\n", app->state);

  if (app->state_change_cb)
    app->state_change_cb (app, oldstate, app->state, app->state_change_cb_data);
  
  return TRUE;
}

guint
startup_app_get_game_state (StartupApp *app)
{
  g_return_val_if_fail (app != NULL, GAME_UNLOADED);

  return app->state;
}

void
startup_app_set_top_cb (StartupApp  *app,
                        startup_app_top_cb *top_cb,
                        gpointer     data)
{
  g_return_if_fail (app != NULL);
  app->top_cb = top_cb;
  app->top_cb_data = data;
}

void 
startup_app_set_state_change_cb (StartupApp  *app,
                                 startup_app_state_change_cb *state_change_cb,
                                 gpointer     data)
{
  g_return_if_fail (app != NULL);
  app->state_change_cb = state_change_cb;
  app->state_change_cb_data = data;
}
