/**
 * Retu/Vilma headset detection
 *
 * Copyright (C) 2006 Nokia Corporation
 *
 * Written by Juha Yrjl
 *
 * This file is subject to the terms and conditions of the GNU General
 * Public License. See the file "COPYING" in the main directory of this
 * archive for more details.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <linux/module.h>
#include <linux/init.h>
#include <linux/config.h>
#include <linux/kernel.h>
#include <linux/delay.h>
#include <linux/platform_device.h>

#include "retu.h"

#define RETU_ADC_CHANNEL_HOOKDET	0x05

static int retu_headset_initialized;
static int retu_headset_enabled;

static void retu_headset_set_bias(int enable)
{
	if (enable) {
		retu_set_reg_bits(RETU_REG_AUDTXR, (1 << 0) | (1 << 1));
		msleep(2);
		retu_set_reg_bits(RETU_REG_AUDTXR, 1 << 3);
	} else
		retu_clear_reg_bits(RETU_REG_AUDTXR,
				    (1 << 0) | (1 << 1) | (1 << 3));
}

static void retu_headset_enable(void)
{
	if (retu_headset_enabled)
		return;
	retu_headset_enabled = 1;

	retu_headset_set_bias(1);
}

static void retu_headset_disable(void)
{
	if (!retu_headset_enabled)
		return;
	retu_headset_enabled = 0;

	retu_headset_set_bias(0);
}

static ssize_t retu_headset_hookdet_show(struct device *dev,
					 struct device_attribute *attr,
					 char *buf)
{
	int val;

	val = retu_read_adc(RETU_ADC_CHANNEL_HOOKDET);
	return sprintf(buf, "%d\n", val);
}

static DEVICE_ATTR(hookdet, S_IRUGO, retu_headset_hookdet_show, NULL);

static ssize_t retu_headset_enable_show(struct device *dev,
					struct device_attribute *attr,
					char *buf)
{
	return sprintf(buf, "%u\n", retu_headset_enabled);
}

static ssize_t retu_headset_enable_store(struct device *dev,
					 struct device_attribute *attr,
					 const char *buf, size_t count)
{
	int enable;

	if (sscanf(buf, "%u", &enable) != 1)
		return -EINVAL;
	if (enable)
		retu_headset_enable();
	else
	        retu_headset_disable();
	return count;
}

static DEVICE_ATTR(enable, S_IRUGO | S_IWUSR | S_IWGRP,
		   retu_headset_enable_show, retu_headset_enable_store);

static int __init retu_headset_probe(struct platform_device *pdev)
{
	int r;

	retu_headset_initialized = 1;
	r = device_create_file(&pdev->dev, &dev_attr_hookdet);
	if (r < 0)
		return r;
	r = device_create_file(&pdev->dev, &dev_attr_enable);
	if (r < 0) {
		device_remove_file(&pdev->dev, &dev_attr_hookdet);
		return r;
	}

	return 0;
}

static int retu_headset_remove(struct platform_device *pdev)
{
	retu_headset_initialized = 0;
	device_remove_file(&pdev->dev, &dev_attr_hookdet);
	return 0;
}

static int retu_headset_suspend(struct platform_device *pdev, pm_message_t mesg)
{
	return 0;
}

static int retu_headset_resume(struct platform_device *pdev)
{
	return 0;
}


static struct platform_driver retu_headset_driver = {
	.probe		= retu_headset_probe,
	.remove		= retu_headset_remove,
	.suspend	= retu_headset_suspend,
	.resume		= retu_headset_resume,
	.driver		= {
		.name	= "retu-headset",
	},
};

static int __init retu_headset_init(void)
{
	int r;

	printk(KERN_INFO "Retu/Vilma headset driver initializing\n");

	r = platform_driver_register(&retu_headset_driver);
	if (r < 0)
		return r;

	return 0;
}

static void __exit retu_headset_exit(void)
{
	platform_driver_unregister(&retu_headset_driver);
}

module_init(retu_headset_init);
module_exit(retu_headset_exit);

MODULE_DESCRIPTION("Retu/Vilma headset detection");
MODULE_LICENSE("GPL");
MODULE_AUTHOR("Juha Yrjl");
