/*
 * linux/arch/arm/mach-omap1/board-nokia770-mmc.c
 *
 * Copyright (C) 2006 Nokia Corporation
 * Author: Juha Yrjl
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

#include <linux/config.h>
#include <linux/delay.h>

#include <asm/arch/mmc.h>
#include <asm/arch/gpio.h>
#include <asm/arch/board.h>

static int power_gpio;
static int switch_gpio;
static int wp_gpio;
static struct timer_list cover_debounce_timer;
static struct device *mmc_device;

static int nokia770_mmc_set_power(struct device *dev, int slot, int power_on,
				  int vdd)
{
#ifdef CONFIG_MMC_DEBUG
	printk("MMC: Set slot %d power: %s (vdd %d)\n", slot + 1,
	       power_on ? "on" : "off", vdd);
#endif
	BUG_ON(slot != 0);
	omap_set_gpio_dataout(power_gpio, power_on);
	if (power_on)
		msleep(10);
	return 0;
}

static int nokia770_mmc_get_ro(struct device *dev, int slot)
{
	int ro;

	BUG_ON(slot != 0);
	ro = omap_get_gpio_datain(wp_gpio);
#ifdef CONFIG_MMC_DEBUG
	printk("MMC: Get RO slot %d: %s\n", slot, ro ?
	       "read-only" : "read-write");
#endif
	return ro;
}

#undef  FUNC_MUX_CTRL_8
#define FUNC_MUX_CTRL_8		0xfffe1024

#define W19_GPIO		48

static int nokia770_mmc_set_bus_mode(struct device *dev, int slot, int mode)
{
	u32 l;

#ifdef CONFIG_MMC_DEBUG
	printk("MMC: Set bus mode: %s\n", mode == MMC_BUSMODE_OPENDRAIN ?
	       "open-drain" : "push-pull");
#endif
	if (mode == MMC_BUSMODE_OPENDRAIN) {
		/* Really nasty hack to work around a bug in OMAP (?) that
		 * results in DATDIR1 not being correct at init time.
		 * We multiplex it as GPIO to force it high. */
		omap_set_gpio_dataout(W19_GPIO, 1);
		omap_set_gpio_direction(W19_GPIO, 0);
		l = omap_readl(FUNC_MUX_CTRL_8);
		l |= 7 << 15;
		omap_writel(l, FUNC_MUX_CTRL_8);
	} else {
		omap_set_gpio_direction(W19_GPIO, 1);
		l = omap_readl(FUNC_MUX_CTRL_8);
		l &= ~(7 << 15);
		l |= 6 << 15;
		omap_writel(l, FUNC_MUX_CTRL_8);
	}
	return 0;
}

static int nokia770_mmc_get_cover_state(struct device *dev, int slot)
{
	int is_closed;

	BUG_ON(slot != 0);
	is_closed = !omap_get_gpio_datain(switch_gpio);
	return is_closed;
}

static void nokia770_mmc_cover_timer(unsigned long arg)
{
	int is_closed;

	if (mmc_device == NULL)
		return;

	is_closed = !omap_get_gpio_datain(switch_gpio);
	if (is_closed)
		omap_mmc_notify_cover_event(mmc_device, 0, is_closed);
}

static irqreturn_t nokia770_mmc_cover_handler(int irq, void *dev_id,
					      struct pt_regs *regs)
{
	int is_closed;

	is_closed = !omap_get_gpio_datain(switch_gpio);

	if (mmc_device == NULL)
		return IRQ_HANDLED;

	/* When the cover is being opened, we want to be notified ASAP.
	 * If it is being closed, we debounce. */
	if (!is_closed) {
		del_timer_sync(&cover_debounce_timer);
		omap_mmc_notify_cover_event(mmc_device, 0, 0);
	} else
		mod_timer(&cover_debounce_timer, jiffies + msecs_to_jiffies(50));
	return IRQ_HANDLED;
}

static int nokia770_mmc_late_init(struct device *dev)
{
	mmc_device = dev;
	return 0;
}

static void nokia770_mmc_cleanup(struct device *dev)
{
	del_timer_sync(&cover_debounce_timer);
	mmc_device = NULL;
}

static struct omap_mmc_platform_data nokia770_mmc_data = {
	.enabled		= 1,
	.nr_slots		= 1,
	.wire4			= 0,
	.max_freq		= 12000000,
	.init			= nokia770_mmc_late_init,
	.cleanup		= nokia770_mmc_cleanup,
	.slots[0] = {
		.set_power	= nokia770_mmc_set_power,
		.get_cover_state= nokia770_mmc_get_cover_state,
		.ocr_mask	= MMC_VDD_32_33 | MMC_VDD_33_34,
		.name		= "external",
	},
};


void __init nokia770_mmc_init(void)
{
	const struct omap_mmc_config *config;
	const struct omap_mmc_conf *conf;
	int r;

	config = omap_get_config(OMAP_TAG_MMC, struct omap_mmc_config);
	if (config == NULL) {
		printk(KERN_ERR "MMC config not found\n");
		return;
	}

	conf = &config->mmc[1];

	if (!conf->enabled)
		return;

	if (conf->wire4) {
		nokia770_mmc_data.wire4 = 1;
		wp_gpio = conf->wp_pin;
		if (wp_gpio >= 0) {
			if (omap_request_gpio(wp_gpio) < 0)
				BUG();
			omap_set_gpio_direction(wp_gpio, 1);
			nokia770_mmc_data.slots[0].get_ro = nokia770_mmc_get_ro;
		}
		nokia770_mmc_data.slots[0].set_bus_mode =
			nokia770_mmc_set_bus_mode;
		if (omap_request_gpio(W19_GPIO) < 0)
			goto err;
		/* On this HW revision we can use a higher frequency */
		nokia770_mmc_data.max_freq = 24000000;
	} else
		wp_gpio = -1;

	power_gpio = conf->power_pin;
	if (omap_request_gpio(power_gpio) < 0)
		goto err;
	omap_set_gpio_dataout(power_gpio, 0);
	omap_set_gpio_direction(power_gpio, 0);

	switch_gpio = conf->switch_pin;
	if (omap_request_gpio(switch_gpio) < 0)
		goto err;
	omap_set_gpio_direction(switch_gpio, 1);
	setup_timer(&cover_debounce_timer, nokia770_mmc_cover_timer, 0);
	r = request_irq(OMAP_GPIO_IRQ(switch_gpio), nokia770_mmc_cover_handler,
			SA_TRIGGER_RISING | SA_TRIGGER_FALLING,
			"MMC cover", NULL);
	if (r < 0)
		goto err;

	omap_set_mmc_info(2, &nokia770_mmc_data);

	return;
err:
	printk(KERN_ERR "Unable to get MMC resources\n");
}
