/**
  @file inquiry.c

  @author Johan Hedberg <johan.hedberg@nokia.com>

  Copyright (C) 2006 Nokia Corporation. All rights reserved.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License, version 2, as
  published by the Free Software Foundation.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA

*/  
#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <unistd.h>
#include <errno.h>
#include <termios.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <netinet/in.h>

#include <bluetooth/bluetooth.h>
#include <bluetooth/hci.h>
#include <bluetooth/hci_lib.h>

#include <glib.h>
#include <glib-object.h>

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include "log.h"
#include "dbus.h"
#include "devlist.h"
#include "inquiry.h"

/* hci0 */
#define DEV_NUMBER 0

struct remote_dev {
    bdaddr_t  bda;
    uint8_t   cls[3];
    uint8_t   rssi;
    uint16_t  clock;
    gboolean  resolved;
};

static GSList *dev_list = NULL;

static int new_devs = 0;
static int inquiry_times = INQUIRY_TIMES;
static gboolean name_req_in_progress = FALSE;
static bdaddr_t name_req_bda;

static int hci_sock = -1;

/* So we don't send cancel_inquiry when not necessary */
static gboolean inquiry_in_progress = FALSE;

/* Originally from hcidump */
static int event_socket(int *events)
{
    struct sockaddr_hci addr;
    struct hci_filter flt;
    int sock, i;

    /* Create HCI socket */
    sock = socket(AF_BLUETOOTH, SOCK_RAW, BTPROTO_HCI);
    if (sock < 0) {
        error("Can't create HCI socket: %s", g_strerror(errno));
    }

    /* Setup filter */
    hci_filter_clear(&flt);
    hci_filter_set_ptype(HCI_EVENT_PKT, &flt);
    for (i = 0; events[i] != 0; i++)
        hci_filter_set_event(events[i], &flt);
    if (setsockopt(sock, SOL_HCI, HCI_FILTER, &flt, sizeof(flt)) < 0)
        error("Can't set HCI filter: %s", g_strerror(errno));

    /* Bind socket to the HCI device */
    addr.hci_family = AF_BLUETOOTH;
    addr.hci_dev = DEV_NUMBER;
    if (bind(sock, (struct sockaddr *)&addr, sizeof(addr)) < 0)
        error("Can't attach to device hci%d. %s(%d)",
                DEV_NUMBER, g_strerror(errno), errno);

    return sock;
}

static int send_read_remote_name_cmd(int sock, bdaddr_t ba, uint16_t clk)
{
    int ret;
    remote_name_req_cp cp;

    memset(&cp, 0, sizeof(cp));

    cp.bdaddr         = ba;
    cp.pscan_rep_mode = 0x01;
    cp.clock_offset   = clk;

    ret = hci_send_cmd(sock, OGF_LINK_CTL, OCF_REMOTE_NAME_REQ,
                       REMOTE_NAME_REQ_CP_SIZE, &cp);

    if (ret == 0) {
        name_req_in_progress = TRUE;
        name_req_bda = ba;
    }

    return ret;
}

static int send_inquiry_cmd(int sock)
{
    int ret;
    inquiry_cp inq;

    if (inquiry_in_progress)
        return 0;

    inq.num_rsp = 100;
    inq.length  = INQUIRY_LENGTH + ((INQUIRY_TIMES - inquiry_times) * 4);

    /* General/Unlimited Inquiry Access Code (GIAC) */
    inq.lap[0] = 0x33;
    inq.lap[1] = 0x8b;
    inq.lap[2] = 0x9e;
    
    ret = hci_send_cmd(sock, OGF_LINK_CTL, OCF_INQUIRY,
                       INQUIRY_CP_SIZE , &inq);
    inquiry_in_progress = TRUE;

    return ret;
}

static int send_inquiry_cancel_cmd(int sock)
{
    int ret;

    if (!inquiry_in_progress) {
        debug("Not sending unecessary inquiry cancel cmd");
        return 0;
    }

    ret = hci_send_cmd(sock, OGF_LINK_CTL, OCF_INQUIRY_CANCEL, 0 , NULL);
    inquiry_in_progress = FALSE;

    return ret;
}

static int send_name_request_cancel_cmd(int sock)
{
    int ret;

    if (!name_req_in_progress) {
        debug("Not sending unecessary name request cancel cmd");
        return 0;
    }

    ret = hci_send_cmd(sock, OGF_LINK_CTL, OCF_REMOTE_NAME_REQ_CANCEL,
                       sizeof(bdaddr_t), &name_req_bda);
    inquiry_in_progress = FALSE;

    return ret;
}


static gint cmp_bda(struct remote_dev *dev, bdaddr_t *bda)
{
    return bacmp(&dev->bda, bda);
}

static gint cmp_resolved(struct remote_dev *dev, gconstpointer data)
{
    if (dev->resolved)
        return 1;
    else
        return 0;
}

static struct remote_dev *get_dev(const bdaddr_t *bda)
{
    GSList *entry;

    if (bda)
        entry = g_slist_find_custom(dev_list, bda, (GCompareFunc)cmp_bda);
    else
        entry = g_slist_find_custom(dev_list, NULL, (GCompareFunc)cmp_resolved);

    if (entry == NULL)
        return NULL;

    return entry->data;
}

static void add_dev(bdaddr_t bda, uint8_t cls[3], uint8_t rssi, uint16_t clk)
{
    struct remote_dev *dev;
   
    /* Don't add if device is already in list */
    if (get_dev(&bda) != NULL)
        return;

    dev  = g_new0(struct remote_dev, 1);
    dev->bda        = bda;
    dev->rssi       = rssi;
    dev->clock      = clk;
    dev->resolved   = FALSE;

    memcpy(dev->cls, cls, 3);

    dev_list = g_slist_append(dev_list, dev);
    new_devs++;
}

static void publish_dev(struct remote_dev *dev, const char *name)
{
    char bda[18];
    DevInfo info;

    ba2str(&(dev->bda), bda);

    info.bda   = bda;
    info.name  = name;
    info.rssi  = dev->rssi;
    info.clock = dev->clock;
    memcpy(info.cls, dev->cls, 3);

    if (!send_dbus_dev_found(get_dbus_connection(), &info))
        error("sending D-BUS signal failed!");

    dev->resolved = TRUE;

    new_devs--;
}

static void process_queue(int sock)
{
    struct remote_dev *dev = get_dev(NULL);

    g_assert(dev != NULL);
    if (!dev->resolved)
        send_read_remote_name_cmd(sock, dev->bda, dev->clock);
}

static int handle_name_req_complete(char *data, int sock)
{
    char name[248];
    struct remote_dev *dev;
    evt_remote_name_req_complete *ev = (evt_remote_name_req_complete *)data;

    dev = get_dev(&ev->bdaddr);
    if (dev == NULL) {
        debug("Name req. complete for BDA not in queue!");
        return 0;
    }

    name_req_in_progress = FALSE;

    if (ev->status != 0x00) {
        debug("Get remote name failed with 0x%02x", ev->status);
        name[0] = '\0';
    }
    else {
        char *end;
        ev->name[247] = '\0';
        strncpy(name, (char*)ev->name, sizeof(name));
        if(!g_utf8_validate(name, -1, &end))
            *end = '\0';
    }

    publish_dev(dev, name);

    if (new_devs > 0)
        process_queue(sock);
    else if (!inquiry_in_progress) {
        if ((--inquiry_times) > 0) {
            if (send_inquiry_cmd(hci_sock) < 0) {
                error("Sending inquiry failed\n");
                stop_search();
            }
            return 0;
        }
        else
            return 1;
    }

    return 0;
}

static int handle_inquiry_result_with_rssi(char *data, int sock)
{
    int i;
    uint8_t num_resp = data[0];
    char   *ptr      = &(data[1]);

    for (i = 0; i < (int)num_resp; i++) {
        inquiry_info_with_rssi *info = (inquiry_info_with_rssi *)ptr;

        ptr += INQUIRY_INFO_WITH_RSSI_SIZE;

        /* Ignore duplicates */
        if (get_dev(&info->bdaddr))
            continue;

        add_dev(info->bdaddr, info->dev_class, info->rssi, info->clock_offset);
    }

    return 0;
}

static int handle_inquiry_result(char *data, int sock)
{
    int i;
    uint8_t num_resp = data[0];
    char   *ptr      = &(data[1]);

    for (i = 0; i < (int)num_resp; i++) {
        inquiry_info *info = (inquiry_info *)ptr;

        ptr += INQUIRY_INFO_SIZE;

        /* Ignore duplicates */
        if (get_dev(&info->bdaddr))
            continue;

        add_dev(info->bdaddr, info->dev_class, 0x00, info->clock_offset);
    }

    return 0;
}

/* Return 1 when finished */
static int process_data(char *data, int data_len, int sock)
{
    uint8_t type = data[0];

    if (type == HCI_EVENT_PKT) {
        hci_event_hdr *hdr = (hci_event_hdr *) &data[1];
        char *body = &data[HCI_EVENT_HDR_SIZE + 1];

        switch (hdr->evt) {
            case EVT_INQUIRY_RESULT:
                return handle_inquiry_result(body, sock);

            case EVT_INQUIRY_RESULT_WITH_RSSI:
                return handle_inquiry_result_with_rssi(body, sock);

            case EVT_INQUIRY_COMPLETE:
                inquiry_in_progress = FALSE;

                if (body[0] != 0x00) 
                    error("Inquiry completed with error: 0x%02x\n",
                            body[0]);
                if (new_devs > 0) {
                    /*(void) g_timeout_add(TIMEOUT, timeout_cb, NULL);*/
                    if (!name_req_in_progress)
                        process_queue(sock);
                    return 0;
                }
                else {
                    if ((--inquiry_times) > 0) {
                        if (send_inquiry_cmd(hci_sock) < 0) {
                            error("Sending inquiry failed");
                            stop_search();
                        }
                        return 0;
                    }
                    else
                        return 1;
                }

                return 1;

            case EVT_REMOTE_NAME_REQ_COMPLETE:
                return handle_name_req_complete(body, sock);

            case EVT_CMD_STATUS:
                {
                    evt_cmd_status *ev = (evt_cmd_status *)body;
                    if (ev->status != 0x00) {
                        debug("HCI Command 0x%04X failed with 0x%02x",
                                ev->opcode, ev->status);
                        if (inquiry_in_progress && ev->opcode
                                == htobs(cmd_opcode_pack(OGF_LINK_CTL, OCF_INQUIRY)))
                            return 1;
                        if (name_req_in_progress && ev->opcode
                                == htobs(cmd_opcode_pack(OGF_LINK_CTL, OCF_REMOTE_NAME_REQ)))
                            return 1;
                    }
                    return 0;
                }

            case EVT_STACK_INTERNAL:
                {
                    evt_stack_internal *si = (evt_stack_internal *) body;
                    evt_si_device *sd = (evt_si_device *) &si->data;

                    if (sd->event == HCI_DEV_DOWN) {
                        inquiry_in_progress = FALSE;
                        name_req_in_progress = FALSE;
                        return 1;
                    }
                }
                
                return 0;

            default:
                debug("Invalid event type (0x%02x)", hdr->evt);
                return 1;
        }
    }
    else {
        debug("Invalid packet type (0x%02x)", type);
        return 1;
    }
}

static gboolean hci_cb(GIOChannel *chan, GIOCondition cond, gpointer user_data)
{
    char data[HCI_MAX_EVENT_SIZE];
    int data_len, ret, sock;

    if (cond != G_IO_IN) {
        error("Error on inquiry HCI socket");
        stop_search();
        g_io_channel_unref(chan);
        return FALSE;
    }

    sock = g_io_channel_unix_get_fd(chan);

    data_len = read(sock, data, sizeof(data));
    if (data_len < 0) {
        error("read(hci_sock): %s", g_strerror(errno));
        stop_search();
        g_io_channel_unref(chan);
        return FALSE;
    }

    ret = process_data(data, data_len, sock);

    if (ret != 0) {
        stop_search();
        g_io_channel_unref(chan);
        return FALSE;
    }

    return TRUE;
}

void inquiry_stop(void)
{
    if (hci_sock < 0)
        return;

    send_inquiry_cancel_cmd(hci_sock);
    send_name_request_cancel_cmd(hci_sock);

    close(hci_sock);
    hci_sock = -1;
}

gboolean inquiry_start(void)
{
    GIOChannel *gio;
    static guint hci_id;
    int hci_events[]  = { EVT_INQUIRY_RESULT,
                          EVT_INQUIRY_RESULT_WITH_RSSI,
                          EVT_INQUIRY_COMPLETE,
                          EVT_CMD_STATUS,
                          EVT_REMOTE_NAME_REQ_COMPLETE,
                          EVT_STACK_INTERNAL,
                          0 };

    hci_sock = event_socket(hci_events);
    if (hci_sock < 0) {
        error("Opening raw HCI socket failed");
        return FALSE;
    }

    if (send_inquiry_cmd(hci_sock) < 0) {
        error("Sending inquiry command failed");
        close(hci_sock);
        return FALSE;
    }

    gio = g_io_channel_unix_new(hci_sock);
    hci_id = g_io_add_watch(gio, G_IO_IN | G_IO_ERR | G_IO_HUP | G_IO_NVAL,
            hci_cb, &hci_id);

    return TRUE;
}

