/*
 * Galago Service API
 *
 * Copyright (C) 2004-2006 Christian Hammond
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#include <libgalago/galago-private.h>
#include <libgalago/galago-service.h>
#include <libgalago/galago-assert.h>
#include <libgalago/galago-context.h>
#include <libgalago/galago-context-priv.h>
#include <libgalago/galago-core.h>
#include <libgalago/galago-enum-types.h>
#include <libgalago/galago-marshal.h>
#include <stdio.h>
#include <string.h>

struct _GalagoServicePrivate
{
	GalagoServiceFlags flags;

	char *id;
	char *name;

	GHashTable *accounts_table;
	GList *accounts;
	GList *registered_statuses;
};

enum
{
	PROP_0,
	PROP_FLAGS,
	PROP_ID,
	PROP_NAME
};

enum
{
	ACCOUNT_ADDED,
	ACCOUNT_REMOVED,
	LAST_SIGNAL
};

static GalagoService *_galago_create_service_common(const char *id,
	const char *name, GalagoOrigin origin, const char *obj_path,
	GalagoServiceFlags flags);
static void _galago_dbus_service_add_account(GalagoService *service,
											 GalagoAccount *account);

static GHashTable *service_id_map_table = NULL;

/**************************************************************************
 * Object/Class support
 **************************************************************************/
static void galago_service_destroy(GalagoObject *gobject);
static void galago_service_dbus_message_append(DBusMessageIter *iter,
											   const GalagoObject *object);
static void *galago_service_dbus_message_get(DBusMessageIter *iter);
static void galago_service_dbus_push_full(GalagoObject *object);
static gchar *galago_service_dbus_get_signature(void);
static void galago_service_set_property(GObject *object, guint prop_id,
										const GValue *value, GParamSpec *pspec);
static void galago_service_get_property(GObject *object, guint prop_id,
										GValue *value, GParamSpec *pspec);

static GalagoObjectClass *parent_class = NULL;
static guint signals[LAST_SIGNAL] = {0};

G_DEFINE_TYPE(GalagoService, galago_service, GALAGO_TYPE_OBJECT);

static void
galago_service_class_init(GalagoServiceClass *klass)
{
	GalagoObjectClass *object_class  = GALAGO_OBJECT_CLASS(klass);
	GObjectClass      *gobject_class = G_OBJECT_CLASS(klass);

	parent_class = g_type_class_peek_parent(klass);

	object_class->dbus_interface = GALAGO_DBUS_SERVICE_INTERFACE;

	object_class->destroy             = galago_service_destroy;
	object_class->dbus_message_append = galago_service_dbus_message_append;
	object_class->dbus_message_get    = galago_service_dbus_message_get;
	object_class->dbus_push_full      = galago_service_dbus_push_full;
	object_class->dbus_get_signature  = galago_service_dbus_get_signature;

	gobject_class->set_property = galago_service_set_property;
	gobject_class->get_property = galago_service_get_property;

	/**
	 * GalagoService::account-added:
	 * @service: The object which received the signal.
	 * @account: The account that was added.
	 *
	 * Emitted when an account is added to the service.
	 */
	signals[ACCOUNT_ADDED] =
		g_signal_new("account-added",
					 G_TYPE_FROM_CLASS(klass),
					 G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
					 G_STRUCT_OFFSET(GalagoServiceClass, account_added),
					 NULL, NULL,
					 g_cclosure_marshal_VOID__POINTER,
					 G_TYPE_NONE, 1,
					 G_TYPE_POINTER);

	/**
	 * GalagoService::account-removed:
	 * @service: The object which received the signal.
	 * @account: The account that was removed.
	 *
	 * Emitted when an account is removed to the service.
	 */
	signals[ACCOUNT_REMOVED] =
		g_signal_new("account-removed",
					 G_TYPE_FROM_CLASS(klass),
					 G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
					 G_STRUCT_OFFSET(GalagoServiceClass, account_removed),
					 NULL, NULL,
					 g_cclosure_marshal_VOID__POINTER,
					 G_TYPE_NONE, 1,
					 G_TYPE_POINTER);

	g_object_class_install_property(gobject_class, PROP_FLAGS,
		g_param_spec_flags("flags", "Flags",
						   "The service flags",
						   GALAGO_TYPE_SERVICE_FLAGS, 0,
						   G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY |
						   G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |
						   G_PARAM_STATIC_BLURB));

	g_object_class_install_property(gobject_class, PROP_ID,
		g_param_spec_string("id", "ID",
						   "The service's unique ID",
						   NULL,
						   G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY |
						   G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |
						   G_PARAM_STATIC_BLURB));

	g_object_class_install_property(gobject_class, PROP_NAME,
		g_param_spec_string("name", "Name",
						   "The service's name",
						   NULL,
						   G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY |
						   G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |
						   G_PARAM_STATIC_BLURB));
}

static void
galago_service_init(GalagoService *service)
{
	service->priv = g_new0(GalagoServicePrivate, 1);

	service->priv->accounts_table =
		g_hash_table_new_full(g_str_hash, g_str_equal, g_free, NULL);
}

static void
galago_service_destroy(GalagoObject *object)
{
	GalagoService *service = GALAGO_SERVICE(object);

	if (service->priv != NULL)
	{
		if (service->priv->accounts != NULL)
		{
			g_list_foreach(service->priv->accounts,
						   (GFunc)galago_object_destroy, NULL);
			g_list_free(service->priv->accounts);
		}

		g_hash_table_destroy(service->priv->accounts_table);

		galago_context_push(galago_object_get_context(object));
		galago_context_remove_service(service);
		galago_context_pop();

		g_free(service->priv->id);
		g_free(service->priv->name);
		g_free(service->priv);
		service->priv = NULL;
	}

	if (GALAGO_OBJECT_CLASS(parent_class)->destroy != NULL)
		GALAGO_OBJECT_CLASS(parent_class)->destroy(object);
}

static void
galago_service_dbus_message_append(DBusMessageIter *iter,
								   const GalagoObject *object)
{
	GalagoService *service = (GalagoService *)object;
	const char *obj_path, *id, *name;
	dbus_uint32_t flags;

	obj_path = galago_object_get_dbus_path(GALAGO_OBJECT(service));
	id       = galago_service_get_id(service);
	name     = galago_service_get_name(service);
	flags    = galago_service_get_flags(service);

	galago_dbus_message_iter_append_string_or_nil(iter, obj_path);
	dbus_message_iter_append_basic(iter, DBUS_TYPE_STRING, &id);
	dbus_message_iter_append_basic(iter, DBUS_TYPE_STRING, &name);
	dbus_message_iter_append_basic(iter, DBUS_TYPE_UINT32, &flags);
}

static void *
galago_service_dbus_message_get(DBusMessageIter *iter)
{
	GalagoService *service;
	const char *obj_path, *id, *name;
	dbus_uint32_t flags;

	obj_path = galago_dbus_message_iter_get_string_or_nil(iter);
	dbus_message_iter_next(iter);

	dbus_message_iter_get_basic(iter, &id);
	dbus_message_iter_next(iter);

	dbus_message_iter_get_basic(iter, &name);
	dbus_message_iter_next(iter);

	dbus_message_iter_get_basic(iter, &flags);

	service = _galago_create_service_common(id, name, GALAGO_REMOTE,
											obj_path,
											(GalagoServiceFlags)flags);

	return service;
}

static void
galago_service_dbus_push_full(GalagoObject *object)
{
	GalagoService *service = GALAGO_SERVICE(object);
	GList *l;

	for (l = galago_service_get_accounts(service, FALSE);
		 l != NULL;
		 l = l->next)
	{
		_galago_dbus_service_add_account(service, GALAGO_ACCOUNT(l->data));
	}

	/* We must go through it a second time. */
	for (l = galago_service_get_accounts(service, FALSE);
		 l != NULL;
		 l = l->next)
	{
		galago_dbus_object_push_full(GALAGO_OBJECT(l->data));
	}
}

static gchar *
galago_service_dbus_get_signature(void)
{
	return g_strdup(DBUS_TYPE_STRING_AS_STRING   // object path
					DBUS_TYPE_STRING_AS_STRING   // id
					DBUS_TYPE_STRING_AS_STRING   // name
					DBUS_TYPE_UINT32_AS_STRING); // flags
}

static void
galago_service_set_property(GObject *object, guint prop_id,
							const GValue *value, GParamSpec *pspec)
{
	GalagoService *service = GALAGO_SERVICE(object);

	switch (prop_id)
	{
		case PROP_FLAGS:
			service->priv->flags = g_value_get_flags(value);
			g_object_notify(G_OBJECT(service), "flags");
			break;

		case PROP_ID:
			service->priv->id = g_value_dup_string(value);
			g_object_notify(G_OBJECT(service), "id");
			break;

		case PROP_NAME:
			service->priv->name = g_value_dup_string(value);
			g_object_notify(G_OBJECT(service), "name");
			break;

		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}

static void
galago_service_get_property(GObject *object, guint prop_id,
							GValue *value, GParamSpec *pspec)
{
	GalagoService *service = GALAGO_SERVICE(object);

	switch (prop_id)
	{
		case PROP_FLAGS:
			g_value_set_flags(value, galago_service_get_flags(service));
			break;

		case PROP_ID:
			g_value_set_string(value, galago_service_get_id(service));
			break;

		case PROP_NAME:
			g_value_set_string(value, galago_service_get_name(service));
			break;

		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}


/**************************************************************************
 * GalagoService API
 **************************************************************************/
typedef struct
{
	char *name;
	GalagoServiceFlags flags;

} ServiceMapInfo;

#define ADD_SERVICE_MAP(id, _name, _flags) \
	map_info = g_new0(ServiceMapInfo, 1); \
	map_info->name  = g_strdup((_name)); \
	map_info->flags = (_flags); \
	g_hash_table_insert(service_id_map_table, g_strdup(id), map_info)

static void
destroy_map_info(ServiceMapInfo *map_info)
{
	g_free(map_info->name);
	g_free(map_info);
}

static gpointer
_init_service_id_map_table(gpointer unused)
{
	ServiceMapInfo *map_info;

	service_id_map_table =
		g_hash_table_new_full(g_str_hash, g_str_equal,
							  g_free, (GFreeFunc)destroy_map_info);

	ADD_SERVICE_MAP(GALAGO_SERVICE_ID_AIM,       _("AOL Instant Messenger"), 0);
	ADD_SERVICE_MAP(GALAGO_SERVICE_ID_GADUGADU,  _("Gadu-Gadu"),
					GALAGO_PRESERVE_SPACES);
	ADD_SERVICE_MAP(GALAGO_SERVICE_ID_GROUPWISE, _("Novell GroupWise"), 0);
	ADD_SERVICE_MAP(GALAGO_SERVICE_ID_ICQ,       _("ICQ"),
					GALAGO_PRESERVE_SPACES);
	ADD_SERVICE_MAP(GALAGO_SERVICE_ID_IRC,       _("Internet Relay Chat"),
					GALAGO_PRESERVE_SPACES);
	ADD_SERVICE_MAP(GALAGO_SERVICE_ID_JABBER,    _("Jabber"),
					GALAGO_PRESERVE_SPACES | GALAGO_STRIP_SLASH);
	ADD_SERVICE_MAP(GALAGO_SERVICE_ID_MSN,       _("MSN Messenger"),
					GALAGO_PRESERVE_SPACES);
	ADD_SERVICE_MAP(GALAGO_SERVICE_ID_NAPSTER,   _("Napster"), 0);
	ADD_SERVICE_MAP(GALAGO_SERVICE_ID_SILC,      _("SILC"), 0);
	ADD_SERVICE_MAP(GALAGO_SERVICE_ID_TREPIA,    _("Trepia"), 0);
	ADD_SERVICE_MAP(GALAGO_SERVICE_ID_YAHOO,     _("Yahoo! Messenger"),
					GALAGO_PRESERVE_SPACES);
	ADD_SERVICE_MAP(GALAGO_SERVICE_ID_ZEPHYR,    _("Zephyr"),
					GALAGO_PRESERVE_SPACES | GALAGO_PRESERVE_CASE);

	return NULL;
}

static ServiceMapInfo *
_galago_services_map_id_to_info(const char *id)
{
	static GOnce map_init = G_ONCE_INIT;
	ServiceMapInfo *map_info;
	char *temp;

	g_return_val_if_fail(id  != NULL, NULL);
	g_return_val_if_fail(*id != '\0', NULL);

	g_once(&map_init, _init_service_id_map_table, NULL);

	temp = g_ascii_strdown(id, -1);

	map_info = (ServiceMapInfo *)g_hash_table_lookup(service_id_map_table,
													 temp);
	g_free(temp);

	return map_info;
}

static GalagoService *
_galago_create_service_common(const char *id, const char *name,
							  GalagoOrigin origin, const char *obj_path,
							  GalagoServiceFlags flags)
{
	ServiceMapInfo *map_info;
	GalagoService *service;

	g_return_val_if_fail(galago_is_initted(), NULL);
	g_return_val_if_fail(id  != NULL,         NULL);
	g_return_val_if_fail(*id != '\0',         NULL);
	g_return_val_if_fail(obj_path == NULL || origin == GALAGO_REMOTE, NULL);

	if ((map_info = _galago_services_map_id_to_info(id)) != NULL)
	{
		name  = map_info->name;
		flags = map_info->flags;
	}

	g_return_val_if_fail(name  != NULL, NULL);
	g_return_val_if_fail(*name != '\0', NULL);

	service = galago_context_get_service(id, origin);

	if (service == NULL)
	{
		service = g_object_new(GALAGO_TYPE_SERVICE,
							   "id", id,
							   "name", name,
							   "flags", flags,
							   "origin", origin,
							   NULL);

		/* Set the object path, if specified. */
		if (origin == GALAGO_REMOTE)
			galago_object_set_dbus_path(GALAGO_OBJECT(service), obj_path);

		galago_context_add_service(service);
	}

	return service;
}

/**
 * galago_create_service
 * @id:    The service ID.
 * @name:  The service name.
 * @flags: The flags.
 *
 * Creates a service and adds it to the local cache.
 *
 * The name and flags will be computed if the ID is understood internally.
 * Otherwise, they will have to be provided.
 *
 * Returns: The new service, or a pointer to an existing one of the same
 *          ID.
 */
GalagoService *
galago_create_service(const char *id, const char *name,
					  GalagoServiceFlags flags)
{
	g_return_val_if_fail(galago_is_initted(), NULL);
	g_return_val_if_fail(id != NULL && *id != '\0', NULL);

	return _galago_create_service_common(id, name, GALAGO_LOCAL, NULL, flags);
}

/**
 * galago_service_get_id
 * @service: The service.
 *
 * Returns a service's ID.
 *
 * Returns: The service's ID.
 */
const char *
galago_service_get_id(const GalagoService *service)
{
	g_return_val_if_fail(service != NULL,            NULL);
	g_return_val_if_fail(GALAGO_IS_SERVICE(service), NULL);

	return service->priv->id;
}

/**
 * galago_service_get_name
 * @service: The service.
 *
 * Returns a service's name.
 *
 * Returns: The service's name.
 */
const char *
galago_service_get_name(const GalagoService *service)
{
	g_return_val_if_fail(service != NULL,            NULL);
	g_return_val_if_fail(GALAGO_IS_SERVICE(service), NULL);

	return service->priv->name;
}

/**
 * galago_service_register_status:
 * @service: The service.
 * @status: The status type to register.
 *
 * Registers a type of status that the accounts belonging to this service can
 * have set. The service will take ownership of @status.
 *
 * This can only be called by feeds. It will return with a warning if called
 * by a non-feed.
 */
void
galago_service_register_status(GalagoService *service, GalagoStatus *status)
{
	g_return_if_fail(service != NULL);
	g_return_if_fail(GALAGO_IS_SERVICE(service));
	g_return_if_fail(status != NULL);
	g_return_if_fail(GALAGO_IS_STATUS(status));
	g_return_if_fail(GALAGO_OBJECT_IS_LOCAL(service));
	g_return_if_fail(galago_is_feed());

#if 0
	/* TODO: Check that this isn't already in the list. */
	service->priv->registered_statuses =
		g_list_append(service->priv->registered_statuses, status);
#endif

	if (galago_is_connected() && !galago_is_daemon())
	{
		/*
		 * TODO: We'll need to make sure we re-register upon reconnect.
		 *       That should just involve storing these statuses and
		 *       re-adding.
		 */
		galago_dbus_send_message(GALAGO_OBJECT(service), "RegisterStatus",
			galago_value_new_object(GALAGO_TYPE_STATUS, G_OBJECT(status)),
			NULL);
		g_object_unref(G_OBJECT(status));
	}
}

/**
 * galago_service_get_registered_statuses:
 * @service: The service.
 *
 * Returns a list of statuses that accounts belonging to this service can
 * have set.
 */
GList *
galago_service_get_registered_statuses(const GalagoService *service)
{
	g_return_val_if_fail(service != NULL,            NULL);
	g_return_val_if_fail(GALAGO_IS_SERVICE(service), NULL);

	if (!galago_is_daemon() && galago_is_connected())
	{
		/* TODO: Should store these and free them later. */
		return galago_dbus_send_message_with_reply(
			GALAGO_OBJECT(service), "GetRegisteredStatuses",
			galago_value_new_list(GALAGO_VALUE_TYPE_OBJECT, NULL,
								  (gpointer)GALAGO_TYPE_STATUS),
			NULL);
	}

	return NULL; // service->priv->registered_statuses
}

/**
 * galago_service_get_flags
 * @service: The service.
 *
 * Returns a service's flags.
 *
 * Returns: The flags.
 */
GalagoServiceFlags
galago_service_get_flags(const GalagoService *service)
{
	g_return_val_if_fail(service != NULL,            0);
	g_return_val_if_fail(GALAGO_IS_SERVICE(service), 0);

	return service->priv->flags;
}

/**
 * galago_service_create_account
 * @service:  The service.
 * @person:   The person the account belongs to.
 * @username: The account username.
 *
 * Creates an account belonging to this service.
 *
 * If the account already exists, the existing account will be returned.
 *
 * Returns: The account.
 */
GalagoAccount *
galago_service_create_account(GalagoService *service, GalagoPerson *person,
							  const char *username)
{
	g_return_val_if_fail(service != NULL,                       NULL);
	g_return_val_if_fail(person  != NULL,                       NULL);
	g_return_val_if_fail(GALAGO_IS_SERVICE(service),            NULL);
	g_return_val_if_fail(GALAGO_IS_PERSON(person),              NULL);
	g_return_val_if_fail(username != NULL && *username != '\0', NULL);

	return _galago_account_new(service, person, username, NULL);
}

/**
 * galago_service_get_account
 * @service:  The service.
 * @username: The account's username.
 * @query:    TRUE if a remote query should be done if there is no
 *            local account found, or FALSE.
 *
 * Returns the account with the specified username from a service.
 *
 * Returns: The account, if found, or NULL.
 */
GalagoAccount *
galago_service_get_account(const GalagoService *service, const char *username,
						   gboolean query)
{
	GalagoAccount *account;
	char *norm_username;

	g_return_val_if_fail(service  != NULL,           NULL);
	g_return_val_if_fail(username != NULL,           NULL);
	g_return_val_if_fail(*username != '\0',          NULL);
	g_return_val_if_fail(GALAGO_IS_SERVICE(service), NULL);

	norm_username = galago_service_normalize(service, username);
	account = g_hash_table_lookup(service->priv->accounts_table,
								  norm_username);
	g_free(norm_username);

	if (account == NULL && query && GALAGO_OBJECT_IS_REMOTE(service) &&
		!galago_is_daemon() && galago_is_connected())
	{
		account = galago_dbus_send_message_with_reply(
			GALAGO_OBJECT(service), "GetAccount",
			galago_value_new_object(GALAGO_TYPE_ACCOUNT, NULL),
			galago_value_new(GALAGO_VALUE_TYPE_STRING, &username, NULL),
			NULL);
	}

	return account;
}

/**
 * galago_service_get_account_async
 * @service:   The service.
 * @username:  The account's username.
 * @cb:        The callback function that will be passed the account.
 * @user_data: Custom data to pass to the callback function.
 * @free_func: Optional function to free @user_data when the request completes.
 *
 * Asynchronously retrieves the account with the specified username from
 * a service.
 *
 * For example:
 * <informalexample>
 * <programlisting>
 *  static void
 *  account_received_cb(GalagoService *service,
 *                      GalagoAccount *account,
 *                      gpointer user_data)
 *  {
 *  }
 *  
 *  static void
 *  get_bob_account(GalagoService *service)
 *  {
 *      GalagoCallHandle handle;
 *  
 *      handle = galago_service_get_account_async(service, "Bob",
 *                                                account_received_cb,
 *                                                NULL, NULL);
 *  }
 * </programlisting>
 * </informalexample>
 *
 * Returns: A valid #GalagoCallHandle on success. On error, this will return
 *          %GALAGO_CALL_HANDLE_INVALID.
 *
 * Since: 0.5.2
 */
GalagoCallHandle
galago_service_get_account_async(const GalagoService *service,
								 const char *username,
								 GalagoServiceAccountCb cb,
								 gpointer user_data,
								 GFreeFunc free_func)
{
	GalagoAccount *account;
	GalagoCallHandle handle = GALAGO_CALL_HANDLE_INVALID;

	g_return_val_if_fail(service  != NULL,                      handle);
	g_return_val_if_fail(GALAGO_IS_SERVICE(service),            handle);
	g_return_val_if_fail(username != NULL && *username != '\0', handle);
	g_return_val_if_fail(cb != NULL,                            handle);

	account = galago_service_get_account(service, username, FALSE);

	if (account != NULL)
	{
		cb((GalagoService *)service, account, user_data);
		handle = galago_calls_request_dummy_handle();
	}
	else if (GALAGO_OBJECT_IS_REMOTE(service) && !galago_is_daemon() &&
			 galago_is_connected())
	{
		handle = galago_dbus_send_message_with_reply_async(
			GALAGO_OBJECT(service), "GetAccount", G_CALLBACK(cb),
			user_data, free_func, g_cclosure_marshal_VOID__POINTER,
			galago_value_new_object(GALAGO_TYPE_ACCOUNT, NULL),
			galago_value_new(GALAGO_VALUE_TYPE_STRING, &username, NULL),
			NULL);
	}

	return handle;
}

/**
 * galago_service_get_accounts
 * @service: The service.
 * @query:   TRUE if a remote query should be done if there is no
 *           local account found, or FALSE.
 *
 * Returns a list of all accounts in the service.
 *
 * This may emit an account-added signal for every object that returns. If
 * your code connects to this signal and calls galago_service_get_accounts()
 * as a result, you will want to add a lock so that you don't end up with
 * unwanted side-effects.
 *
 * Returns: A list of all accounts in the service.
 */
GList *
galago_service_get_accounts(const GalagoService *service, gboolean query)
{
	g_return_val_if_fail(service != NULL,            NULL);
	g_return_val_if_fail(GALAGO_IS_SERVICE(service), NULL);

	if (query && GALAGO_OBJECT_IS_REMOTE(service) &&
		!galago_is_daemon() && galago_is_connected())
	{
		GList *temp;

		temp = galago_dbus_send_message_with_reply(
			GALAGO_OBJECT(service), "GetAccounts",
			galago_value_new_list(GALAGO_VALUE_TYPE_OBJECT, NULL,
								  (void *)GALAGO_TYPE_ACCOUNT),
			NULL);
		g_list_free(temp);
	}

	return service->priv->accounts;
}

void
_galago_service_add_account(GalagoService *service, GalagoAccount *account)
{
	const char *username;

	g_return_if_fail(service != NULL);
	g_return_if_fail(account != NULL);
	g_return_if_fail(GALAGO_IS_SERVICE(service));
	g_return_if_fail(GALAGO_IS_ACCOUNT(account));

	username = galago_account_get_username(account);

	/* XXX Remove this when this function is private. */
	if (galago_service_get_account(service, username, FALSE) != NULL)
	{
		g_warning("An account with username %s has already been "
				  "added to service %s",
				  username, galago_service_get_id(service));
		return;
	}

	g_hash_table_insert(service->priv->accounts_table,
						galago_service_normalize(service, username), account);

	service->priv->accounts = g_list_append(service->priv->accounts, account);

	if (GALAGO_OBJECT_IS_LOCAL(service))
		_galago_dbus_service_add_account(service, account);

	g_signal_emit(service, signals[ACCOUNT_ADDED], 0, account);
}

void
_galago_service_remove_account(GalagoService *service, GalagoAccount *account)
{
	const char *username;
	char *norm_username;

	g_return_if_fail(service != NULL);
	g_return_if_fail(account != NULL);
	g_return_if_fail(GALAGO_IS_SERVICE(service));
	g_return_if_fail(GALAGO_IS_ACCOUNT(account));

	username = galago_account_get_username(account);
	norm_username = galago_service_normalize(service, username);
	g_hash_table_remove(service->priv->accounts_table, norm_username);
	g_free(norm_username);

	service->priv->accounts = g_list_remove(service->priv->accounts, account);

	if (GALAGO_OBJECT_IS_LOCAL(service) && galago_is_connected() &&
		galago_is_feed())
	{
		galago_dbus_send_message(GALAGO_OBJECT(service), "RemoveAccount",
			galago_value_new_object(GALAGO_TYPE_ACCOUNT, G_OBJECT(account)),
			NULL);
	}

	g_signal_emit(service, signals[ACCOUNT_REMOVED], 0, account);
}

/**
 * galago_service_normalize
 * @service:  The service.
 * @username: The username to normalize.
 *
 * Normalizes an account username based on the service's normalization
 * flags.
 *
 * Returns: The string. This must be freed.
 */
char *
galago_service_normalize(const GalagoService *service, const char *username)
{
	GString *str;
	const char *c;
	GalagoServiceFlags flags;

	g_return_val_if_fail(service  != NULL,           NULL);
	g_return_val_if_fail(username != NULL,           NULL);
	g_return_val_if_fail(GALAGO_IS_SERVICE(service), NULL);

	flags = galago_service_get_flags(service);
	str   = g_string_new("");

	for (c = username;
		 *c != '\0' && (!(flags & GALAGO_STRIP_SLASH) || *c != '/');
		 c++)
	{
		if (*c == ' ' && !(flags & GALAGO_PRESERVE_SPACES))
		{
			while (*c == ' ')
				c++;
		}

		g_string_append_c(str, *c);
	}

	if (!(flags & GALAGO_PRESERVE_CASE))
		g_string_ascii_down(str);

	return g_string_free(str, FALSE);
}


/**************************************************************************
 * D-BUS Functions
 **************************************************************************/
static void
_galago_dbus_service_add_account(GalagoService *service, GalagoAccount *account)
{
	char *obj_path;

	g_assert(service != NULL);
	g_assert(account != NULL);

	if (!galago_is_connected() || !galago_is_feed())
		return;

	obj_path = galago_dbus_send_message_with_reply(
		GALAGO_OBJECT(service), "AddAccount",
		galago_value_new(GALAGO_VALUE_TYPE_STRING, NULL, NULL),
		galago_value_new_object(GALAGO_TYPE_ACCOUNT, G_OBJECT(account)),
		NULL);

	g_assert(obj_path != NULL);

	galago_object_set_dbus_path(GALAGO_OBJECT(account), obj_path);
	g_free(obj_path);
}
