/*
 * Galago Person API
 *
 * Copyright (C) 2004-2006 Christian Hammond
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#include <libgalago/galago-private.h>
#include <libgalago/galago-person.h>
#include <libgalago/galago-assert.h>
#include <libgalago/galago-context.h>
#include <libgalago/galago-context-priv.h>
#include <libgalago/galago-core.h>
#include <libgalago/galago-marshal.h>
#include <stdio.h>
#include <string.h>

struct _GalagoPersonPrivate
{
	char *id;
	char *session_id;
	char *display_name;
	GalagoImage *photo;
	GHashTable *accounts_table;
	GList *accounts;
};

typedef struct
{
	char *username;
	const GalagoService *service;

} AccountCacheKey;

enum
{
	PROP_0,
	PROP_ID,
	PROP_SESSION_ID,
	PROP_PHOTO
};

enum
{
	ACCOUNT_ADDED,
	ACCOUNT_REMOVED,
	PHOTO_SET,
	LAST_SIGNAL
};

/* galago-core.c */
GalagoAccount *_galago_core_calc_priority_account(const GalagoPerson *person);

static GalagoPerson *_galago_person_new(const char *id,
										const char *session_id,
										GalagoOrigin origin,
										const char *obj_path);
static void _galago_dbus_person_set_photo(GalagoPerson *person,
										  GalagoImage *photo);


/**************************************************************************
 * Object/Class support
 **************************************************************************/
static void galago_person_destroy(GalagoObject *gobject);
static void galago_person_dbus_message_append(DBusMessageIter *iter,
											   const GalagoObject *object);
static void *galago_person_dbus_message_get(DBusMessageIter *iter);
static void galago_person_dbus_push_full(GalagoObject *object);
static gchar *galago_person_dbus_get_signature(void);
static void galago_person_set_gproperty(GObject *object, guint prop_id,
										const GValue *value, GParamSpec *pspec);
static void galago_person_get_gproperty(GObject *object, guint prop_id,
										GValue *value, GParamSpec *pspec);

static GalagoObjectClass *parent_class = NULL;
static guint signals[LAST_SIGNAL] = {0};

G_DEFINE_TYPE(GalagoPerson, galago_person, GALAGO_TYPE_OBJECT);

static void
galago_person_class_init(GalagoPersonClass *klass)
{
	GalagoObjectClass *object_class  = GALAGO_OBJECT_CLASS(klass);
	GObjectClass      *gobject_class = G_OBJECT_CLASS(klass);

	parent_class = g_type_class_peek_parent(klass);

	object_class->dbus_interface = GALAGO_DBUS_PERSON_INTERFACE;
	object_class->supports_attrs = TRUE;

	object_class->destroy             = galago_person_destroy;
	object_class->dbus_message_append = galago_person_dbus_message_append;
	object_class->dbus_message_get    = galago_person_dbus_message_get;
	object_class->dbus_push_full      = galago_person_dbus_push_full;
	object_class->dbus_get_signature  = galago_person_dbus_get_signature;

	gobject_class->set_property = galago_person_set_gproperty;
	gobject_class->get_property = galago_person_get_gproperty;

	/**
	 * GalagoPerson::account-added:
	 * @person: The object which received the signal.
	 * @account: The account added to this person.
	 *
	 * Emitted whenever an account has been added to this person.
	 */
	signals[ACCOUNT_ADDED] =
		g_signal_new("account_added",
					 G_TYPE_FROM_CLASS(klass),
					 G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
					 G_STRUCT_OFFSET(GalagoPersonClass, account_added),
					 NULL, NULL,
					 g_cclosure_marshal_VOID__POINTER,
					 G_TYPE_NONE, 1,
					 G_TYPE_POINTER);

	/**
	 * GalagoPerson::account-removed:
	 * @person: The object which received the signal.
	 * @account: The account added to this person.
	 *
	 * Emitted whenever an account has been removed from this person.
	 */
	signals[ACCOUNT_REMOVED] =
		g_signal_new("account_removed",
					 G_TYPE_FROM_CLASS(klass),
					 G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
					 G_STRUCT_OFFSET(GalagoPersonClass, account_removed),
					 NULL, NULL,
					 g_cclosure_marshal_VOID__POINTER,
					 G_TYPE_NONE, 1,
					 G_TYPE_POINTER);

	/**
	 * GalagoPerson::photo-set:
	 * @person: The object which received the signal.
	 * @photo: The photo set on this person.
	 *
	 * Emitted whenever a photo has been set on this person.
	 */
	signals[PHOTO_SET] =
		g_signal_new("photo_set",
					 G_TYPE_FROM_CLASS(klass),
					 G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
					 G_STRUCT_OFFSET(GalagoPersonClass, photo_set),
					 NULL, NULL,
					 g_cclosure_marshal_VOID__POINTER,
					 G_TYPE_NONE, 1,
					 G_TYPE_POINTER);

	g_object_class_install_property(gobject_class, PROP_ID,
		g_param_spec_string("id", "ID",
							"The person's unique ID",
							NULL,
							G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY |
							G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));

	g_object_class_install_property(gobject_class, PROP_SESSION_ID,
		g_param_spec_string("session-id", "Session ID",
							"The person's session ID",
							NULL,
							G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY |
							G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));

	g_object_class_install_property(gobject_class, PROP_PHOTO,
		g_param_spec_object("photo", "Photo",
							"The person's unique ID",
							GALAGO_TYPE_IMAGE,
							G_PARAM_READWRITE |
							G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
}

static void
cache_key_free(AccountCacheKey *key)
{
	if (key->username != NULL)
		g_free(key->username);

	g_free(key);
}

static unsigned int
cache_key_hash(const void *p)
{
	const AccountCacheKey *key = (AccountCacheKey *)p;

	return g_str_hash(key->username) +
	       g_str_hash(galago_service_get_id(key->service));
}

static gboolean
cache_key_equal(const void *a, const void *b)
{
	const AccountCacheKey *ka = (AccountCacheKey *)a;
	const AccountCacheKey *kb = (AccountCacheKey *)b;

	return (ka == kb ||
			(!strcmp(ka->username, kb->username) &&
			 !strcmp(galago_service_get_id(ka->service),
					 galago_service_get_id(kb->service))));
}

static void
galago_person_init(GalagoPerson *person)
{
	person->priv = g_new0(GalagoPersonPrivate, 1);

	person->priv->accounts_table =
		g_hash_table_new_full(cache_key_hash, cache_key_equal,
							  (GFreeFunc)cache_key_free, NULL);
}

static void
galago_person_destroy(GalagoObject *object)
{
	GalagoPerson *person = GALAGO_PERSON(object);

	if (person->priv != NULL)
	{
		if (person->priv->accounts != NULL)
		{
			g_list_foreach(person->priv->accounts,
						   (GFunc)galago_object_destroy, NULL);
			g_list_free(person->priv->accounts);
		}

		g_hash_table_destroy(person->priv->accounts_table);

		galago_context_push(galago_object_get_context(object));
		galago_context_remove_person(person);
		galago_context_pop();

		g_free(person->priv->id);
		g_free(person->priv->session_id);
		g_free(person->priv->display_name);
		g_free(person->priv);
		person->priv = NULL;
	}

	if (GALAGO_OBJECT_CLASS(parent_class)->destroy != NULL)
		GALAGO_OBJECT_CLASS(parent_class)->destroy(object);
}

static void
galago_person_dbus_message_append(DBusMessageIter *iter,
								  const GalagoObject *object)
{
	GalagoPerson *person = (GalagoPerson *)object;
	const char *obj_path, *session_id, *id;
	gboolean is_me;

	obj_path   = galago_object_get_dbus_path(GALAGO_OBJECT(person));
	id         = galago_person_get_id(person);
	session_id = galago_person_get_session_id(person);
	is_me      = galago_person_is_me(person);

	galago_dbus_message_iter_append_string_or_nil(iter, obj_path);
	galago_dbus_message_iter_append_string_or_nil(iter, id);
	dbus_message_iter_append_basic(iter, DBUS_TYPE_STRING, &session_id);
	dbus_message_iter_append_basic(iter, DBUS_TYPE_BOOLEAN, &is_me);
}

static void *
galago_person_dbus_message_get(DBusMessageIter *iter)
{
	GalagoPerson *person;
	const char *obj_path, *uid, *session_id;
	gboolean me;

	obj_path = galago_dbus_message_iter_get_string_or_nil(iter);
	dbus_message_iter_next(iter);

	uid = galago_dbus_message_iter_get_string_or_nil(iter);
	dbus_message_iter_next(iter);

	dbus_message_iter_get_basic(iter, &session_id);
	dbus_message_iter_next(iter);

	dbus_message_iter_get_basic(iter, &me);

	if (me)
		uid = GALAGO_ME_ID;

	person = _galago_person_new(uid, session_id, GALAGO_REMOTE, obj_path);

	return person;
}

static void
galago_person_dbus_push_full(GalagoObject *object)
{
	GalagoPerson *person = (GalagoPerson *)object;

	_galago_dbus_person_set_photo(person,
		galago_person_get_photo(person, FALSE));

	if (GALAGO_OBJECT_CLASS(parent_class)->dbus_push_full != NULL)
		GALAGO_OBJECT_CLASS(parent_class)->dbus_push_full(object);
}

static gchar *
galago_person_dbus_get_signature(void)
{
	return g_strdup(DBUS_TYPE_STRING_AS_STRING    // obj path
					DBUS_TYPE_STRING_AS_STRING    // id
					DBUS_TYPE_STRING_AS_STRING    // session id
					DBUS_TYPE_BOOLEAN_AS_STRING); // me
}

static void
galago_person_set_gproperty(GObject *object, guint prop_id,
							const GValue *value, GParamSpec *pspec)
{
	GalagoPerson *person = GALAGO_PERSON(object);

	switch (prop_id)
	{
		case PROP_ID:
			_galago_person_set_id(person, g_value_get_string(value));
			break;

		case PROP_SESSION_ID:
			_galago_person_set_session_id(person, g_value_get_string(value));
			break;

		case PROP_PHOTO:
			galago_person_set_photo(person,
				GALAGO_IMAGE(g_value_get_object(value)));
			break;

		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}

static void
galago_person_get_gproperty(GObject *object, guint prop_id,
							GValue *value, GParamSpec *pspec)
{
	GalagoPerson *person = GALAGO_PERSON(object);

	switch (prop_id)
	{
		case PROP_ID:
			g_value_set_string(value, galago_person_get_id(person));
			break;

		case PROP_SESSION_ID:
			g_value_set_string(value, galago_person_get_session_id(person));
			break;

		case PROP_PHOTO:
			g_value_set_object(value, galago_person_get_photo(person, TRUE));
			break;

		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}


/**************************************************************************
 * GalagoPerson API
 **************************************************************************/
static GalagoPerson *
_galago_person_new(const char *uid, const char *session_id,
				   GalagoOrigin origin, const char *obj_path)
{
	GalagoPerson *person = NULL;

	g_return_val_if_fail(galago_is_initted(), NULL);

	if (session_id != NULL)
	{
		person = galago_context_get_person_with_session_id(session_id,
														   origin);
	}
	else if (uid != NULL)
	{
		person = galago_context_get_person(uid, origin);
	}

	if (person == NULL)
	{
		person = g_object_new(GALAGO_TYPE_PERSON,
							  "id", uid,
							  "session_id", session_id,
							  "origin", origin,
							  NULL);

		/* Set the object path, if specified. */
		if (origin == GALAGO_REMOTE)
			galago_object_set_dbus_path(GALAGO_OBJECT(person), obj_path);

		galago_context_add_person(person);
	}

	return person;
}

/**
 * galago_create_person
 * @uid: The person UID.
 *
 * Creates a person and adds it to the local cache.
 *
 * Returns: The new person, or an existing one if one already exists
 *          with this UID.
 */
GalagoPerson *
galago_create_person(const char *uid)
{
	G_LOCK_DEFINE_STATIC(session_id_lock);
	static gulong next_session_id_index = 1;
	char *session_id = NULL;
	GalagoPerson *person;

	g_return_val_if_fail(galago_is_initted(), NULL);

	G_LOCK(session_id_lock);
	session_id = g_strdup_printf("session-id-%ld", next_session_id_index++);
	G_UNLOCK(session_id_lock);

	person = _galago_person_new(uid, session_id, GALAGO_LOCAL, NULL);

	if (session_id != NULL)
		g_free(session_id);

	return person;
}

/**
 * galago_person_set_me
 * @person: The person.
 *
 * Sets a person to be the "Me" person. This is a one-way thing. You
 * can't make a "Me" person a normal person.
 *
 * This is an internal function. Please use galago_get_me() instead of
 * ever using this.
 */
void
galago_person_set_me(GalagoPerson *person)
{
	g_return_if_fail(person != NULL);
	g_return_if_fail(GALAGO_IS_PERSON(person));

	if (galago_person_is_me(person))
		return;

	_galago_person_set_id(person, GALAGO_ME_ID);
}

/**
 * galago_person_is_me
 * @person: The person.
 *
 * Returns whether or not a person is the "Me" person.
 *
 * Returns: TRUE if the person is the "Me" person, or FALSE.
 */
gboolean
galago_person_is_me(const GalagoPerson *person)
{
	g_return_val_if_fail(person != NULL,           FALSE);
	g_return_val_if_fail(GALAGO_IS_PERSON(person), FALSE);

	return person->priv->id != NULL &&
	       !strcmp(person->priv->id, GALAGO_ME_ID);
}

void
_galago_person_set_id(GalagoPerson *person, const char *id)
{
	g_return_if_fail(person != NULL);
	g_return_if_fail(GALAGO_IS_PERSON(person));

	if (person->priv->id != NULL)
		g_free(person->priv->id);

	person->priv->id = (id == NULL ? NULL : g_strdup(id));

	g_object_notify(G_OBJECT(person), "id");
}

void
_galago_person_set_session_id(GalagoPerson *person, const char *session_id)
{
	g_return_if_fail(person != NULL);
	g_return_if_fail(GALAGO_IS_PERSON(person));

	if (person->priv->session_id != NULL)
		g_free(person->priv->session_id);

	person->priv->session_id =
		(session_id == NULL ? NULL : g_strdup(session_id));

	g_object_notify(G_OBJECT(person), "session-id");
}

/**
 * galago_person_get_id
 * @person: The person.
 *
 * Returns a person's application-specific unique ID.
 *
 * Returns: The person's ID.
 */
const char *
galago_person_get_id(const GalagoPerson *person)
{
	g_return_val_if_fail(person != NULL,           NULL);
	g_return_val_if_fail(GALAGO_IS_PERSON(person), NULL);

	return person->priv->id;
}

/**
 * galago_person_get_session_id
 * @person: The person.
 *
 * Returns a person's session ID.
 *
 * Returns: The person's session ID.
 */
const char *
galago_person_get_session_id(const GalagoPerson *person)
{
	g_return_val_if_fail(person != NULL,           NULL);
	g_return_val_if_fail(GALAGO_IS_PERSON(person), NULL);

	return person->priv->session_id;
}

/**
 * galago_person_get_display_name
 * @person: The person.
 *
 * Returns the display name of a person, if any.
 *
 * The returned display name is from a static buffer. If you wish to store
 * the name, you must g_strdup it.
 *
 * Returns: The person's display name, or NULL.
 */
const char *
galago_person_get_display_name(const GalagoPerson *person)
{
	const char *first_name, *last_name;

	g_return_val_if_fail(person != NULL,           NULL);
	g_return_val_if_fail(GALAGO_IS_PERSON(person), NULL);

	/* XXX This is all a hack, but for now, it should work. Improve this! */

	g_free(person->priv->display_name);
	person->priv->display_name = NULL;

	first_name = galago_object_get_attr_string(GALAGO_OBJECT(person),
											   GALAGO_PERSON_ATTR_FIRST_NAME);
	last_name = galago_object_get_attr_string(GALAGO_OBJECT(person),
											  GALAGO_PERSON_ATTR_LAST_NAME);

	if (first_name != NULL || last_name != NULL)
	{
		person->priv->display_name = g_strdup_printf("%s%s%s",
			(first_name == NULL ? "" : first_name),
			(first_name != NULL && last_name != NULL ? " " : ""),
			(last_name == NULL ? "" : last_name));
	}

	return person->priv->display_name;
}

/**
 * galago_person_set_photo
 * @person: The person.
 * @photo:  The photo to set.
 *
 * Sets the person's photo.
 *
 * The person should be a local person.
 */
void
galago_person_set_photo(GalagoPerson *person, GalagoImage *photo)
{
	g_return_if_fail(person != NULL);
	g_return_if_fail(GALAGO_IS_PERSON(person));
	g_return_if_fail(photo == NULL || GALAGO_IS_IMAGE(photo));

	if (person->priv->photo == photo)
		return;

	if (person->priv->photo != NULL)
	{
		GalagoImage *old_photo = person->priv->photo;
		person->priv->photo = NULL;
		galago_object_destroy(GALAGO_OBJECT(old_photo));
	}

	person->priv->photo = photo;

	if (GALAGO_OBJECT_IS_LOCAL(person))
		_galago_dbus_person_set_photo(person, photo);

	g_object_notify(G_OBJECT(person), "photo");

	/* XXX Is this redundant? */
	g_signal_emit(person, signals[PHOTO_SET], 0, photo);
}

/**
 * galago_person_get_photo
 * @person: The person.
 * @query:  TRUE if a remote query should be done if there is no
 *          local photo found, or FALSE.
 *
 * Returns the person's photo.
 *
 * Returns: The photo, if found, or NULL.
 */
GalagoImage *
galago_person_get_photo(const GalagoPerson *person, gboolean query)
{
	g_return_val_if_fail(person != NULL,           NULL);
	g_return_val_if_fail(GALAGO_IS_PERSON(person), NULL);

	if (person->priv->photo == NULL && query &&
		GALAGO_OBJECT_IS_REMOTE(person) && !galago_is_daemon() &&
		galago_is_connected())
	{
		person->priv->photo =
			galago_dbus_send_message_with_reply(GALAGO_OBJECT(person),
				"GetPhoto",
				galago_value_new_object(GALAGO_TYPE_IMAGE, NULL),
				NULL);
		g_object_notify(G_OBJECT(person), "photo");
	}

	return person->priv->photo;
}

/**
 * galago_person_has_accounts
 * @person: The person.
 * @query:  TRUE if a query should be done to check the server-side
 *          state of the accounts, or FALSE.
 *
 * Returns whether the person has any accounts at all.
 *
 * Returns: TRUE if the person has at least one account, or FALSE otherwise.
 */
gboolean
galago_person_has_accounts(const GalagoPerson *person, gboolean query)
{
	g_return_val_if_fail(person != NULL,           FALSE);
	g_return_val_if_fail(GALAGO_IS_PERSON(person), FALSE);

	if (query && GALAGO_OBJECT_IS_REMOTE(person) && !galago_is_daemon() &&
		galago_is_connected())
	{
		gsize num_accounts =
			GPOINTER_TO_SIZE(galago_dbus_send_message_with_reply(
				GALAGO_OBJECT(person), "GetAccountsCount",
				galago_value_new(GALAGO_VALUE_TYPE_UINT, NULL, NULL),
				NULL));

		return (num_accounts > 0);
	}

	return galago_person_get_accounts(person, query) != NULL;
}

/**
 * galago_person_get_accounts
 * @person: The person.
 * @query:  TRUE if a remote query should be done if there is no
 *          local account found, or FALSE.
 *
 * Returns a list of accounts in the person.
 *
 * Returns: The list of accounts.
 */
GList *
galago_person_get_accounts(const GalagoPerson *person, gboolean query)
{
	g_return_val_if_fail(person != NULL,           NULL);
	g_return_val_if_fail(GALAGO_IS_PERSON(person), NULL);

	if (query && GALAGO_OBJECT_IS_REMOTE(person) && !galago_is_daemon() &&
		galago_is_connected())
	{
		g_list_free(galago_dbus_send_message_with_reply(GALAGO_OBJECT(person),
			"GetAccounts",
			galago_value_new_list(GALAGO_VALUE_TYPE_OBJECT, NULL,
								  (void *)GALAGO_TYPE_ACCOUNT),
			NULL));
	}

	return person->priv->accounts;
}

GalagoAccount *
_galago_person_default_calc_priority_account(const GalagoPerson *person)
{
	GList *l, *accounts;
	GalagoPresence *priority_presence = NULL;
	GalagoAccount *account = NULL;

	for (l = accounts = galago_person_get_accounts(person, TRUE);
		 l != NULL;
		 l = l->next)
	{
		GalagoAccount *account = (GalagoAccount *)l->data;
		GalagoPresence *presence = galago_account_get_presence(account, TRUE);

		if (galago_presence_compare(priority_presence, presence) > 0)
			priority_presence = presence;
	}

	if (priority_presence != NULL)
	{
		account = galago_presence_get_account(priority_presence);
	}
	else if (accounts != NULL)
	{
		account = (GalagoAccount *)accounts->data;
	}

	return account;
}

/**
 * galago_person_get_priority_account
 * @person: The person.
 *
 * Returns the most available "priority" account.
 *
 * Returns: The priority account.
 */
GalagoAccount *
galago_person_get_priority_account(const GalagoPerson *person)
{
	g_return_val_if_fail(person != NULL,           NULL);
	g_return_val_if_fail(GALAGO_IS_PERSON(person), NULL);

	if (galago_person_get_accounts(person, TRUE) == NULL)
		return NULL;

	return _galago_core_calc_priority_account(person);
}

void
_galago_person_add_account(GalagoPerson *person, GalagoAccount *account)
{
	const char *username;
	GalagoService *service;
	AccountCacheKey *key;

	g_return_if_fail(person  != NULL);
	g_return_if_fail(account != NULL);
	g_return_if_fail(GALAGO_IS_PERSON(person));
	g_return_if_fail(GALAGO_IS_ACCOUNT(account));

	username = galago_account_get_username(account);
	service  = galago_account_get_service(account);

	if (galago_person_get_account(person, service, username, FALSE) != NULL)
	{
#if 0
		g_warning("An account with username %s on service %s has "
				  "already been added to person %s",
				  username, galago_service_get_id(service),
				  galago_person_get_id(person));
#endif

		return;
	}

	key = g_new0(AccountCacheKey, 1);
	key->username = galago_service_normalize(service, username);
	key->service  = service;

	g_hash_table_insert(person->priv->accounts_table, key, account);

	person->priv->accounts = g_list_append(person->priv->accounts, account);

	g_signal_emit(person, signals[ACCOUNT_ADDED], 0, account);
}

void
_galago_person_remove_account(GalagoPerson *person, GalagoAccount *account)
{
	AccountCacheKey key;
	GalagoService *service;
	const char *username;

	g_return_if_fail(person  != NULL);
	g_return_if_fail(account != NULL);
	g_return_if_fail(GALAGO_IS_PERSON(person));
	g_return_if_fail(GALAGO_IS_ACCOUNT(account));

	service  = galago_account_get_service(account);
	username = galago_account_get_username(account);

	key.username = galago_service_normalize(service, username);
	key.service  = galago_account_get_service(account);

	g_hash_table_remove(person->priv->accounts_table, &key);

	g_free(key.username);

	person->priv->accounts = g_list_remove(person->priv->accounts, account);

	g_signal_emit(person, signals[ACCOUNT_REMOVED], 0, account);
}

/**
 * galago_person_get_account
 * @person:   The person.
 * @service:  The service.
 * @username: The username.
 * @query:    TRUE if a remote query should be done if there is no
 *            local account found, or FALSE.
 *
 * Returns the account with the specified username and service.
 *
 * Returns: The account, if found, or NULL.
 */
GalagoAccount *
galago_person_get_account(const GalagoPerson *person,
						  const GalagoService *service,
						  const char *username,
						  gboolean query)
{
	GalagoAccount *account;
	AccountCacheKey key;

	g_return_val_if_fail(person   != NULL,           NULL);
	g_return_val_if_fail(service  != NULL,           NULL);
	g_return_val_if_fail(username != NULL,           NULL);
	g_return_val_if_fail(GALAGO_IS_PERSON(person),   NULL);
	g_return_val_if_fail(GALAGO_IS_SERVICE(service), NULL);
	g_return_val_if_fail(galago_object_get_origin(GALAGO_OBJECT(person)) ==
						 galago_object_get_origin(GALAGO_OBJECT(service)),
						 NULL);

	key.username = galago_service_normalize(service, username);
	key.service  = service;

	account = g_hash_table_lookup(person->priv->accounts_table, &key);

	g_free(key.username);

	if (account == NULL && query && GALAGO_OBJECT_IS_REMOTE(person) &&
		!galago_is_daemon() && galago_is_connected())
	{
		account = galago_dbus_send_message_with_reply(
			GALAGO_OBJECT(person), "GetAccount",
			galago_value_new_object(GALAGO_TYPE_ACCOUNT, NULL),
			galago_value_new_object(GALAGO_TYPE_SERVICE, G_OBJECT(service)),
			galago_value_new(GALAGO_VALUE_TYPE_STRING, &username, NULL),
			NULL);
	}

	return account;
}

/**************************************************************************
 * D-BUS Functions
 **************************************************************************/
static void
_galago_dbus_person_set_photo(GalagoPerson *person, GalagoImage *photo)
{
	if (!galago_is_connected() || !galago_is_feed())
		return;

	if (photo == NULL)
	{
		galago_dbus_send_message(GALAGO_OBJECT(person), "UnsetPhoto", NULL);
	}
	else
	{
		galago_dbus_send_message(GALAGO_OBJECT(person), "SetPhoto",
			galago_value_new_object(GALAGO_TYPE_IMAGE, G_OBJECT(photo)),
			NULL);
	}
}
