/**
 * Galago  D-BUS Interface.
 * @file protocol.hh
 * Defines the DBUS interface of Galago
 */

/** D-BUS Service */
#define GALAGO_DBUS_SERVICE "org.freedesktop.Galago"

/** Error signals */
#define GALAGO_DBUS_ERROR_INVALID_PROPERTY "org.freedesktop.Galago.Error.InvalidProperty"

/** Error signals */
#define GALAGO_DBUS_ERROR_OBJECT_NOT_FOUND "org.freedesktop.Galago.Error.ObjectNotFound"

namespace org.freedesktop.Galago {

/**
 * Account D-BUS messaging
 */
class Account
{
public:
	Service Service;     //!< Parent service
	Person Person;       //!< Person who owns this account
	string Username;     //!< Account username
	string DisplayName;  //!< Name intended for display purposes
	bool Connected;      //!< Connected flag

	/**
	 * @name Methods
	 * @{
	 */

	/**
	 * Sets whether or not the account is connected.
	 *
	 * @param connected The connected state.
	 */
	void SetConnected(bool connected);

	/**
	 * Adds a contact to an account.
	 *
	 * @param contact The contact's account to add.
	 */
	void AddContact(Account contact);

	/**
	 * Removes a contact from an account.
	 *
	 * @param contact The contact's account to remove.
	 */
	void RemoveContact(Account contact);

	/**
	 * Returns account's contact with the specified username in an account.
	 *
	 * @param username Contac's username.
	 *
	 * @return The contact's account, or
	 *         org.freedesktop.Galago.Error.ObjectNotFound if not found.
	 * @throw org.freedesktop.Galago.Error.ObjectNotFound
	 */
	Account GetContact(string username);

	/**
	 * Returns an array of accounts of users seen from this account.
	 *
	 * This may emit a contact-added signal for every object that returns. If
	 * your code connects to this signal and calls galago_account_get_contacts()
	 * as a result, you will want to add a lock so that you don't end up with
	 * unwanted side-effects.
	 *
	 * @return An array of accounts of other users.
	 */
	Array<Account> GetContacts();

	/**
	 * Returns the number of accounts of users seen from this account.
	 *
	 * @return The number of accounts of other users.
	 */
	uint32 GetContactsCount();

	/**
	 * Creates an account's presence.
	 *
	 * An account only has one presence. If this is called multiple times,
	 * the first presence object's path will be returned.
	 *
	 * @param presence The presence to set.
	 *
	 * @return The object path of the presence.
	 */
	string SetPresence(Presence presence);

	/**
	 * Destroys the account's presence.
	 *
	 * This removes the presence from the account.
	 */
	void DestroyPresence();

	/**
	 * Returns the account's presence.
	 *
	 * @return The presence, or org.freedesktop.Galago.Error.ObjectNotFound
	 *         if not found.
	 * @throw org.freedesktop.Galago.Error.ObjectNotFound
	 */
	Presence GetPresence();

	/**
	 * Sets the account's avatar.
	 *
	 * @param avatar The avatar to set.
	 */
	void SetAvatar(Image avatar);

	/**
	 * Returns the account's avatar.
	 *
	 * @return The avatar image, or
	 *         org.freedesktop.Galago.Error.ObjectNotFound if not found.
	 * @throw org.freedesktop.Galago.Error.ObjectNotFound
	 */
	Image GetAvatar();

	/**
	 * Unsets the account's avatar.
	 */
	void UnsetAvatar();

	/**
	 * Sets the account's displayed name.
	 *
	 * This is the alias that the account may be shown as on another client.
	 * It's purely optional.
	 *
	 * @param displayName The display name.
	 */
	void SetDisplayName(string displayName);

	/**
	 * Returns the account's displayed name.
	 *
	 * If the displayed name is not explicitly set, this will return the
	 * screen name.
	 *
	 * @return The displayed name.
	 */
	string GetDisplayName();

	/*@}*/

	/**
	 * @name Signals
	 * @{
	 */

	/**
	 * Emitted when the account is connected to the server.
	 */
	Connected();

	/**
	 * Emitted when the account is disconnected from the server.
	 */
	Disconnected();

	/**
	 * Emitted when the account's presence is set.
	 *
	 * @param presence The presence that was set.
	 */
	PresenceSet(Presence presence);

	/**
	 * Emitted when the account's presence is unset.
	 */
	PresenceUnset();

	/**
	 * Emitted when the account's avatar is set.
	 *
	 * @param avatar The new avatar image.
	 */
	AvatarSet(Image avatar);

	/**
	 * Emitted when the account's avatar is unset.
	 */
	AvatarUnset();

	/**
	 * Emitted when the account's display name is changed.
	 *
	 * @param displayName The account's new display name.
	 */
	DisplayNameChanged(string displayName);

	/*@}*/
};


/**
 * Galago Core D-BUS messaging
 */
class Core
{
public:
	/**
	 * @name Methods
	 * @{
	 */

	/**
	 * Adds a service to Galago's list of services.
	 *
	 * @param service The service to add.
	 *
	 * @return The service's new object path.
	 */
	string AddService(Service service);

	/**
	 * Removes a service from Galago's list of services.
	 *
	 * @param service The service to remove.
	 */
	void RemoveService(Service service);

	/**
	 * Adds a person to Galago's person cache.
	 *
	 * @param person The person to add.
	 *
	 * @return The person's new object path.
	 */
	string AddPerson(Person person);

	/**
	 * Removes a person from Galago's person cache.
	 *
	 * @param person The person to remove.
	 */
	void RemovePerson(Person person);

	/**
	 * Registers a Galago connection to the daemon.
	 *
	 * @param appName      The ID name used for this connection. It's often
	 *                     the name of the program. It must be alphanumeric.
	 * @param feed         true if this connection should feed presence data.
	 *                     false if it will only be querying for data.
	 * @param[out] uid     The resulting UID assigned to this connection.
	 * @param[out] objPath The resulting connection object path for this
	 *                     connection.
	 */
	void Register(string appName, bool feed, string *uid, string *objPath);

	/**
	 * Unregisters a Galago connection from the daemon.
	 *
	 * @param objPath The object path.
	 */
	void Unregister(string objPath);

	/**
	 * Returns the service with the specified ID.
	 *
	 * @param id The service ID.
	 *
	 * @return The service, or org.freedesktop.Galago.Error.ObjectNotFound
	 *         if not found.
	 * @throw org.freedesktop.Galago.Error.ObjectNotFound
	 */
	Service GetService(string id);

	/**
	 * Returns an array of known services.
	 *
	 * @return The array of services
	 */
	Array<Service> GetServices();

	/**
	 * Returns the number of known services.
	 *
	 * @return The number of services.
	 */
	uint32 GetServicesCount();

	/**
	 * Returns the "Me" person.
	 *
	 * @return The "Me" person.
	 */
	Person GetMe();

	/**
	 * Returns the person with the specified ID.
	 *
	 * @param id The person ID.
	 *
	 * @return The person, or org.freedesktop.Galago.Error.ObjectNotFound
	 *         if not found.
	 * @throw org.freedesktop.Galago.Error.ObjectNotFound
	 */
	Person GetPerson(string id);

	/**
	 * Returns an array of known people.
	 *
	 * @return An array of people.
	 */
	Array<Person> GetPeople();

	/**
	 * Returns the number of known people.
	 *
	 * @return The number of people.
	 */
	uint32 GetPeopleCount();

	/**
	 * Shuts down Galago daemon.
	 */
	void ShutDown();

	/*@}*/

	/**
	 * @name Signals
	 * @{
	 */

	/**
	 * Emitted when D-BUS service is added to the daemon.
	 *
	 * @param service The added service.
	 */
	ServiceAdded(Service service);

	/**
	 * Emitted when D-BUS service is removed from the daemon.
	 *
	 * @param service The removed service.
	 */
	ServiceRemoved(Service service);

	/*@}*/
};


/**
 * Image data.
 */
class Image
{
public:
	Array<byte> ImageData; //!< The image data.
};


/**
 * Person D-BUS messaging
 */
class Person
{
public:
	string ObjPath;   //!< Object path of the person.
	string Id;        //!< The person's ID.
	string sessionId; //!< Client-defined per-session handle ID.
	bool IsMe;        //!< Flag specifying if this person is the "Me" person.

	/**
	 * @name Methods
	 * @{
	 */

	/**
	 * Sets a property on a person.
	 *
	 * This is limited to string, boolean, and uint32 value types.
	 *
	 * @param name   The name of the property to set.
	 * @param value  The value of the property.
	 */
	void SetProperty(string name, GalagoValue value);

	/**
	 * Removes a property from a person.
	 *
	 * @param name The name of the property to remove.
	 */
	void RemoveProperty(string name);

	/**
	 * Returns the value of a property on a person.
	 *
	 * @param name The name of the property.
	 *
	 * @return The property value, or
	 *         org.freedesktop.Galago.Error.InvalidProperty if not found.
	 * @throw org.freedesktop.Galago.Error.InvalidProperty
	 */
	GalagoValue GetProperty(string name);

	/**
	 * Returns a person's account with the specified username and service.
	 *
	 * @param service  The service
	 * @param username The account's username.
	 *
	 * @return The account, or org.freedesktop.Galago.Error.ObjectNotFound
	 *         if not found.
	 * @throw org.freedesktop.Galago.Error.ObjectNotFound
	 */
	Account GetAccount(Service service, string username);

	/**
	 * Returns an array of accounts in the person.
	 *
	 * @return Array of accounts.
	 */
	Array<Account> GetAccounts();

	/**
	 * Returns the number of accounts in the person.
	 *
	 * @return The number of accounts.
	 */
	uint32 GetAccountsCount();

	/**
	 * Sets the person's photo.
	 *
	 * @param photo The photo to set.
	 */
	void SetPhoto(Image photo);

	/**
	 * Returns the person's photo.
	 *
	 * @return The photo, or org.freedesktop.Galago.Error.ObjectNotFound
	 *         if not found.
	 * @throw org.freedesktop.Galago.Error.ObjectNotFound
	 */
	Image GetPhoto();

	/**
	 * Unsets the person's photo.
	 */
	void UnsetPhoto();

	/*@}*/

	/**
	 * @name Signals
	 * @{
	 */

	/**
	 * Emitted when the person's photo is set.
	 *
	 * @param photo The photo set.
	 */
	PhotoSet(Image photo);

	/**
	 * Emitted when the person's photo is unset.
	 */
	PhotoUnset();

	/**
	 * Emitted when the person's property is set.
	 *
	 * @param name  The name of the property to set.
	 * @param value The value of the property. This must be a bool, uint32, or
	 *              string.
	 */
	PropertySet(string name, variant value);

	/**
	 * Emitted when the person's property is unset.
	 *
	 * @param name The name of the property to unset.
	 */
	PropertyRemoved(string name);

	/*@}*/
};


/**
 * @interface Presence
 *
 * Presence D-BUS messaging
 */
class Presence
{
public:
	Account Account;        //!< The parent account.
	bool Idle;              //!< Idle state.
	uint32 IdleStartTime;   //!< The time the idle state was set.
	Array<Status> Statuses; //!< Array of statuses.

	/**
	 * @name Methods
	 * @{
	 */

	/**
	 * Adds a single status to a presence.
	 *
	 * @param status The status.
	 *
	 * @return The status's new object path.
	 */
	string AddStatus(Status status);

	/**
	 * Removes a single status from a presence.
	 *
	 * The status being removed must not be an exclusive status.
	 *
	 * @param statusId The ID of the status to remove.
	 */
	void RemoveStatus(string statusId);

	/**
	 * Sets an array of statuses in a presence.
	 *
	 * @param statuses The array of statuses.
	 */
	void SetStatuses(Array<Status> statuses);

	/**
	 * Sets the idle state on a presence.
	 *
	 * @param idle          true if the user is idle, otherwise false.
	 * @param idleStartTime The time the idle state was set. This is only
	 *                      used if @a idle is true.
	 */
	void SetIdle(bool idle, uint32 idleStartTime);

	/*@}*/

	/**
	 * @name Signals
	 * @{
	 */

	/**
	 * Emitted when the presence's idle state is updated.
	 *
	 * @param idle          true if the user is idle, or false.
	 * @param idleStartTime The time that the idle state was set.
	 */
	IdleUpdated(bool idle, uint32 idleStartTime);

	/**
	 * Emitted when the presence's status is added.
	 *
	 * @param status The added status.
	 */
	StatusAdded(Status status);

	/**
	 * Emitted when the presence's status is removed.
	 *
	 * @param statusId The ID of the removed status.
	 */
	StatusRemoved(string statusId);

	/*@}*/
};


/**
 * Service D-BUS messaging
 */
class Service
{
public:
	string ObjPath; //!< The object path of the service.
	string Id;      //!< The ID of the service.
	string Name;    //!< The name of the service.
	uint32 Flags;   //!< The flags.

	/**
	 * @name Methods
	 * @{
	 */

	/**
	 * Adds an account to a service.
	 *
	 * @param account The account to add.
	 *
	 * @return The account's new object path.
	 */
	string AddAccount(Account account);

	/**
	 * Removes an account from a service.
	 *
	 * @param account The account to remove.
	 */
	void RemoveAccount(Account account);

	/**
	 * Returns an account with the specified username in the service.
	 *
	 * @param username The account's username.
	 *
	 * @return The account, or org.freedesktop.Galago.Error.ObjectNotFound
	 *         if not found.
	 * @throw org.freedesktop.Galago.Error.ObjectNotFound
	 */
	Account GetAccount(string username);

	/**
	 * Returns an array of all accounts in the service.
	 *
	 * @return An array of all accounts in the service.
	 */
	Array<Account> GetAccounts();

	/**
	 * Returns the number of all accounts in the service.
	 *
	 * @return The number of all accounts in the service.
	 */
	uint32 GetAccountsCount();

	/*@}*/

	/**
	 * @name Signals
	 * @{
	 */

	/**
	 * Emitted when the account is added to a service.
	 *
	 * @param account The added account.
	 */
	AccountAdded(Account account);

	/**
	 * Emitted when the account is removed from a service.
	 *
	 * @param account The removed account.
	 */
	AccountRemoved(Account account);

	/*@}*/
};


/**
 * A status in a Presence.
 */
class Status
{
public:
	byte Type;                    //!< The type of status.
	string Id;                    //!< The ID of the status.
	string Name;                  //!< The human-readable name of the status.
	bool Exclusive;               //!< The exclusive property of this status.
	Array<StatusAttr> Attributes; //!< The array of status attributes.
};


/**
 * An attribute in a status.
 *
 * The attribute value must be a bool, string, uint32, or double.
 */
class StatusAttr
{
public:
	string Name;   //!< The name of the attribute.
	variant Value; //!< The value of the attribute.
};


namespace Error {

/**
 * An error indicating that the property specified was invalid.
 */
class InvalidProperty
{
};


/**
 * An error indicating that the object specified was not found.
 */
class ObjectNotFound
{
};


} /* namespace Error */


} /* namespace org.freedesktop.Galago */
