/**
 * @file libgalago/galago-context-priv.h Galago Context Private API
 * @ingroup Presence
 *
 * @Copyright (C) 2004-2006 Christian Hammond
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#ifndef _GALAGO_CONTEXT_PRIV_H_
#define _GALAGO_CONTEXT_PRIV_H_

typedef struct _GalagoContextOps GalagoContextOps;

#include <libgalago/galago-person.h>
#include <libgalago/galago-service.h>

/**
 * The context operations.
 */
struct _GalagoContextOps
{
	/** Called when a service is added. */
	void (*service_added)(GalagoService *service);
	/** Called when a service is removed. */
	void (*service_removed)(GalagoService *service);
	/** Called when a person is added. */
	void (*person_added)(GalagoPerson *person);
	/** Called when a person is removed. */
	void (*person_removed)(GalagoPerson *person);
};

G_BEGIN_DECLS

/**************************************************************************/
/** @name Galago Context API                                              */
/**************************************************************************/
/*@{*/

/**
 * Sets the context ops to use for a context.
 *
 * @param context The context.
 * @param ops     The ops structure.
 */
void galago_context_set_ops(GalagoContext *context, GalagoContextOps *ops);

/**
 * Adds a service to the context's services cache.
 *
 * @param service The service to add.
 */
void galago_context_add_service(GalagoService *service);

/**
 * Removes a service from the context's services cache.
 *
 * @param service The service to remove.
 */
void galago_context_remove_service(GalagoService *service);

/**
 * Adds a person to the context's person cache.
 *
 * @param person The person to add.
 */
void galago_context_add_person(GalagoPerson *person);

/**
 * Removes a person from the context's person cache.
 *
 * @param person The person to remove.
 */
void galago_context_remove_person(GalagoPerson *person);

/**
 * Adds an object to the internal object tree.
 *
 * @internal This is used internally.
 *
 * @param obj The object to add.
 */
void galago_context_add_object(GalagoObject *obj);

/**
 * Removes an object from the internal object tree.
 *
 * @internal This is used internally.
 *
 * @param obj The object to remove.
 */
void galago_context_remove_object(GalagoObject *obj);

/**
 * Clears all objects of a specified origin from a context.
 *
 * @internal This is used internally.
 *
 * @param origin The origin of objects to clear.
 */
void galago_context_clear_objects(GalagoOrigin origin);

/*@}*/

G_END_DECLS

#endif /* _GALAGO_CONTEXT_PRIV_H_ */
