/*
 * This file is part of maemo-af-desktop
 *
 * Copyright (C) 2006 Nokia Corporation.
 *
 * Author:  Moises Martinez <moises.martinez@nokia.com>
 * Contact: Karoliina Salminen <karoliina.t.salminen@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include "hildon-desktop-window.h" 

#define HILDON_DESKTOP_WINDOW_GET_PRIVATE(object) \
        (G_TYPE_INSTANCE_GET_PRIVATE ((object), HILDON_DESKTOP_TYPE_WINDOW, HildonDesktopWindowPrivate))

G_DEFINE_TYPE (HildonDesktopWindow, hildon_desktop_window, GTK_TYPE_WINDOW);

enum {
  PROP_0,
  PROP_CONFIG_FILE,
  PROP_PLUGIN_DIR
};

enum {
  SIGNAL_SELECT_PLUGINS,
  SIGNAL_SAVE,
  SIGNAL_LOAD,
  N_SIGNALS
};

static gint signals[N_SIGNALS];  

struct _HildonDesktopWindowPrivate 
{
  gchar *config_file;
  gchar *plugin_dir;
};

static void 
hildon_desktop_window_finalize (GObject *object)
{
  HildonDesktopWindow *desktop_window;
  HildonDesktopWindowPrivate *priv;

  desktop_window = HILDON_DESKTOP_WINDOW (object);

  priv = HILDON_DESKTOP_WINDOW_GET_PRIVATE (desktop_window);

  if (desktop_window->container)
  {
    g_object_unref (desktop_window->container);
    desktop_window = NULL;
  }

  if (priv->config_file)
  {
    g_free (priv->config_file);
    priv->config_file = NULL;
  }

  if (priv->plugin_dir)
  {
    g_free (priv->plugin_dir);
    priv->plugin_dir = NULL;
  }

  G_OBJECT_CLASS (hildon_desktop_window_parent_class)->finalize (object);
}

static void
hildon_desktop_window_set_property (GObject      *object,
                                    guint         prop_id,
                                    const GValue *value,
                                    GParamSpec   *pspec)
{
  switch (prop_id)
  {
    case PROP_CONFIG_FILE:
      hildon_desktop_window_set_config_file (
                               HILDON_DESKTOP_WINDOW (object),
                               (gchar *) g_value_get_string (value));
      break;

    case PROP_PLUGIN_DIR:
      hildon_desktop_window_set_plugin_dir (
                               HILDON_DESKTOP_WINDOW (object),
                               (gchar *) g_value_get_string (value));
      break;

    default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
  }
}

static void
hildon_desktop_window_get_property (GObject      *object,
                                    guint         prop_id,
                                    GValue       *value,
                                    GParamSpec   *pspec)
{
  HildonDesktopWindowPrivate *priv;
 
  priv = HILDON_DESKTOP_WINDOW_GET_PRIVATE (object);

  switch (prop_id)
  {
    case PROP_CONFIG_FILE:
        g_value_set_string (value, priv->config_file);
        break;

    case PROP_PLUGIN_DIR:
        g_value_set_string (value, priv->plugin_dir);
        break;

    default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
  }
}

static void 
hildon_desktop_window_init (HildonDesktopWindow *desktop_window)
{
  desktop_window->priv = HILDON_DESKTOP_WINDOW_GET_PRIVATE (desktop_window);

  desktop_window->container = NULL;

  desktop_window->priv->plugin_dir = NULL;
}

static void
hildon_desktop_window_class_init (HildonDesktopWindowClass *desktopwindow_class)
{
  GParamSpec *pspec;
  GObjectClass *g_object_class = G_OBJECT_CLASS (desktopwindow_class);

  g_object_class->finalize     = hildon_desktop_window_finalize;
  g_object_class->set_property = hildon_desktop_window_set_property;
  g_object_class->get_property = hildon_desktop_window_get_property;

  signals[SIGNAL_SELECT_PLUGINS] =
        g_signal_new ("select-plugins",
                      G_OBJECT_CLASS_TYPE (g_object_class),
                      G_SIGNAL_RUN_FIRST,
		      G_STRUCT_OFFSET (HildonDesktopWindowClass, select_plugins),
                      NULL, NULL,
                      g_cclosure_marshal_VOID__VOID,
                      G_TYPE_NONE, 0);

  signals[SIGNAL_SAVE] =
        g_signal_new ("save",
                      G_OBJECT_CLASS_TYPE (g_object_class),
                      G_SIGNAL_RUN_FIRST,
		      G_STRUCT_OFFSET (HildonDesktopWindowClass, save),
                      NULL, NULL,
                      g_cclosure_marshal_VOID__VOID,
                      G_TYPE_NONE, 0);

  signals[SIGNAL_LOAD] =
        g_signal_new ("load",
                      G_OBJECT_CLASS_TYPE (g_object_class),
                      G_SIGNAL_RUN_FIRST,
		      G_STRUCT_OFFSET (HildonDesktopWindowClass, load),
                      NULL, NULL,
                      g_cclosure_marshal_VOID__VOID,
                      G_TYPE_NONE, 0);

  pspec = g_param_spec_pointer ("plugin-dir",
                                "Plugin Directory",
                                "Plugin Directory",
                                G_PARAM_READWRITE);

  g_object_class_install_property (g_object_class,
                                   PROP_PLUGIN_DIR,
                                   pspec);

  pspec = g_param_spec_pointer ("config-file",
                                "Config File",
                                "Config File",
                                G_PARAM_READWRITE);

  g_object_class_install_property (g_object_class,
                                   PROP_CONFIG_FILE,
                                   pspec);

  g_type_class_add_private (g_object_class, sizeof (HildonDesktopWindowPrivate));
}

void 
hildon_desktop_window_set_sensitive (HildonDesktopWindow *window, gboolean sensitive)
{
/* TODO: */
}	

void 
hildon_desktop_window_set_focus (HildonDesktopWindow *window, gboolean focus)
{
/* TODO: */
}

const gchar * 
hildon_desktop_window_get_config_file (HildonDesktopWindow *window)
{
  HildonDesktopWindowPrivate *priv;

  g_return_val_if_fail (window != NULL, NULL);
  g_return_val_if_fail (HILDON_DESKTOP_IS_WINDOW (window), NULL);

  priv = HILDON_DESKTOP_WINDOW_GET_PRIVATE (window);

  return priv->config_file;
}

void
hildon_desktop_window_set_config_file (HildonDesktopWindow *window, 
                                       gchar               *config_file)
{
  HildonDesktopWindowPrivate *priv;

  g_return_if_fail (window != NULL);
  g_return_if_fail (HILDON_DESKTOP_IS_WINDOW (window));

  priv = HILDON_DESKTOP_WINDOW_GET_PRIVATE (window);

  if (priv->config_file)
    g_free (priv->config_file);

  priv->config_file = g_strdup (config_file);
}

const gchar * 
hildon_desktop_window_get_plugin_dir (HildonDesktopWindow *window)
{
  HildonDesktopWindowPrivate *priv;

  g_return_val_if_fail (window != NULL, NULL);
  g_return_val_if_fail (HILDON_DESKTOP_IS_WINDOW (window), NULL);

  priv = HILDON_DESKTOP_WINDOW_GET_PRIVATE (window);

  return priv->plugin_dir;
}

void
hildon_desktop_window_set_plugin_dir (HildonDesktopWindow *window, 
                                      gchar               *plugin_dir)
{
  HildonDesktopWindowPrivate *priv;

  g_return_if_fail (window != NULL);
  g_return_if_fail (HILDON_DESKTOP_IS_WINDOW (window));

  priv = HILDON_DESKTOP_WINDOW_GET_PRIVATE (window);

  if (priv->plugin_dir)
    g_free (priv->plugin_dir);

  priv->plugin_dir = g_strdup (plugin_dir);
}
