/* -*- mode:C; c-file-style:"gnu"; -*- */
/*
 * This file is part of hildon-desktop
 *
 * Copyright (C) 2006, 2007 Nokia Corporation.
 *
 * Author:  Johan Bilien <johan.bilien@nokia.com>
 * Contact: Karoliina Salminen <karoliina.t.salminen@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <libosso.h>
#include <osso-helplib.h>
#include <hildon-widgets/hildon-file-chooser-dialog.h>
#include <hildon-widgets/hildon-color-button.h>
#include <hildon-widgets/hildon-caption.h>
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif


#include <string.h>

#include <glib.h>
#include <glib/gi18n.h>

#include <gtk/gtktreemodel.h>
#include <gtk/gtkcombobox.h>
#include <gtk/gtkliststore.h>
#include <gtk/gtkhbox.h>
#include <gtk/gtkcellrenderertext.h>
#include <gtk/gtkcelllayout.h>

#include <libosso.h>
#include <osso-helplib.h>
#include <hildon-widgets/hildon-file-chooser-dialog.h>
#include <hildon-widgets/hildon-color-button.h>
#include <hildon-widgets/hildon-caption.h>

#include "hd-home-background-dialog.h"
#include "hd-home-l10n.h"
#include "background-manager/hildon-background-manager.h"

#define BG_DESKTOP_GROUP           "Desktop Entry"
#define BG_DESKTOP_IMAGE_NAME      "Name"
#define BG_DESKTOP_IMAGE_FILENAME  "File"
#define BG_DESKTOP_IMAGE_PRIORITY  "X-Order"
#define BG_IMG_INFO_FILE_TYPE      "desktop"

#define HOME_BG_IMG_DEFAULT_PRIORITY  15327 /* this is a random number */

#define HILDON_HOME_HC_USER_IMAGE_DIR    "MyDocs/.images"

#define HILDON_HOME_FILE_CHOOSER_ACTION_PROP  "action"
#define HILDON_HOME_FILE_CHOOSER_TITLE_PROP   "title"
#define HILDON_HOME_FILE_CHOOSER_TITLE        _("home_ti_select_image")
#define HILDON_HOME_FILE_CHOOSER_SELECT_PROP  "open-button-text"
#define HILDON_HOME_FILE_CHOOSER_SELECT       _("home_bd_select_image")
#define HILDON_HOME_FILE_CHOOSER_EMPTY_PROP   "empty-text"
#define HILDON_HOME_FILE_CHOOSER_EMPTY        _("home_li_no_images")

#define HILDON_HOME_SET_BG_RESPONSE_IMAGE    GTK_RESPONSE_YES
#define HILDON_HOME_SET_BG_RESPONSE_PREVIEW  GTK_RESPONSE_APPLY

#define HILDON_HOME_BACKGROUND_NO_IMAGE  "no_image"


#define HD_HOME_BACKGROUND_DIALOG_GET_PRIVATE(obj) \
        (G_TYPE_INSTANCE_GET_PRIVATE ((obj), HD_TYPE_HOME_BACKGROUND_DIALOG, HDHomeBackgroundDialogPrivate))

G_DEFINE_TYPE (HDHomeBackgroundDialog, hd_home_background_dialog, GTK_TYPE_DIALOG);

enum 
{ 	 
  BG_IMAGE_NAME, 	 
  BG_IMAGE_FILENAME, 	 
  BG_IMAGE_PRIORITY
}; 

enum
{
  PROP_0,
  PROP_BACKGROUND_DIR
};

struct _HDHomeBackgroundDialogPrivate
{
  gchar            *background_dir;
  GtkListStore     *combobox_contents;
  GtkTreeIter       active_iter;
  GtkWidget        *color_button;
  GtkWidget        *img_combo;
  GtkWidget        *mode_combo;
};

static void
hd_home_background_dialog_set_property (GObject      *gobject,
                                        guint         prop_id,
                                        const GValue *value,
                                        GParamSpec   *pspec);

static void
hd_home_background_dialog_get_property (GObject      *gobject,
                                        guint         prop_id,
                                        GValue *value,
                                        GParamSpec   *pspec);


static void
hd_home_background_dialog_response (GtkDialog *dialog, 
                                    gint       arg);
static void
hd_home_background_dialog_set_background_dir (HDHomeBackgroundDialog *dialog,
                                              const gchar *dir);

static void
hd_home_background_dialog_background_dir_changed 
                                    (HDHomeBackgroundDialog *dialog);


static void
hd_home_background_dialog_class_init (HDHomeBackgroundDialogClass *klass)
{
  GParamSpec       *pspec;
  GObjectClass     *object_class;
  GtkDialogClass   *dialog_class;

  object_class = G_OBJECT_CLASS (klass);
  dialog_class = GTK_DIALOG_CLASS (klass);

  object_class->set_property = hd_home_background_dialog_set_property;
  object_class->get_property = hd_home_background_dialog_get_property;

  dialog_class->response = hd_home_background_dialog_response;

  klass->background_dir_changed = hd_home_background_dialog_background_dir_changed;

  g_signal_new ("background-dir-changed",
                G_OBJECT_CLASS_TYPE (object_class),
                G_SIGNAL_RUN_FIRST,
                G_STRUCT_OFFSET (HDHomeBackgroundDialogClass,
                                 background_dir_changed),
                NULL,
                NULL,
                g_cclosure_marshal_VOID__VOID,
                G_TYPE_NONE,
                0);

  pspec = g_param_spec_string ("background-dir",
                               "Background directory",
                               "Directory where to look for default backgrounds",
                               "",
                               G_PARAM_READWRITE | G_PARAM_CONSTRUCT);

  g_object_class_install_property (object_class,
                                   PROP_BACKGROUND_DIR,
                                   pspec);

  g_type_class_add_private (klass, sizeof (HDHomeBackgroundDialogPrivate));

}

static void
hd_home_background_dialog_init (HDHomeBackgroundDialog *dialog)
{
  HDHomeBackgroundDialogPrivate    *priv;
  GtkWidget                        *hbox_color;
  GtkWidget                        *hbox_image;
  GtkWidget                        *hbox_mode;
  GtkWidget                        *color_caption;
  GtkWidget                        *image_caption;
  GtkWidget                        *mode_caption;
  GtkCellRenderer                  *renderer;
  GtkSizeGroup                     *group;
  const gchar                      *image_modes[] = { 
    HH_SET_BG_MODE_CENTERED,
    HH_SET_BG_MODE_SCALED,
    HH_SET_BG_MODE_STRETCHED,
    HH_SET_BG_MODE_TILED
  };

  priv = HD_HOME_BACKGROUND_DIALOG_GET_PRIVATE (dialog);
  
  gtk_dialog_add_buttons (GTK_DIALOG (dialog),
                          HH_SET_BG_OK,
                          GTK_RESPONSE_OK,
                          HH_SET_BG_PREVIEW,
                          HILDON_HOME_SET_BG_RESPONSE_PREVIEW,
                          HH_SET_BG_IMAGE,
                          HILDON_HOME_SET_BG_RESPONSE_IMAGE,
                          HH_SET_BG_CANCEL,
                          GTK_RESPONSE_CANCEL,
                          NULL);

  gtk_dialog_set_has_separator (GTK_DIALOG (dialog), FALSE);
  gtk_window_set_title (GTK_WINDOW (dialog), HH_SET_BG_TITLE);

#if 0
  /* Add help button */
  ossohelp_dialog_help_enable (GTK_DIALOG (dialog),
                               HH_HELP_SET_BACKGROUND,
                               home_get_osso_context ());
#endif

  /* Hildon Caption HBoxes */
  hbox_color = gtk_hbox_new (FALSE, 10);
  hbox_image = gtk_hbox_new (FALSE, 10);
  hbox_mode  = gtk_hbox_new (FALSE, 10);
     
#if 0
  /* Widgets for Hildon Captions in Set Background Image dialog */
  background_manager_get_color (bm, &current_color);
  priv->color_button = hildon_color_button_new_with_color (&current_color);
#endif
  
  priv->color_button = hildon_color_button_new ();
  
  priv->img_combo =
    gtk_combo_box_new_with_model (GTK_TREE_MODEL (priv->combobox_contents));
  
#if 0
  if(!background_manager_get_image_uri (bm) ||
     g_str_equal(background_manager_get_image_uri (bm),
		 HILDON_HOME_BACKGROUND_NO_IMAGE))
#endif
  
  renderer = gtk_cell_renderer_text_new ();
  gtk_cell_layout_pack_start (GTK_CELL_LAYOUT (priv->img_combo),
                              renderer, 
                              TRUE);

  gtk_cell_layout_set_attributes (GTK_CELL_LAYOUT (priv->img_combo),
                                  renderer, "text", 
                                  BG_IMAGE_NAME, NULL);
  
  priv->mode_combo = gtk_combo_box_new_text ();
  
  gtk_combo_box_append_text (GTK_COMBO_BOX (priv->mode_combo), 
                             image_modes[BACKGROUND_CENTERED]);
  gtk_combo_box_append_text (GTK_COMBO_BOX (priv->mode_combo), 
                             image_modes[BACKGROUND_SCALED]);
  gtk_combo_box_append_text (GTK_COMBO_BOX (priv->mode_combo), 
                             image_modes[BACKGROUND_STRETCHED]);
  gtk_combo_box_append_text (GTK_COMBO_BOX (priv->mode_combo), 
                             image_modes[BACKGROUND_TILED]);
  
#if 0
  /* Centered is a default scaling value */
  gtk_combo_box_set_active (GTK_COMBO_BOX (combobox_mode_select), 
			    background_manager_get_mode (bm));
#endif

  /* Hildon captions in Set Background Image dialog */
  group = GTK_SIZE_GROUP (gtk_size_group_new (GTK_SIZE_GROUP_HORIZONTAL));
  color_caption =
      hildon_caption_new (group,
                          HH_SET_BG_COLOR_TITLE,  
                          priv->color_button,
                          NULL,
                          HILDON_CAPTION_OPTIONAL);
  image_caption = 
      hildon_caption_new (group, 
                          HH_SET_BG_IMAGE_TITLE,
                          priv->img_combo, 
                          NULL, 
                          HILDON_CAPTION_OPTIONAL);
  mode_caption =
      hildon_caption_new (group,
                          HH_SET_BG_MODE_TITLE,
                          priv->mode_combo,
                          NULL,
                          HILDON_CAPTION_OPTIONAL);
     
  gtk_box_pack_start (GTK_BOX (hbox_color), color_caption,
                      FALSE, FALSE, 0); 
  gtk_box_pack_start (GTK_BOX (hbox_image), image_caption,
                      TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (hbox_mode), mode_caption,
                      TRUE, TRUE, 0);

  gtk_box_set_spacing (GTK_BOX (GTK_DIALOG (dialog)->vbox), 10);
  gtk_container_add (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox), hbox_color);
  gtk_container_add (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox), hbox_image);
  gtk_container_add (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox), hbox_mode);

  hildon_caption_set_child_expand (HILDON_CAPTION (image_caption), TRUE);
  hildon_caption_set_child_expand (HILDON_CAPTION (mode_caption), TRUE);
  
  /* Let the WM take care of the positioning */
  gtk_window_set_position (GTK_WINDOW (dialog), GTK_WIN_POS_NONE);
}

static void
hd_home_background_dialog_set_property (GObject      *gobject,
                                        guint         prop_id,
                                        const GValue *value,
                                        GParamSpec   *pspec)
{
  switch (prop_id)
  {
    case PROP_BACKGROUND_DIR:
        hd_home_background_dialog_set_background_dir (
                          HD_HOME_BACKGROUND_DIALOG (gobject),
                          g_value_get_string (value));
        break;
    default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID (gobject, prop_id, pspec);
        break;
  }
}

static void
hd_home_background_dialog_get_property (GObject      *gobject,
                                        guint         prop_id,
                                        GValue       *value,
                                        GParamSpec   *pspec)
{
  HDHomeBackgroundDialogPrivate    *priv;

  priv = HD_HOME_BACKGROUND_DIALOG_GET_PRIVATE (gobject);

  switch (prop_id)
  {
    case PROP_BACKGROUND_DIR:
        g_value_set_string (value, priv->background_dir);
        break;
    default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID (gobject, prop_id, pspec);
        break;
  }
}

#if 0
static gboolean
home_bgd_select_file_dialog (GtkWidget   * parent,
			     GtkComboBox * box,
			     gboolean      built_in)
{
  GtkTreeModel *tree = gtk_combo_box_get_model (box);
  GtkWidget *dialog;
  GtkFileFilter *mime_type_filter;
  gint response;
  gchar *name_str, *dot;
  gchar *chooser_name;
  gchar *image_dir;
  
  dialog =
    hildon_file_chooser_dialog_new_with_properties(
					  GTK_WINDOW(parent),
					  HILDON_HOME_FILE_CHOOSER_ACTION_PROP,
					  GTK_FILE_CHOOSER_ACTION_OPEN,
					  HILDON_HOME_FILE_CHOOSER_TITLE_PROP,
					  HILDON_HOME_FILE_CHOOSER_TITLE,
					  HILDON_HOME_FILE_CHOOSER_SELECT_PROP,
					  HILDON_HOME_FILE_CHOOSER_SELECT,
					  HILDON_HOME_FILE_CHOOSER_EMPTY_PROP,
					  HILDON_HOME_FILE_CHOOSER_EMPTY,
					  NULL);

#if 0
  /* Add help button */
  ossohelp_dialog_help_enable(GTK_DIALOG(dialog), 
			      HILDON_HOME_SELECT_IMAGE_HELP_TOPIC,
			      home_get_osso_context ());
#endif
        
  mime_type_filter = gtk_file_filter_new();
  gtk_file_filter_add_mime_type (mime_type_filter, "image/jpeg");
  gtk_file_filter_add_mime_type (mime_type_filter, "image/gif");
  gtk_file_filter_add_mime_type (mime_type_filter, "image/png");
  gtk_file_filter_add_mime_type (mime_type_filter, "image/bmp");
  gtk_file_filter_add_mime_type (mime_type_filter, "image/tiff");
  gtk_file_filter_add_mime_type (mime_type_filter, "sketch/png");

  gtk_file_chooser_set_filter (GTK_FILE_CHOOSER (dialog),
			       mime_type_filter);

  image_dir = g_build_path("/",
                           g_getenv ("HOME"),
                           HILDON_HOME_HC_USER_IMAGE_DIR,
                           NULL);

  if(!gtk_file_chooser_set_current_folder(GTK_FILE_CHOOSER (dialog),
                                          image_dir))
    g_warning ("Couldn't set default image dir for dialog %s", 
               image_dir);

  g_free (image_dir);
  
  response = gtk_dialog_run (GTK_DIALOG (dialog));
    
  if (response == GTK_RESPONSE_OK) 
    {

      chooser_name = 
          gtk_file_chooser_get_uri (GTK_FILE_CHOOSER (dialog));

      name_str = g_filename_from_uri (chooser_name, NULL, NULL);
      if (!name_str)
        name_str = g_strdup (chooser_name);

      /* remove extension */
      if ((dot = g_strrstr (name_str, ".")) && dot != name_str)
        *dot = '\0';

      if (name_str && *name_str) 
        {
          GtkTreeIter iter;            
          GtkTreePath *path;
          gint priority = 0;
          gtk_list_store_append (GTK_LIST_STORE (tree), &iter);

          gtk_list_store_set (GTK_LIST_STORE (tree),
                             &iter,
                             BG_IMAGE_NAME, name_str,
                             BG_IMAGE_FILENAME, chooser_name,
                             BG_IMAGE_PRIORITY, G_MAXINT, -1);

          gtk_combo_box_set_active_iter (box, &iter);

          path = gtk_tree_model_get_path (tree, &iter);
          gtk_tree_path_next (path);

          if (gtk_tree_model_get_iter (tree, &iter, path)) 
            {
              priority = home_bgd_get_priority (tree, &iter);
              if (priority == G_MAXINT) 
                {
                  gtk_list_store_remove(GTK_LIST_STORE(tree), &iter);

                }
            }
          gtk_tree_path_free(path);
        }

      /* Clean up */
      g_free(name_str);
      gtk_widget_destroy(dialog);    

      return TRUE;
    }  

  gtk_widget_destroy(dialog);

  return FALSE;
}
#endif

static void
hd_home_background_dialog_response (GtkDialog *dialog, 
                                    gint       arg)
{
  HDHomeBackgroundDialogPrivate *priv;

  priv = HD_HOME_BACKGROUND_DIALOG_GET_PRIVATE (dialog);

#if 0
  g_debug ("response_data = {\n"
           "  img_combo    .= %s\n"
           "  mode_combo   .= %s\n"
           "  color_button .= %s\n"
           "  applied      .= %s\n"
           "  built_in     .= %s\n"
           "  has_preview  .= %s\n"
           "}",
           g_type_name (G_OBJECT_TYPE (d->img_combo)),
           g_type_name (G_OBJECT_TYPE (d->mode_combo)),
           g_type_name (G_OBJECT_TYPE (d->color_button)),
           d->applied ? "<true>" : "<false>",
           d->built_in ? "<true>" : "<false>",
           d->has_preview ? "<true>" : "<false>");
#endif

  switch (arg) 
    {
      case HILDON_HOME_SET_BG_RESPONSE_IMAGE:
      /*    home_bgd_select_file_dialog (dialog, img_combo, d->built_in);*/
          break;
      case HILDON_HOME_SET_BG_RESPONSE_PREVIEW:
          break;
      case GTK_RESPONSE_OK:
          gtk_widget_hide (GTK_WIDGET(dialog));
          break;
      case GTK_RESPONSE_CANCEL:
      case GTK_RESPONSE_DELETE_EVENT:
          gtk_widget_hide (GTK_WIDGET (dialog));
          break;
      default:
          break;
    }

}


static void
hd_home_background_dialog_background_dir_changed 
                                    (HDHomeBackgroundDialog *dialog)
{
  HDHomeBackgroundDialogPrivate *priv;
  GDir                          *dir = NULL;
  GError                        *error = NULL;
  GtkTreeIter                    iterator;
  const gchar                   *image_desc_file = NULL;
  GKeyFile                      *kfile;

  g_return_if_fail (HD_IS_HOME_BACKGROUND_DIALOG (dialog) && dir);
  priv = HD_HOME_BACKGROUND_DIALOG_GET_PRIVATE (dialog);

  if (priv->combobox_contents)
    gtk_list_store_clear (priv->combobox_contents);
  else
    priv->combobox_contents =
      gtk_list_store_new (3,
                          G_TYPE_STRING, /*localised descriptive name */
                          G_TYPE_STRING, /* image file path & name */
                          G_TYPE_INT     /* image priority */);

  if (!priv->background_dir)
    return;

  dir = g_dir_open (priv->background_dir,
                    0,
                    &error);

  if (error)
  {
    g_warning ("Could not open background directory %s: %s",
               priv->background_dir,
               error->message);
    g_error_free (error);
    return;
  }


  /* No background file option */
  gtk_list_store_append (priv->combobox_contents, &iterator);
  gtk_list_store_set (priv->combobox_contents,
                      &iterator,
                      BG_IMAGE_NAME, HH_SET_BG_IMAGE_NONE,
                      BG_IMAGE_FILENAME, NULL,
                      BG_IMAGE_PRIORITY, 0, -1);

  kfile = g_key_file_new ();

  while ((image_desc_file = g_dir_read_name (dir)))
    {
      gchar *image_name = NULL;
      gchar *image_path = NULL;
      gchar *filename   = NULL;
      gint   image_order = 0;
      
      if (!g_str_has_suffix (image_desc_file, BG_IMG_INFO_FILE_TYPE)) 
        continue;

      error = NULL;

      filename = g_build_filename (priv->background_dir,
                                   image_desc_file,
                                   NULL);

      if (!g_key_file_load_from_file (kfile,
                                      filename,
                                      G_KEY_FILE_NONE,
                                      &error))
        goto end_of_loop;
      

      image_name = g_key_file_get_string (kfile,
                                          BG_DESKTOP_GROUP,
                                          BG_DESKTOP_IMAGE_NAME,
                                          &error);

      if (!image_name || error) goto end_of_loop;

      image_path = g_key_file_get_string (kfile,
                                          BG_DESKTOP_GROUP,
                                          BG_DESKTOP_IMAGE_FILENAME,
                                          &error);

      if (!image_path || error) goto end_of_loop;

      image_order = g_key_file_get_integer (kfile,
                                            BG_DESKTOP_GROUP,
                                            BG_DESKTOP_IMAGE_PRIORITY,
                                            &error);

      if (error) goto end_of_loop;

#if 0
      current_uri = background_manager_get_image_uri (bm);

      if (current_uri)
        {
          gchar *current_path;
          GError *error = NULL;

          current_path = g_filename_from_uri (current_uri, NULL, &error);
          if (error)
            {
              g_warning ("Unable to get the filename from uri `%s': %s",
                         current_uri,
                         error->message);

              g_error_free (error);

              /* we let the background manager deal with it */
              current_path = g_strdup (current_uri);
            }

          g_debug ("current filename [%s], processing [%s]",
                   current_path, image_path);

          if (image_path &&
              g_str_equal(image_path, current_path)) 
            {
              g_debug ("current BG uses built in image");
              resp_data->built_in = TRUE;
            }

          g_free (current_path);
        }
#endif

      gtk_list_store_append (priv->combobox_contents, &iterator);

      gtk_list_store_set (priv->combobox_contents,
                          &iterator,
                          BG_IMAGE_NAME,
                          /* work around a strange behavior of gettext for
                           * empty strings */
                          ((image_name && *image_name) ? _(image_name) : image_path),
                          BG_IMAGE_FILENAME, image_path,
                          BG_IMAGE_PRIORITY, image_order,
                          -1);

end_of_loop:
      if (error)
        g_warning ("Error when reading background descriptions: %s",
                   error->message);
      g_clear_error (&error);

      g_free (filename);
      g_free (image_name);
      g_free (image_path);
    } 

  if (dir) 
    g_dir_close (dir);             
      
  g_key_file_free (kfile);

  gtk_tree_sortable_set_sort_column_id (GTK_TREE_SORTABLE (priv->combobox_contents),
                                        BG_IMAGE_PRIORITY, 
                                        GTK_SORT_ASCENDING);
}

static void
hd_home_background_dialog_set_background_dir (HDHomeBackgroundDialog *dialog,
                                              const gchar *dir)
{
  HDHomeBackgroundDialogPrivate *priv;

  g_return_if_fail (HD_IS_HOME_BACKGROUND_DIALOG (dialog) && dir);
  priv = HD_HOME_BACKGROUND_DIALOG_GET_PRIVATE (dialog);

  if (!priv->background_dir 
      || !g_str_equal (dir, priv->background_dir))
  {
    g_free (priv->background_dir);

    priv->background_dir = g_strdup (dir);
    g_object_notify (G_OBJECT (dialog), "background-dir");
    g_signal_emit_by_name (dialog, "background-dir-changed");
  }
}

