/*
 * This file is part of hildon-desktop
 *
 * Copyright (C) 2006 Nokia Corporation.
 *
 * Author:  Johan Bilien <johan.bilien@nokia.com>
 * Contact: Karoliina Salminen <karoliina.t.salminen@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include "hd-plugin-loader-legacy.h"
#include <libhildondesktop/hildon-home-applet.h>
#include <libhildondesktop/tasknavigator-item-wrapper.h>
#include <glib/gkeyfile.h>
#include <string.h> /* strcmp */

#define HD_PLUGIN_LOADER_LEGACY_GET_PRIVATE(obj) \
        (G_TYPE_INSTANCE_GET_PRIVATE ((obj), HD_TYPE_PLUGIN_LOADER_LEGACY, HDPluginLoaderLegacyPrivate))

G_DEFINE_TYPE (HDPluginLoaderLegacy, hd_plugin_loader_legacy, HD_TYPE_PLUGIN_LOADER);

#define HD_PLUGIN_LOADER_LEGACY_GROUP           "Desktop Entry"
#define HD_PLUGIN_LOADER_LEGACY_KEY_TYPE        "Type"
#define HD_PLUGIN_LOADER_LEGACY_KEY_X_HOME_APPLET           "X-home-applet"
#define HD_PLUGIN_LOADER_LEGACY_KEY_X_HOME_APPLET_RESIZABLE "X-home-applet-resizable"
#define HD_PLUGIN_LOADER_LEGACY_KEY_X_TN        "X-task-navigator-plugin"

/* Type of legacy plugin */
typedef enum {
  HD_PLUGIN_LOADER_LEGACY_TYPE_UNKNOWN = 0,
  HD_PLUGIN_LOADER_LEGACY_TYPE_HOME,
  HD_PLUGIN_LOADER_LEGACY_TYPE_STATUS_BAR,
  HD_PLUGIN_LOADER_LEGACY_TYPE_NAVIGATOR
} HDPluginLoaderLegacyType;

/* Used to map plugin symbols to object signals */
typedef struct _SymbolMapping {
  const gchar *symbol_name;
  const gchar *signal;
} SymbolMapping;

typedef GtkWidget * (*HDPluginLoaderLegacyInitFunc) (HDPluginLoaderLegacy *l,
                                                     GError **error);

typedef struct _HDPluginLoaderLegacyPrivate
{
  GModule                      *module;
  HDPluginLoaderLegacyType      plugin_type;
  HDPluginLoaderLegacyInitFunc  init_func;
  SymbolMapping                *symbol_mapping;
  gpointer                      module_data;
  const gchar                  *library_key;
  const gchar                  *module_dir;

} HDPluginLoaderLegacyPrivate;

static const SymbolMapping home_symbol_mapping[] = 
        { {"hildon_home_applet_lib_initialize", NULL},
          {"hildon_home_applet_lib_background", "background"},
          {"hildon_home_applet_lib_foreground", "foreground"},
          {"hildon_home_applet_lib_settings", "settings"},
          {"hildon_home_applet_lib_deinitialize", "destroy"},
          {NULL, NULL}};
enum {
  HD_PLUGIN_LOADER_LEGACY_HOME_SYMBOL_INITIALIZE = 0,
  HD_PLUGIN_LOADER_LEGACY_HOME_SYMBOL_BACKGROUND,
  HD_PLUGIN_LOADER_LEGACY_HOME_SYMBOL_FOREGROUND,
  HD_PLUGIN_LOADER_LEGACY_HOME_SYMBOL_SETTINGS,
  HD_PLUGIN_LOADER_LEGACY_HOME_SYMBOL_DESTROY,
  HD_PLUGIN_LOADER_LEGACY_HOME_N_SYMBOLS
};

static const SymbolMapping navigator_symbol_mapping[] = 
        { {"hildon_navigator_lib_create", NULL},
          {"hildon_navigator_lib_initialize_menu", NULL},
          {"hildon_navigator_lib_get_button_widget", NULL},
          {"hildon_navigator_lib_destroy", "destroy"},
          {NULL, NULL}};
enum {
  HD_PLUGIN_LOADER_LEGACY_NAVIGATOR_SYMBOL_CREATE = 0,
  HD_PLUGIN_LOADER_LEGACY_NAVIGATOR_SYMBOL_INITIALIZE_MENU,
  HD_PLUGIN_LOADER_LEGACY_NAVIGATOR_SYMBOL_GET_BUTTON,
  HD_PLUGIN_LOADER_LEGACY_NAVIGATOR_SYMBOL_DESTROY,
  HD_PLUGIN_LOADER_LEGACY_NAVIGATOR_N_SYMBOLS
};

static GtkWidget *
hd_plugin_loader_legacy_load (HDPluginLoader *loader, GError **error);

static void
hd_plugin_loader_legacy_guess_type (HDPluginLoaderLegacy *loader);

static GtkWidget *
hd_plugin_loader_legacy_home_init (HDPluginLoaderLegacy *loader,
                                   GError **error);

static GtkWidget *
hd_plugin_loader_legacy_navigator_init (HDPluginLoaderLegacy *loader,
                                        GError **error);
  
static void
hd_plugin_loader_legacy_connect_signals (HDPluginLoaderLegacy *loader,
                                         GtkWidget *item,
                                         GError **error);

static void
hd_plugin_loader_legacy_open_module (HDPluginLoaderLegacy *loader,
                                     GError **error);

static void
hd_plugin_loader_legacy_finalize (GObject *loader)
{
  G_OBJECT_CLASS (hd_plugin_loader_legacy_parent_class)->finalize (loader);
}

static void
hd_plugin_loader_legacy_init (HDPluginLoaderLegacy *loader)
{
}

static void
hd_plugin_loader_legacy_class_init (HDPluginLoaderLegacyClass *class)
{
  GObjectClass *object_class;
  HDPluginLoaderClass *loader_class;

  object_class = G_OBJECT_CLASS (class);
  loader_class = HD_PLUGIN_LOADER_CLASS (class);
  
  object_class->finalize = hd_plugin_loader_legacy_finalize;
  loader_class->load = hd_plugin_loader_legacy_load;

  g_type_class_add_private (object_class, sizeof (HDPluginLoaderLegacyPrivate));
}

static GtkWidget *
hd_plugin_loader_legacy_load (HDPluginLoader *loader, GError **error)
{
  HDPluginLoaderLegacyPrivate      *priv;
  GKeyFile                         *keyfile;
  GtkWidget                        *item = NULL;
  GError                           *local_error = NULL;

  g_return_val_if_fail (loader, NULL);
 
  priv = HD_PLUGIN_LOADER_LEGACY_GET_PRIVATE (loader);

  keyfile = hd_plugin_loader_get_key_file (loader);

  if (!keyfile)
    {
      g_set_error (error,
                   hd_plugin_loader_error_quark (),
                   HD_PLUGIN_LOADER_ERROR_KEYFILE,
                   "A keyfile required to load legacy plugins");
      return NULL;
    }

  hd_plugin_loader_legacy_guess_type (HD_PLUGIN_LOADER_LEGACY (loader));

  switch (priv->plugin_type)
    {
      case HD_PLUGIN_LOADER_LEGACY_TYPE_HOME:
          priv->symbol_mapping = (SymbolMapping *)home_symbol_mapping;
          priv->init_func = hd_plugin_loader_legacy_home_init;
          priv->library_key = HD_PLUGIN_LOADER_LEGACY_KEY_X_HOME_APPLET;
          priv->module_dir = HD_PLUGIN_LOADER_LEGACY_HOME_MODULE_PATH;
          break;
      
      case HD_PLUGIN_LOADER_LEGACY_TYPE_NAVIGATOR:
          priv->symbol_mapping = (SymbolMapping *)navigator_symbol_mapping;
          priv->init_func = hd_plugin_loader_legacy_navigator_init;
          priv->library_key = HD_PLUGIN_LOADER_LEGACY_KEY_X_TN;
          priv->module_dir = HD_PLUGIN_LOADER_LEGACY_NAVIGATOR_MODULE_PATH;
          break;
      default:
          g_set_error (error,
                       hd_plugin_loader_error_quark (),
                       HD_PLUGIN_LOADER_ERROR_UNKNOWN_TYPE,
                       "Could not determine the Hildon Desktop plugin type");
          return NULL;
    }

  /* Open the module */
  hd_plugin_loader_legacy_open_module (HD_PLUGIN_LOADER_LEGACY (loader),
                                       &local_error);
  if (local_error) goto error;

  /* Call the correct initialization plugin,
   * get the DesktomItem back */
  item = priv->init_func (HD_PLUGIN_LOADER_LEGACY (loader), &local_error);
  if (local_error) goto error;

  /* Now connect all the callbacks to the proper signals */
  hd_plugin_loader_legacy_connect_signals (HD_PLUGIN_LOADER_LEGACY (loader),
                                           item,
                                           &local_error);
  if (local_error) goto error;

  return item;

error:

  g_propagate_error (error, local_error);
  if (item)
    gtk_widget_destroy (item);
  if (priv->module)
    {
      g_module_close (priv->module);
      priv->module = NULL;
    }
  return NULL;
}

static void
hd_plugin_loader_legacy_open_module (HDPluginLoaderLegacy *loader,
                                     GError **error)
{
  HDPluginLoaderLegacyPrivate *priv;
  GKeyFile     *keyfile;
  GError       *keyfile_error = NULL;
  gchar        *module_file = NULL;
  gchar        *module_path = NULL;
  
  priv = HD_PLUGIN_LOADER_LEGACY_GET_PRIVATE (loader);

  keyfile = hd_plugin_loader_get_key_file (HD_PLUGIN_LOADER (loader));
  
  module_file = g_key_file_get_string (keyfile,
                                       HD_PLUGIN_LOADER_LEGACY_GROUP,
                                       priv->library_key,
                                       &keyfile_error);

  if (keyfile_error)
    {
      g_propagate_error (error, keyfile_error);
      return;
    }

  if (module_file && module_file[0] == G_DIR_SEPARATOR)
    module_path = module_file;
  else
    {
      module_path = g_build_path (G_DIR_SEPARATOR_S,
                                  priv->module_dir,
                                  module_file,
                                  NULL);

      g_free (module_file);
    }

  priv->module = g_module_open (module_path, 0);
  g_free (module_path);

  if (!priv->module)
    {
      g_set_error (error,
                   hd_plugin_loader_error_quark (),
                   HD_PLUGIN_LOADER_ERROR_OPEN,
                   g_module_error ());
      return;
    }

}

static void
hd_plugin_loader_legacy_connect_signals (HDPluginLoaderLegacy *loader,
                                         GtkWidget *item,
                                         GError **error)
{
  HDPluginLoaderLegacyPrivate      *priv;
  SymbolMapping *mapping;
  priv = HD_PLUGIN_LOADER_LEGACY_GET_PRIVATE (loader);

  g_return_if_fail (priv->symbol_mapping);
  g_return_if_fail (priv->module);
  mapping = priv->symbol_mapping;

  while (mapping->symbol_name)
    {
      gpointer symbol = NULL;

      if (!mapping->signal)
        {
          mapping ++;
          continue;
        }

      g_module_symbol (priv->module, mapping->symbol_name, &symbol);

      fprintf (stderr, "Found symbol %s, connecting to %s\n", mapping->symbol_name, mapping->signal );

      if (symbol)
        g_signal_connect_swapped (G_OBJECT (item),
                                  mapping->signal,
                                  symbol,
                                  priv->module_data);

      mapping ++;

    }

}

static void
hd_plugin_loader_legacy_guess_type (HDPluginLoaderLegacy *loader)
{
  HDPluginLoaderLegacyPrivate   *priv;
  GKeyFile                      *keyfile;
  gchar                         *value;
  g_return_if_fail (loader);
 
  priv = HD_PLUGIN_LOADER_LEGACY_GET_PRIVATE (loader);

  keyfile = hd_plugin_loader_get_key_file (HD_PLUGIN_LOADER (loader));

  /* Check for the Type = (which is only used for legacy Home applet */
  value = g_key_file_get_string (keyfile,
                                 HD_PLUGIN_LOADER_LEGACY_GROUP,
                                 HD_PLUGIN_LOADER_LEGACY_KEY_TYPE,
                                 NULL);

  if (value && strcmp (value, "HildonHomeApplet") == 0)
    {
      priv->plugin_type = HD_PLUGIN_LOADER_LEGACY_TYPE_HOME;
      g_free (value);
      return;
    }
      
  g_free (value);

  /* Check for the X-task-navigator-plugin = field */
  value = g_key_file_get_string (keyfile,
                                 HD_PLUGIN_LOADER_LEGACY_GROUP,
                                 HD_PLUGIN_LOADER_LEGACY_KEY_X_TN,
                                 NULL);

  if (value)
    {
      priv->plugin_type = HD_PLUGIN_LOADER_LEGACY_TYPE_NAVIGATOR;
      g_free (value);
      return;
    }

}

static GtkWidget *
hd_plugin_loader_legacy_home_init (HDPluginLoaderLegacy    *loader,
                                   GError                 **error)
{
  typedef void       *(*HomeInitializeFn)   (void *state_data, 
                                             int *state_size,
                                             GtkWidget **widget);

  HDPluginLoaderLegacyPrivate *priv;
  GtkWidget        *applet = NULL;
  GtkWidget        *module_widget = NULL;
  gpointer          state_data = NULL;
  gint              state_data_length = 0;
  SymbolMapping    *s;
  gpointer          symbol;

  priv = HD_PLUGIN_LOADER_LEGACY_GET_PRIVATE (loader);

  s = &priv->symbol_mapping[HD_PLUGIN_LOADER_LEGACY_HOME_SYMBOL_INITIALIZE];

  g_module_symbol (priv->module, s->symbol_name, &symbol);

  if (!symbol)
    {
      g_set_error (error,
                   hd_plugin_loader_error_quark (),
                   HD_PLUGIN_LOADER_ERROR_SYMBOL,
                   g_module_error ());
      return NULL;
    }

  priv->module_data = ((HomeInitializeFn)symbol) (&state_data,
                                                  &state_data_length,
                                                  &module_widget);

  if (!module_widget || !GTK_IS_WIDGET (module_widget))
    {
      g_set_error (error,
                   hd_plugin_loader_error_quark (),
                   HD_PLUGIN_LOADER_ERROR_INIT,
                   "Hildon Home Home Applet did not return a widget");
      return NULL;
    }

  applet = hildon_home_applet_new ();

  gtk_container_add (GTK_CONTAINER (applet), module_widget);

  return applet;
}

static GtkWidget *
hd_plugin_loader_legacy_navigator_init (HDPluginLoaderLegacy    *loader,
                                        GError                 **error)
{
  typedef void         *(*NavigatorCreateFn)           (void);
  typedef GtkWidget    *(*NavigatorGetButtonFn)        (gpointer data);
  typedef void         *(*NavigatorInitializeFn)       (gpointer data);

  HDPluginLoaderLegacyPrivate *priv;
  GtkWidget        *item = NULL;
  GtkWidget        *module_widget = NULL;
  SymbolMapping    *s;
  gpointer          symbol = NULL;

  priv = HD_PLUGIN_LOADER_LEGACY_GET_PRIVATE (loader);
  
  s = &priv->symbol_mapping[HD_PLUGIN_LOADER_LEGACY_NAVIGATOR_SYMBOL_CREATE];
  
  g_module_symbol (priv->module, s->symbol_name, &symbol);

  if (!symbol)
    {
      g_set_error (error,
                   hd_plugin_loader_error_quark (),
                   HD_PLUGIN_LOADER_ERROR_SYMBOL,
                   g_module_error ());
      return NULL;
    }
  
  priv->module_data = ((NavigatorCreateFn)symbol) ();

  s = &priv->symbol_mapping[HD_PLUGIN_LOADER_LEGACY_NAVIGATOR_SYMBOL_GET_BUTTON];
  
  symbol = NULL;
  g_module_symbol (priv->module, s->symbol_name, &symbol);

  if (!symbol)
    {
      g_set_error (error,
                   hd_plugin_loader_error_quark (),
                   HD_PLUGIN_LOADER_ERROR_SYMBOL,
                   g_module_error ());
      return NULL;
    }

  module_widget = ((NavigatorGetButtonFn)symbol) (priv->module_data);
  
  if (!module_widget || !GTK_IS_WIDGET (module_widget))
    {
      g_set_error (error,
                   hd_plugin_loader_error_quark (),
                   HD_PLUGIN_LOADER_ERROR_INIT,
                   "Hildon Navigator Item did not return a widget");
      return NULL;
    }

  item = g_object_new (TASKNAVIGATOR_TYPE_ITEM, NULL);

  gtk_container_add (GTK_CONTAINER (item), module_widget);
  
  s = &priv->symbol_mapping[HD_PLUGIN_LOADER_LEGACY_NAVIGATOR_SYMBOL_INITIALIZE_MENU];
  symbol = NULL;
  g_module_symbol (priv->module, s->symbol_name, &symbol);

  if (!symbol)
    {
      g_set_error (error,
                   hd_plugin_loader_error_quark (),
                   HD_PLUGIN_LOADER_ERROR_SYMBOL,
                   g_module_error ());
      return NULL;
    }

  ((NavigatorInitializeFn)symbol) (priv->module_data);

  return item;
}
