/*
 * This file is part of hildon-desktop
 *
 * Copyright (C) 2006 Nokia Corporation.
 *
 * Author:  Lucas Rocha <lucas.rocha@nokia.com>
 * Contact: Karoliina Salminen <karoliina.t.salminen@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <glib-object.h>

#include "hd-plugin-loader.h"
#include "hd-plugin-loader-legacy.h"
#include "hd-plugin-loader-factory.h"
#include "hd-config.h"

#define HD_PLUGIN_LOADER_FACTORY_GET_PRIVATE(object) \
        (G_TYPE_INSTANCE_GET_PRIVATE ((object), HD_TYPE_PLUGIN_LOADER_FACTORY, HDPluginLoaderFactoryPrivate))

G_DEFINE_TYPE (HDPluginLoaderFactory, hd_plugin_loader_factory, G_TYPE_OBJECT);

struct _HDPluginLoaderFactoryPrivate 
{
  GHashTable *registry;
};

static void
hd_plugin_loader_factory_init (HDPluginLoaderFactory *factory)
{
  factory->priv = HD_PLUGIN_LOADER_FACTORY_GET_PRIVATE (factory);

  factory->priv->registry = g_hash_table_new (g_str_hash, g_str_equal);
}

static gboolean
hd_plugin_loader_factory_free_loader (gchar *key, GObject *object)
{
  g_free (key);
  g_object_unref (object);

  return TRUE;
}

static void
hd_plugin_loader_factory_finalize (GObject *object)
{
  HDPluginLoaderFactoryPrivate *priv;

  g_return_if_fail (object != NULL);
  g_return_if_fail (HD_IS_PLUGIN_LOADER_FACTORY (object));

  priv = HD_PLUGIN_LOADER_FACTORY (object)->priv;
 
  if (priv->registry != NULL) 
  {
    g_hash_table_foreach_remove (priv->registry, 
                                 (GHRFunc) hd_plugin_loader_factory_free_loader, 
                                 NULL);

    g_hash_table_destroy (priv->registry);
  }

  G_OBJECT_CLASS (hd_plugin_loader_factory_parent_class)->finalize (object);
}

static void
hd_plugin_loader_factory_class_init (HDPluginLoaderFactoryClass *class)
{
  GObjectClass *g_object_class = (GObjectClass *) class;
  
  g_object_class->finalize = hd_plugin_loader_factory_finalize;
 
  g_type_class_add_private (g_object_class, sizeof (HDPluginLoaderFactoryPrivate));
}

GObject *
hd_plugin_loader_factory_new ()
{
  GObject *factory = g_object_new (HD_TYPE_PLUGIN_LOADER_FACTORY, NULL);

  return factory;
}

HDPluginLoader *
hd_plugin_loader_factory_create (HDPluginLoaderFactory *factory,
                                 const gchar           *plugin_path)
{
  HDPluginLoaderFactoryPrivate *priv;
  HDPluginLoader *loader = NULL;
  GKeyFile *keyfile;
  gchar *type;
  GError *error = NULL;

  g_return_val_if_fail (plugin_path != NULL, NULL);
  g_return_val_if_fail (factory != NULL, NULL);
  g_return_val_if_fail (HD_IS_PLUGIN_LOADER_FACTORY (factory), NULL);

  priv = factory->priv;

  keyfile = g_key_file_new ();

  g_key_file_load_from_file (keyfile,
                             plugin_path,
                             G_KEY_FILE_NONE,
                             &error);

  if (error)
  {
    g_warning ("Error loading plugin desktop file: %s", error->message);
    g_error_free (error);

    return NULL;
  }

  type = g_key_file_get_string (keyfile,
                                HD_PLUGIN_CONFIG_GROUP,
                                HD_PLUGIN_CONFIG_KEY_TYPE,
                                &error);

  if (error)
  {
    if (error->domain == G_KEY_FILE_ERROR &&
        error->code == G_KEY_FILE_ERROR_KEY_NOT_FOUND)
    {
      /* If no Type= assume old API */
      type = g_strdup (HD_PLUGIN_LOADER_OLD_API);
      g_error_free (error);
    }
    else
    {
      g_warning ("Error reading plugin desktop file: %s", error->message);
      g_error_free (error);
      return NULL;
    }
  }
  loader = (HDPluginLoader *) g_hash_table_lookup (priv->registry, type);

  if (!loader) 
  {
    if (!g_ascii_strcasecmp (type, HD_PLUGIN_LOADER_OLD_API) ||
        !g_ascii_strcasecmp (type, HD_PLUGIN_LOADER_OLD_API_HOME)) 
    {
      /* Create instance of Old API plugin loader and 
         add to registry for later use */
      loader = g_object_new (HD_TYPE_PLUGIN_LOADER_LEGACY, NULL);
      g_hash_table_insert (priv->registry, type, loader);
    }
    else
    {
      g_warning ("Unknown Plugin Loader type: %s", type);
    }
  }

  hd_plugin_loader_set_key_file (loader, keyfile);

  g_free (type);

  return loader;
}
