/*
 * This file is part of hildon-desktop
 *
 * Copyright (C) 2006 Nokia Corporation.
 *
 * Author:  Johan Bilien <johan.bilien@nokia.com>
 * Contact: Karoliina Salminen <karoliina.t.salminen@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include "hd-home-window.h"

#include <libosso.h>
#include <osso-helplib.h>

#include <libhildondesktop/hildon-home-area.h>
#include <libhildondesktop/hildon-home-titlebar.h>

#include <libhildondesktop/hildon-home-l10n.h>

G_DEFINE_TYPE (HDHomeWindow, hd_home_window, HILDON_TYPE_HOME_WINDOW);

typedef struct _HDHomeWindowPrivate
{
  osso_context_t    *osso_context;
} HDHomeWindowPrivate;

#define HD_HOME_WINDOW_GET_PRIVATE(obj) \
(G_TYPE_INSTANCE_GET_PRIVATE ((obj), HD_TYPE_HOME_WINDOW, HDHomeWindowPrivate))

/* Properties */
enum {
  PROP_OSSO_CONTEXT = 1
};

static void
hd_home_window_get_property (GObject    *gobject,
                             guint       prop_id,
                             GValue     *value,
                             GParamSpec *pspec);

static void
hd_home_window_set_property (GObject      *gobject,
                             guint         prop_id,
                             const GValue *value,
                             GParamSpec   *pspec);

static void
hd_home_window_set_osso_context (HDHomeWindow *window, osso_context_t *osso);


static void
hd_home_window_class_init (HDHomeWindowClass *window_class)
{
  GParamSpec               *pspec;
  GObjectClass             *object_class;

  object_class = G_OBJECT_CLASS (window_class);

  object_class->set_property = hd_home_window_set_property;
  object_class->get_property = hd_home_window_get_property;

  pspec = g_param_spec_pointer ("osso-context",
                                "Osso Context",
                                "Osso context to be used by the window",
                                (G_PARAM_READWRITE | G_PARAM_CONSTRUCT));

  g_object_class_install_property (object_class,
                                   PROP_OSSO_CONTEXT,
                                   pspec);

  g_type_class_add_private (window_class, sizeof (HDHomeWindowPrivate));

}

static void
hd_home_window_init (HDHomeWindow *window)
{
}

static void
hd_home_window_set_property (GObject      *gobject,
                             guint         prop_id,
                             const GValue *value,
                             GParamSpec   *pspec)
{
  HDHomeWindow *window = HD_HOME_WINDOW (gobject);

  switch (prop_id)
    {
      case PROP_OSSO_CONTEXT:
          hd_home_window_set_osso_context (window,
                                           (osso_context_t *)
                                           g_value_get_pointer (value));

          break;
      default:
          G_OBJECT_WARN_INVALID_PROPERTY_ID (gobject, prop_id, pspec);
          break;
    }
}

static void
hd_home_window_get_property (GObject    *gobject,
                             guint       prop_id,
                             GValue     *value,
                             GParamSpec *pspec)
{
  HDHomeWindowPrivate *priv = HD_HOME_WINDOW_GET_PRIVATE (gobject);

  switch (prop_id)
    {
      case PROP_OSSO_CONTEXT:
          g_value_set_pointer (value, priv->osso_context);
          break;
      default:
          G_OBJECT_WARN_INVALID_PROPERTY_ID (gobject, prop_id, pspec);
          break;
    }
}

static void
hd_home_window_applet_activate (HDHomeWindow *window,
                                const gchar *applet_path)
{
  HDHomeWindowPrivate  *priv ;
  osso_return_t         res;

  priv = HD_HOME_WINDOW_GET_PRIVATE (window);

  if (!priv->osso_context)
    return;

  res = osso_cp_plugin_execute (priv->osso_context,
                                applet_path,
                                NULL,
                                TRUE);
  switch (res)
    {
      case OSSO_OK:
          break;
      case OSSO_ERROR:
          g_warning ("OSSO_ERROR (No plugin found)\n");
          break;
      case OSSO_RPC_ERROR:
          g_warning ("OSSO_RPC_ERROR (RPC failed for plugin)\n");
          break;
      case OSSO_INVALID:
          g_warning ("OSSO_INVALID (Invalid argument)\n");
          break;
      default:
          g_warning ("Unknown error!\n");
          break;
    }

}

static void
hd_home_window_help_activate (HDHomeWindow *window)
{
  HildonDesktopWindow  *dwindow = HILDON_DESKTOP_WINDOW (window);
  HDHomeWindowPrivate  *priv;
  HildonHomeArea       *area;
  osso_return_t         res;

  g_return_if_fail (HILDON_IS_HOME_AREA (dwindow->container));
  area = HILDON_HOME_AREA (dwindow->container);

  priv = HD_HOME_WINDOW_GET_PRIVATE (window);

  if (!priv->osso_context)
    return;

  if (hildon_home_area_get_layout_mode (area))
    res = ossohelp_show (priv->osso_context,
                         HH_HELP_LAYOUT_MODE,
                         OSSO_HELP_SHOW_DIALOG);
  else
    res = ossohelp_show (priv->osso_context,
                         HH_HELP,
                         0);

  switch (res)
    {
    case OSSO_OK:
      break;
    case OSSO_ERROR:
      g_warning ("OSSO_ERROR (No help for such topic ID)\n");
      break;
    case OSSO_RPC_ERROR:
      g_warning ("OSSO_RPC_ERROR (Unable to contact HelpApp or Browser)\n");
      break;
    case OSSO_INVALID:
      g_warning ("OSSO_INVALID (Parameter not formally correct)\n");
      break;
    default:
      g_warning ("Unknown error!\n");
      break;
    }
}

static void
hd_home_window_set_osso_context (HDHomeWindow *window,
                                 osso_context_t *osso_context)
{
  HDHomeWindowPrivate *priv;

  g_return_if_fail (HD_IS_HOME_WINDOW (window));

  priv = HD_HOME_WINDOW_GET_PRIVATE (window);

  if (priv->osso_context != osso_context)
    {
      GtkWidget *titlebar;

      priv->osso_context = osso_context;
      g_object_notify (G_OBJECT (window), "osso-context");

      titlebar = hildon_home_window_get_titlebar (HILDON_HOME_WINDOW (window));

      if (HILDON_IS_HOME_TITLEBAR (titlebar))
        {
          g_signal_connect_swapped (titlebar, "help-activate",
                                    G_CALLBACK (hd_home_window_help_activate),
                                    window);
          
          g_signal_connect_swapped (titlebar, "applet-activate",
                                    G_CALLBACK (hd_home_window_applet_activate),
                                    window);
        }
    }
}


