/*
 * This file is part of maemo-af-desktop
 *
 * Copyright (C) 2006 Nokia Corporation.
 *
 * Author:  Moises Martinez <moises.martinez@nokia.com>
 * Contact: Karoliina Salminen <karoliina.t.salminen@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include "desktop-panel.h"
#include "hildon-desktop-item.h"

#define DESKTOP_PANEL_GET_PRIVATE(object) \
        (G_TYPE_INSTANCE_GET_PRIVATE ((object), DESKTOP_TYPE_PANEL, DesktopPanel))

G_DEFINE_TYPE (DesktopPanel, desktop_panel, GTK_TYPE_BOX);

enum
{
  PROP_0,
  PROP_ORI,
  PROP_ITEM_WIDTH,
  PROP_ITEM_HEIGHT
};

enum
{
  SIGNAL_FLIP,
  N_SIGNALS
};

static gint signals[N_SIGNALS];

static void desktop_panel_class_init         (DesktopPanelClass *panel_class);
static void desktop_panel_init               (DesktopPanel *panel);

static void desktop_panel_get_property       (GObject *object, 
					      guint prop_id, 
					      GValue *value, 
					      GParamSpec *pspec);

static void desktop_panel_set_property       (GObject *object, 
					      guint prop_id, 
					      const GValue *value, 
					      GParamSpec *pspec);

static void desktop_panel_size_request       (GtkWidget *widget, 
					      GtkRequisition *requisition);
static void desktop_panel_size_allocate      (GtkWidget *widget, 
					      GtkAllocation *allocation);
static void desktop_panel_vbox_size_request  (GtkWidget *widget, 
				              GtkRequisition *requisition);
static void desktop_panel_hbox_size_request  (GtkWidget *widget, 
					      GtkRequisition *requisition);
static void desktop_panel_vbox_size_allocate (GtkWidget *widget, 
					      GtkAllocation *allocation);
static void desktop_panel_hbox_size_allocate (GtkWidget *widget, 
					      GtkAllocation *allocation);

static void desktop_panel_calc_positions     (DesktopPanel *panel, 
					      HildonDesktopItem *item);

static void desktop_panel_cadd               (GtkContainer *container,
					      GtkWidget *widget);

static void 
desktop_panel_class_init (DesktopPanelClass *panel_class)
{
  GObjectClass      *object_class    = G_OBJECT_CLASS      (panel_class);
  GtkWidgetClass    *widget_class    = GTK_WIDGET_CLASS    (panel_class);
  GtkContainerClass *container_class = GTK_CONTAINER_CLASS (panel_class); 

  widget_class->size_request   = desktop_panel_size_request;
  widget_class->size_allocate  = desktop_panel_size_allocate;

  panel_class->get_orientation = desktop_panel_get_orientation;
  panel_class->set_orientation = desktop_panel_set_orientation;
  panel_class->flip_panel      = desktop_panel_flip_panel;  
  panel_class->add_button      = desktop_panel_add_button;
  panel_class->panel_flipped   = NULL;

  container_class->add = desktop_panel_cadd;

  object_class->get_property = desktop_panel_get_property;
  object_class->set_property = desktop_panel_set_property;

  signals[SIGNAL_FLIP] =
        g_signal_new("desktop_panel_flip",
                     G_OBJECT_CLASS_TYPE(object_class),
                     G_SIGNAL_RUN_FIRST,
		     G_STRUCT_OFFSET (DesktopPanelClass,panel_flipped),
                     NULL, NULL,
                     g_cclosure_marshal_VOID__VOID,
                     G_TYPE_NONE, 0);

  g_object_class_install_property (object_class,
                                   PROP_ORI,
                                   g_param_spec_int ("orientation",
                                                     "orientation",
                                                     "orientation",
                                                     GTK_ORIENTATION_HORIZONTAL,
                                                     GTK_ORIENTATION_VERTICAL,
                                                     GTK_ORIENTATION_HORIZONTAL,
                                                     G_PARAM_CONSTRUCT | G_PARAM_READWRITE));

  g_object_class_install_property (object_class,
                                   PROP_ITEM_WIDTH,
                                   g_param_spec_int ("item_width",
                                                     "itwidth",
                                                     "item's width in the container",
                                                     0,
                                                     G_MAXINT,
                                                     0,
                                                     G_PARAM_CONSTRUCT | G_PARAM_READWRITE));

  g_object_class_install_property (object_class,
                                   PROP_ITEM_HEIGHT,
                                   g_param_spec_int ("item_height",
                                                     "itheigth",
                                                     "item's width in the container",
                                                     0,
                                                     G_MAXINT,
                                                     0,
                                                     G_PARAM_CONSTRUCT | G_PARAM_READWRITE));
}

static void 
desktop_panel_init (DesktopPanel *panel)
{
  panel->orient = GTK_ORIENTATION_HORIZONTAL;
  panel->item_width = panel->item_height = 0;
}

static void 
desktop_panel_cadd (GtkContainer *container, GtkWidget *widget)
{
  g_return_if_fail (DESKTOP_IS_PANEL (container));

  desktop_panel_add_button (DESKTOP_PANEL (container),widget);
}

static void 
desktop_panel_get_property (GObject *object, 
			   guint prop_id, 
			   GValue *value, 
			   GParamSpec *pspec)
{
  DesktopPanel *panel;

  g_assert (object && DESKTOP_IS_PANEL (object));

  panel = DESKTOP_PANEL (object);

  switch (prop_id)
  {
    case PROP_ORI:
      g_value_set_int (value, panel->orient);
      break;
    case PROP_ITEM_WIDTH:
      g_value_set_int (value, panel->item_width);
      break;
    case PROP_ITEM_HEIGHT:
      g_value_set_int (value, panel->item_height);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;						
  }
}

static void 
desktop_panel_set_property (GObject *object,
			   guint prop_id, 
			   const GValue *value,
			   GParamSpec *pspec)
{
  DesktopPanel *panel;

  g_assert (object && DESKTOP_IS_PANEL (object));

  panel = DESKTOP_PANEL (object);

  switch (prop_id)
  {
    case PROP_ORI:	   
      panel->orient = g_value_get_int (value);
      break;
    case PROP_ITEM_WIDTH:
      panel->item_width = g_value_get_int (value);
      break;
    case PROP_ITEM_HEIGHT:
      panel->item_height = g_value_get_int (value);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static void 
desktop_panel_size_request (GtkWidget *widget, GtkRequisition *requisition)
{
  DesktopPanel *panel = DESKTOP_PANEL (widget);

  if (panel->orient & GTK_ORIENTATION_VERTICAL)
    desktop_panel_vbox_size_request (widget,requisition);
  else
    desktop_panel_hbox_size_request (widget,requisition);
    
}

static void 
desktop_panel_size_allocate (GtkWidget *widget, GtkAllocation *allocation)
{
  DesktopPanel *panel = DESKTOP_PANEL (widget);

  if (panel->orient & GTK_ORIENTATION_VERTICAL)
    desktop_panel_vbox_size_allocate (widget,allocation);
  else
    desktop_panel_hbox_size_allocate (widget,allocation);
}

/* The next 4 functions are from gtkvbox.c and gtkhbox.c
 * GTK - The GIMP Toolkit
 * Copyright (C) 1995-1997 Peter Mattis, Spencer Kimball and Josh MacDonald
 *
 */

static void
desktop_panel_vbox_size_request (GtkWidget      *widget,
                                 GtkRequisition *requisition)
{
  GtkBox *box;
  GtkBoxChild *child;
  GtkRequisition child_requisition;
  GList *children;
  gint nvis_children;
  gint height;

  box = GTK_BOX (widget);
  requisition->width = 0;
  requisition->height = 0;
  nvis_children = 0;

  children = box->children;
  while (children)
  {
    child = children->data;
    children = children->next;

    if (GTK_WIDGET_VISIBLE (child->widget))
    {
      gtk_widget_size_request (child->widget, &child_requisition);

      if (box->homogeneous)
      {
        height = child_requisition.height + child->padding * 2;
        requisition->height = MAX (requisition->height, height);
      }
      else
      {
        requisition->height += child_requisition.height + child->padding * 2;
      }

      requisition->width = MAX (requisition->width, child_requisition.width);

      nvis_children += 1;
    }
  }

  if (nvis_children > 0)
  {
    if (box->homogeneous)
      requisition->height *= nvis_children;
    requisition->height += (nvis_children - 1) * box->spacing;
  }

  requisition->width += GTK_CONTAINER (box)->border_width * 2;
  requisition->height += GTK_CONTAINER (box)->border_width * 2;
}

static void
desktop_panel_vbox_size_allocate (GtkWidget     *widget,
                             GtkAllocation *allocation)
{
  GtkBox *box;
  GtkBoxChild *child;
  GList *children;
  GtkAllocation child_allocation;
  gint nvis_children;
  gint nexpand_children;
  gint child_height;
  gint height;
  gint extra;
  gint y;

  box = GTK_BOX (widget);
  widget->allocation = *allocation;

  nvis_children = 0;
  nexpand_children = 0;
  children = box->children;

  while (children)
  {
    child = children->data;
    children = children->next;

    if (GTK_WIDGET_VISIBLE (child->widget))
    {
      nvis_children += 1;
      if (child->expand)
        nexpand_children += 1;
    }
  }

  if (nvis_children > 0)
  {
    if (box->homogeneous)
    {
      height = (allocation->height -
               GTK_CONTAINER (box)->border_width * 2 -
               (nvis_children - 1) * box->spacing);
      extra = height / nvis_children;
    }
    else if (nexpand_children > 0)
    {
      height = (gint) allocation->height - (gint) widget->requisition.height;
      extra = height / nexpand_children;
    }
    else
    {
      height = 0;
      extra = 0;
    }

    y = allocation->y + GTK_CONTAINER (box)->border_width;
    child_allocation.x = allocation->x + GTK_CONTAINER (box)->border_width;
    child_allocation.width = MAX (1, (gint) allocation->width - (gint) GTK_CONTAINER (box)->border_width * 2);

    children = box->children;
    while (children)
    {
      child = children->data;
      children = children->next;

      if ((child->pack == GTK_PACK_START) && GTK_WIDGET_VISIBLE (child->widget))
      {
        if (box->homogeneous)
        {
          if (nvis_children == 1)
            child_height = height;
          else
            child_height = extra;

          nvis_children -= 1;
          height -= extra;
        }
        else
        {
          GtkRequisition child_requisition;

          gtk_widget_get_child_requisition (child->widget, &child_requisition);
          child_height = child_requisition.height + child->padding * 2;

          if (child->expand)
          {
            if (nexpand_children == 1)
              child_height += height;
            else
              child_height += extra;

            nexpand_children -= 1;
            height -= extra;
          }
        }

        if (child->fill)
        {
          child_allocation.height = MAX (1, child_height - (gint)child->padding * 2);
          child_allocation.y = y + child->padding;
        }
        else
        {
          GtkRequisition child_requisition;

          gtk_widget_get_child_requisition (child->widget, &child_requisition);
          child_allocation.height = child_requisition.height;
          child_allocation.y = y + (child_height - child_allocation.height) / 2;
        }

        gtk_widget_size_allocate (child->widget, &child_allocation);

        y += child_height + box->spacing;
      }
    }

    y = allocation->y + allocation->height - GTK_CONTAINER (box)->border_width;

    children = box->children;
    while (children)
    {
      child = children->data;
      children = children->next;

      if ((child->pack == GTK_PACK_END) && GTK_WIDGET_VISIBLE (child->widget))
      {
        GtkRequisition child_requisition;
        gtk_widget_get_child_requisition (child->widget, &child_requisition);

        if (box->homogeneous)
        {
          if (nvis_children == 1)
            child_height = height;
          else
            child_height = extra;

          nvis_children -= 1;
          height -= extra;
        }
        else
        {
           child_height = child_requisition.height + child->padding * 2;

           if (child->expand)
           {
             if (nexpand_children == 1)
               child_height += height;
             else
               child_height += extra;

             nexpand_children -= 1;
             height -= extra;
           }
         }

         if (child->fill)
         {
           child_allocation.height = MAX (1, child_height - (gint)child->padding * 2);
           child_allocation.y = y + child->padding - child_height;
         }
         else
         {
           child_allocation.height = child_requisition.height;
           child_allocation.y = y + (child_height - child_allocation.height) / 2 - child_height;
         }

         gtk_widget_size_allocate (child->widget, &child_allocation);

         y -= (child_height + box->spacing);
       }
    }
  }
}

static void
desktop_panel_hbox_size_request (GtkWidget      *widget,
		       		 GtkRequisition *requisition)
{
  GtkBox *box;
  GtkBoxChild *child;
  GList *children;
  gint nvis_children;
  gint width;

  box = GTK_BOX (widget);
  requisition->width = 0;
  requisition->height = 0;
  nvis_children = 0;

  children = box->children;
  while (children)
  {
    child = children->data;
    children = children->next;

    if (GTK_WIDGET_VISIBLE (child->widget))
    {
      GtkRequisition child_requisition;

      gtk_widget_size_request (child->widget, &child_requisition);

      if (box->homogeneous)
      {
	width = child_requisition.width + child->padding * 2;
	requisition->width = MAX (requisition->width, width);
      }
      else
      {
	requisition->width += child_requisition.width + child->padding * 2;
      }

      requisition->height = MAX (requisition->height, child_requisition.height);

      nvis_children += 1;
    }
  }

  if (nvis_children > 0)
  {
    if (box->homogeneous)
      requisition->width *= nvis_children;
    requisition->width += (nvis_children - 1) * box->spacing;
  }

  requisition->width += GTK_CONTAINER (box)->border_width * 2;
  requisition->height += GTK_CONTAINER (box)->border_width * 2;
}

static void
desktop_panel_hbox_size_allocate (GtkWidget     *widget,
			     GtkAllocation *allocation)
{
  GtkBox *box;
  GtkBoxChild *child;
  GList *children;
  GtkAllocation child_allocation;
  gint nvis_children;
  gint nexpand_children;
  gint child_width;
  gint width;
  gint extra;
  gint x;
  GtkTextDirection direction;

  box = GTK_BOX (widget);
  widget->allocation = *allocation;

  direction = gtk_widget_get_direction (widget);
  
  nvis_children = 0;
  nexpand_children = 0;
  children = box->children;

  while (children)
  {
    child = children->data;
    children = children->next;

    if (GTK_WIDGET_VISIBLE (child->widget))
    {
      nvis_children += 1;
      if (child->expand)
	nexpand_children += 1;
    }
  }

  if (nvis_children > 0)
  {
    if (box->homogeneous)
    {
      width = (allocation->width -
	      GTK_CONTAINER (box)->border_width * 2 -
	      (nvis_children - 1) * box->spacing);
      extra = width / nvis_children;
    }
    else if (nexpand_children > 0)
    {
      width = (gint) allocation->width - (gint) widget->requisition.width;
      extra = width / nexpand_children;
    }
    else
    {
      width = 0;
      extra = 0;
    }

    x = allocation->x + GTK_CONTAINER (box)->border_width;
    child_allocation.y = allocation->y + GTK_CONTAINER (box)->border_width;
    child_allocation.height = MAX (1, (gint) allocation->height - (gint) GTK_CONTAINER (box)->border_width * 2);

    children = box->children;
    while (children)
    {
      child = children->data;
      children = children->next;

      if ((child->pack == GTK_PACK_START) && GTK_WIDGET_VISIBLE (child->widget))
      {
	if (box->homogeneous)
	{
	  if (nvis_children == 1)
	    child_width = width;
	  else
	    child_width = extra;

	  nvis_children -= 1;
	  width -= extra;
	}
	else
	{
	  GtkRequisition child_requisition;

	  gtk_widget_get_child_requisition (child->widget, &child_requisition);

	  child_width = child_requisition.width + child->padding * 2;

	  if (child->expand)
	  {
	    if (nexpand_children == 1)
	      child_width += width;
	    else
	      child_width += extra;

	    nexpand_children -= 1;
	    width -= extra;
          }
	}

	if (child->fill)
	{
	  child_allocation.width = MAX (1, (gint) child_width - (gint) child->padding * 2);
	  child_allocation.x = x + child->padding;
	}
	else
	{
	  GtkRequisition child_requisition;

	  gtk_widget_get_child_requisition (child->widget, &child_requisition);
	  child_allocation.width = child_requisition.width;
	  child_allocation.x = x + (child_width - child_allocation.width) / 2;
	}

	if (direction == GTK_TEXT_DIR_RTL)
	  child_allocation.x = 
           allocation->x + allocation->width - (child_allocation.x - allocation->x) - child_allocation.width;

	  gtk_widget_size_allocate (child->widget, &child_allocation);

	  x += child_width + box->spacing;
      }
    }

    x = allocation->x + allocation->width - GTK_CONTAINER (box)->border_width;

    children = box->children;
    while (children)
    {
      child = children->data;
      children = children->next;

      if ((child->pack == GTK_PACK_END) && GTK_WIDGET_VISIBLE (child->widget))
      {
	GtkRequisition child_requisition;
	gtk_widget_get_child_requisition (child->widget, &child_requisition);

        if (box->homogeneous)
        {
          if (nvis_children == 1)
            child_width = width;
          else
            child_width = extra;

          nvis_children -= 1;
          width -= extra;
        }
        else
        {
	  child_width = child_requisition.width + child->padding * 2;

          if (child->expand)
          {
            if (nexpand_children == 1)
              child_width += width;
            else
              child_width += extra;

            nexpand_children -= 1;
             width -= extra;
           }
         }

         if (child->fill)
         {
           child_allocation.width = MAX (1, (gint)child_width - (gint)child->padding * 2);
           child_allocation.x = x + child->padding - child_width;
         }
         else
         {
	   child_allocation.width = child_requisition.width;
           child_allocation.x = x + (child_width - child_allocation.width) / 2 - child_width;
         }

	 if (direction == GTK_TEXT_DIR_RTL)
	   child_allocation.x = 
             allocation->x + allocation->width - (child_allocation.x - allocation->x) - child_allocation.width;

           gtk_widget_size_allocate (child->widget, &child_allocation);

           x -= (child_width + box->spacing);
       }
    }
  }
}

static void 
desktop_panel_calc_positions (DesktopPanel *panel, HildonDesktopItem *item)
{
  /* FIXME: Please, implement me smoothly and very optimized */
	
  /* FIXME: This is not that smooth implementation, this only add the item*/ 	
  gtk_box_pack_start (GTK_BOX (panel), GTK_WIDGET (item),FALSE,FALSE,0);
  gtk_widget_show    (GTK_WIDGET (item)); 
}

/* Public declarations */

void 
desktop_panel_add_button (DesktopPanel *panel, GtkWidget *widget)
{
  GtkWidget *panel_widget;
  GtkRequisition requisition;

  g_return_if_fail (panel &&
                    widget && 
		    DESKTOP_IS_PANEL (panel) &&
		    GTK_IS_WIDGET (panel));		

  panel_widget = GTK_WIDGET (panel);

  if (panel->item_width == 0 || panel->item_height == 0) 
  {
  
    gtk_widget_size_request (widget,&requisition);

    if (panel->orient & GTK_ORIENTATION_HORIZONTAL)
      requisition.height = panel_widget->allocation.height;
    else
      requisition.width  = panel_widget->allocation.width;

    gtk_widget_set_size_request (widget,
			         requisition.width,
			         requisition.height);
  }
  else
  { 
    gtk_widget_set_size_request (widget,
			         panel->item_width,
			         panel->item_height);
  }

  if (HILDON_DESKTOP_IS_ITEM (widget))
    desktop_panel_calc_positions (panel,HILDON_DESKTOP_ITEM (widget));/*FIXME: Do this! */
  else
  {
    gtk_box_pack_start (GTK_BOX (panel), widget, FALSE, FALSE, 0);
    /*gtk_widget_show    (widget);*/
  }
}

void 
desktop_panel_set_orientation (DesktopPanel *panel, GtkOrientation orientation)
{
  g_return_if_fail (panel && DESKTOP_IS_PANEL (panel));
#ifndef PERFORMANCE
  g_object_set (G_OBJECT (panel),"orientation",orientation,NULL);
#else
  panel->orient = orientation;
#endif
}

GtkOrientation 
desktop_panel_get_orientation (DesktopPanel *panel)
{
  g_assert (panel && DESKTOP_IS_PANEL (panel));
#ifndef PERFORMANCE
  GtkOrientation orientation;
  
  g_object_get (G_OBJECT (panel), "orientation", &orientation,NULL);

  return orientation;
#else
 return panel->orient;
#endif
}

void 
desktop_panel_flip_panel (DesktopPanel *panel)
{
  g_return_if_fail (panel && DESKTOP_IS_PANEL (panel));
#ifndef PERFORMANCE
  GtkOrientation orientation;

  g_object_get (G_OBJECT (panel), "orientation", &orientation,NULL);
  g_object_set (G_OBJECT (panel), "orientation", !orientation,NULL);
#else
  panel->orient = !priv->orient;
#endif
}
