// -*- mode: cpp; mode: fold -*-
// Description								/*{{{*/
// $Id: acquire-item.h,v 1.26.2.3 2004/01/02 18:51:00 mdz Exp $
/* ######################################################################

   Acquire Item - Item to acquire

   When an item is instantiated it will add it self to the local list in
   the Owner Acquire class. Derived classes will then call QueueURI to 
   register all the URI's they wish to fetch at the initial moment.   
   
   Two item classes are provided to provide functionality for downloading
   of Index files and downloading of Packages.
   
   A Archive class is provided for downloading .deb files. It does Md5
   checking and source location as well as a retry algorithm.
   
   ##################################################################### */
									/*}}}*/
#ifndef PKGLIB_ACQUIRE_ITEM_H
#define PKGLIB_ACQUIRE_ITEM_H

#include <apt-pkg/acquire.h>
#include <apt-pkg/indexfile.h>
#include <apt-pkg/vendor.h>
#include <apt-pkg/sourcelist.h>
#include <apt-pkg/pkgrecords.h>
#include <apt-pkg/indexrecords.h>

#include <list>

#ifdef __GNUG__
#pragma interface "apt-pkg/acquire-item.h"
#endif 

// Item to acquire
class pkgAcquire::Item
{  
   protected:
   
   // Some private helper methods for registering URIs
   pkgAcquire *Owner;
   inline void QueueURI(ItemDesc &Item)
                 {Owner->Enqueue(Item);};
   inline void Dequeue() {Owner->Dequeue(this);};
   
   // Safe rename function with timestamp preservation
   void Rename(string From,string To);
   
   public:

   // State of the item
   enum {StatIdle, StatFetching, StatDone, StatError, StatAuthError} Status;
   string ErrorText;
   unsigned long FileSize;
   unsigned long PartialSize;   
   const char *Mode;
   unsigned long ID;
   bool Complete;
   bool Local;

   // Number of queues we are inserted into
   unsigned int QueueCounter;
   
   // File to write the fetch into
   string DestFile;

   // Action members invoked by the worker
   virtual void Failed(string Message,pkgAcquire::MethodConfig *Cnf);
   virtual void Done(string Message,unsigned long Size,string Md5Hash,
		     pkgAcquire::MethodConfig *Cnf);
   virtual void Start(string Message,unsigned long Size);
   virtual string Custom600Headers() {return string();};
   virtual string DescURI() = 0;
   virtual string ShortDesc() {return DescURI();}
   virtual void Finished() {};
   
   // Inquire functions
   virtual string MD5Sum() {return string();};
   pkgAcquire *GetOwner() {return Owner;};
   virtual bool IsTrusted() {return false;};
   virtual bool IsMyFile(string file)
     { return file == flNotDir (DestFile); }

   Item(pkgAcquire *Owner);
   virtual ~Item();
};

// item for index diffs

struct DiffInfo {
   string file;
   string sha1;
   unsigned long size;
};

class pkgAcqDiffIndex : public pkgAcquire::Item
{
 protected:
   bool Debug;
   pkgAcquire::ItemDesc Desc;
   string RealURI;
   string ExpectedMD5;
   string CurrentPackagesFile;
   string Description;

 public:
   // Specialized action members
   virtual void Failed(string Message,pkgAcquire::MethodConfig *Cnf);
   virtual void Done(string Message,unsigned long Size,string Md5Hash,
		     pkgAcquire::MethodConfig *Cnf);
   virtual string DescURI() {return RealURI + "Index";};
   virtual string Custom600Headers();

   // helpers
   bool ParseDiffIndex(string IndexDiffFile);
   
   pkgAcqDiffIndex(pkgAcquire *Owner,string URI,string URIDesc,
		   string ShortDesct, string ExpectedMD5);
};

class pkgAcqIndexDiffs : public pkgAcquire::Item
{
   protected:
   bool Debug;
   pkgAcquire::ItemDesc Desc;
   string RealURI;
   string ExpectedMD5;

   // this is the SHA-1 sum we expect after the patching
   string Description;
   vector<DiffInfo> available_patches;
   enum {StateFetchIndex,StateFetchDiff,StateUnzipDiff,StateApplyDiff} State;

   public:
   
   // Specialized action members
   virtual void Failed(string Message,pkgAcquire::MethodConfig *Cnf);
   virtual void Done(string Message,unsigned long Size,string Md5Hash,
		     pkgAcquire::MethodConfig *Cnf);
   virtual string DescURI() {return RealURI + "Index";};

   // various helpers
   bool QueueNextDiff();
   bool ApplyDiff(string PatchFile);
   void Finish(bool allDone=false);

   pkgAcqIndexDiffs(pkgAcquire *Owner,string URI,string URIDesc,
		    string ShortDesct, string ExpectedMD5,
		    vector<DiffInfo> diffs=vector<DiffInfo>());
};

// Item class for index files
class pkgAcqIndex : public pkgAcquire::Item
{
   protected:
   
   bool Decompression;
   bool Erase;
   pkgAcquire::ItemDesc Desc;
   string RealURI;
   string ExpectedMD5;
   string CompressionExtension;

   public:
   
   // Specialized action members
   virtual void Failed(string Message,pkgAcquire::MethodConfig *Cnf);
   virtual void Done(string Message,unsigned long Size,string Md5Hash,
		     pkgAcquire::MethodConfig *Cnf);
   virtual string Custom600Headers();
   virtual string DescURI() {return RealURI + CompressionExtension;};

   pkgAcqIndex(pkgAcquire *Owner,string URI,string URIDesc,
	       string ShortDesct, string ExpectedMD5, string compressExt="");
};

struct IndexTarget
{
   string URI;
   string Description;
   string ShortDesc;
   string MetaKey;
};

// Item class for index signatures
class pkgAcqMetaSig : public pkgAcquire::Item
{
   protected:
   
   pkgAcquire::ItemDesc Desc;
   string RealURI,MetaIndexURI,MetaIndexURIDesc,MetaIndexShortDesc;
   indexRecords* MetaIndexParser;
   const vector<struct IndexTarget*>* IndexTargets;

   public:
   
   // Specialized action members
   virtual void Failed(string Message,pkgAcquire::MethodConfig *Cnf);
   virtual void Done(string Message,unsigned long Size,string Md5Hash,
		     pkgAcquire::MethodConfig *Cnf);
   virtual string Custom600Headers();
   virtual string DescURI() {return RealURI; };

   pkgAcqMetaSig(pkgAcquire *Owner,string URI,string URIDesc, string ShortDesc,
		 string MetaIndexURI, string MetaIndexURIDesc, string MetaIndexShortDesc,
		 const vector<struct IndexTarget*>* IndexTargets,
		 indexRecords* MetaIndexParser);
};

// Item class for index signatures
class pkgAcqMetaIndex : public pkgAcquire::Item
{
   protected:
   
   pkgAcquire::ItemDesc Desc;
   string RealURI; // FIXME: is this redundant w/ Desc.URI?
   string SigFile;
   const vector<struct IndexTarget*>* IndexTargets;
   indexRecords* MetaIndexParser;
   bool AuthPass;
   // required to deal gracefully with problems caused by incorrect ims hits
   bool IMSHit; 

   bool VerifyVendor(string Message);
   void RetrievalDone(string Message);
   void AuthDone(string Message);
   void QueueIndexes(bool verify);
   
   public:
   
   // Specialized action members
   virtual void Failed(string Message,pkgAcquire::MethodConfig *Cnf);
   virtual void Done(string Message,unsigned long Size,string Md5Hash,
		     pkgAcquire::MethodConfig *Cnf);
   virtual string Custom600Headers();
   virtual string DescURI() {return RealURI; };
   virtual bool IsMyFile (string file);

   pkgAcqMetaIndex(pkgAcquire *Owner,
		   string URI,string URIDesc, string ShortDesc,
		   string SigFile,
		   const vector<struct IndexTarget*>* IndexTargets,
		   indexRecords* MetaIndexParser);
};

// Item class for archive files
class pkgAcqArchive : public pkgAcquire::Item
{
   protected:
   
   // State information for the retry mechanism
   pkgCache::VerIterator Version;
   pkgAcquire::ItemDesc Desc;
   pkgSourceList *Sources;
   pkgRecords *Recs;
   string MD5;
   string &StoreFilename;
   std::list<pkgCache::VerFileIterator> VerFileCandidates;
   std::list<pkgCache::VerFileIterator>::const_iterator CurVerFile;
   unsigned int Retries;
   int TrustLevel; 

   // Queue the next available file for download.
   bool QueueNext();
   
   public:
   
   // Specialized action members
   virtual void Failed(string Message,pkgAcquire::MethodConfig *Cnf);
   virtual void Done(string Message,unsigned long Size,string Md5Hash,
		     pkgAcquire::MethodConfig *Cnf);
   virtual string MD5Sum() {return MD5;};
   virtual string DescURI() {return Desc.URI;};
   virtual string ShortDesc() {return Desc.ShortDesc;};
   virtual void Finished();
   virtual bool IsTrusted();
   
   pkgAcqArchive(pkgAcquire *Owner,pkgSourceList *Sources,
		 pkgRecords *Recs,pkgCache::VerIterator const &Version,
		 string &StoreFilename);
};

// Fetch a generic file to the current directory
class pkgAcqFile : public pkgAcquire::Item
{
   pkgAcquire::ItemDesc Desc;
   string Md5Hash;
   unsigned int Retries;
   
   public:
   
   // Specialized action members
   virtual void Failed(string Message,pkgAcquire::MethodConfig *Cnf);
   virtual void Done(string Message,unsigned long Size,string Md5Hash,
		     pkgAcquire::MethodConfig *Cnf);
   virtual string MD5Sum() {return Md5Hash;};
   virtual string DescURI() {return Desc.URI;};

   // If DestFilename is empty, download to DestDir/<basename> if
   // DestDir is non-empty, $CWD/<basename> otherwise.  If
   // DestFilename is NOT empty, DestDir is ignored and DestFilename
   // is the absolute name to which the file should be downloaded.
   pkgAcqFile(pkgAcquire *Owner, string URI, string MD5, unsigned long Size,
	      string Desc, string ShortDesc,
	      const string &DestDir="", const string &DestFilename="");
};

/* For influencing the IsTrusted decision when acquiring a new version
   of a package and for influencing which source is selected if the
   highest version of a package is available from multiple sources.

   The hook should return a integer indicating the 'trust level' that
   a given index should be afforded for a given package.  Indices with
   higher trust levels will be preferred.

   The default behavior is to use a trust level of zero for unsigned
   repositories and a level of one for repositories with valid
   signatures.

   The IsTrusted predicate on a pkgAcqArchive object will return true
   when the highest trust level is non-zero, false otherwise.

   A trust level can be negative.  In that case, the index will never
   be considered as a source for the package.
*/

void
apt_set_index_trust_level_for_package_hook (int (*hook)
					    (pkgIndexFile *Index,
					     const pkgCache::VerIterator &V));

#endif
