/**
  @file fl.c

  @author Johan Hedberg <johan.hedberg@nokia.com>

  Copyright (C) 2004 Nokia. All rights reserved.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
    
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <getopt.h>
#include <time.h>

#include <glib.h>

#include <libxml/parser.h>
#include <libxml/tree.h>
#include <libxml/xmlmemory.h>

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#define STRCEQ(s1, s2) (g_ascii_strcasecmp((s1), (s2)) == 0)

#include "obc-main.h"
#include "fl.h"

/* Taken from Imendio's GnomeVFS OBEX module (om-utils.c) */
static time_t parse_iso8601(const gchar *str) {
    struct tm tm;
    gint      nr;
    gchar     tz;
    time_t    time;
    time_t    tz_offset = 0;

    memset(&tm, 0, sizeof (struct tm));

    nr = sscanf(str, "%04u%02u%02uT%02u%02u%02u%c",
            &tm.tm_year, &tm.tm_mon, &tm.tm_mday,
            &tm.tm_hour, &tm.tm_min, &tm.tm_sec,
            &tz);

    /* Fixup the tm values */
    tm.tm_year -= 1900;       /* Year since 1900 */
    tm.tm_mon--;              /* Months since January, values 0-11 */
    tm.tm_isdst = -1;         /* Daylight savings information not avail */

    if (nr < 6) /* Invalid time format */
        return -1;

    time = mktime (&tm);

#if defined(HAVE_TM_GMTOFF)
    tz_offset = tm.tm_gmtoff;
#elif defined(HAVE_TIMEZONE)
    tz_offset = -timezone;
    if (tm.tm_isdst > 0)
        tz_offset += 3600;
#endif

    if (nr == 7)   /* Date/Time was in localtime (to remote device)
                    * already. Since we don't know anything about the
                    * timezone on that one we won't try to apply UTC offset
                    */
        time += tz_offset;

    return time;
}

static FlEntry *fl_entry_new(void) {
    FlEntry *e;

    e = g_new0(FlEntry, 1);
    e->size     = -1;
    e->modified = -1;
    e->created  = -1;

    return e;
}

static void fl_entry_free(FlEntry *e) {
    g_free(e->name);
    g_free(e->type);
    g_free(e);
}

static GSList *add_file(GSList *l, xmlNode *node, gboolean dir) {
    FlEntry *e;
    char *name, *size, *modified, *created, *type;

    e = fl_entry_new();

    name     = (char *) xmlGetProp(node, (xmlChar*)"name");
    type     = (char *) xmlGetProp(node, (xmlChar*)"type");
    size     = (char *) xmlGetProp(node, (xmlChar*)"size");
    modified = (char *) xmlGetProp(node, (xmlChar*)"modified");
    created  = (char *) xmlGetProp(node, (xmlChar*)"created");

    e->dir = dir;

    if (name) {
        e->name = g_strdup(name);
        xmlFree(name);
    }
    if (type) {
        e->type = g_strdup(type);
        xmlFree(type);
    }
    if (modified) {
        e->modified = parse_iso8601(modified);
        xmlFree(modified);
    }
    if (created) {
        e->created = parse_iso8601(created);
        xmlFree(created);
    }
    if (size) {
        e->size = atoi(size);
        xmlFree(size);
    }

    return g_slist_append(l, e);
}

static GSList *add_parent(GSList *l, xmlNode *node) {
    FlEntry *e;

    e = fl_entry_new();

    e->dir = TRUE;
    e->name = g_strdup("..");

    return g_slist_append(l, e);
}

static GSList *make_fl_list(xmlNode *root) {
    GSList *l;
    xmlNode *cur_node = NULL, *fl_node = NULL;

    l = NULL;

    for (cur_node = root; cur_node; cur_node = cur_node->next) {
        if (cur_node->type == XML_ELEMENT_NODE &&
            STRCEQ((gchar*)cur_node->name, "folder-listing")) {
                fl_node = cur_node->children;
                break;
        }
    }

    if (fl_node == NULL)
        return NULL;

    for (cur_node = fl_node; cur_node; cur_node = cur_node->next) {
        if (cur_node->type == XML_ELEMENT_NODE) {
            if (STRCEQ((gchar*)cur_node->name, "parent-folder"))
                l = add_parent(l, cur_node);
            else if (STRCEQ((gchar*)cur_node->name, "folder"))
                l = add_file(l, cur_node, TRUE);
            else if (STRCEQ((gchar*)cur_node->name, "file"))
                l = add_file(l, cur_node, FALSE);
            else
                printf("ignoring element: %s\n", cur_node->name);
        }
    }

    return l;
}

static void print_element_names(xmlNode * a_node) {
    xmlNode *cur_node = NULL;

    for (cur_node = a_node; cur_node; cur_node = cur_node->next) {
        switch (cur_node->type) {
            case XML_ELEMENT_NODE:
                printf("XML_ELEMENT_NODE: %s\n", cur_node->name);
                break;

            case XML_ATTRIBUTE_NODE:
                printf("XML_ATTRIBUTE_NODE: %s\n", cur_node->name);
                break;

            case XML_TEXT_NODE:
                printf("XML_TEXT_NODE: %s\n", cur_node->name);
                break;

            case XML_CDATA_SECTION_NODE:
                printf("XML_CDATA_SECTION_NODE: %s\n", cur_node->name);
                break;

            default:
                printf("Unknown node (%d): %s\n", cur_node->type, cur_node->name);
                break;
        }

        print_element_names(cur_node->children);
    }
}

static xmlDocPtr parse_fl_xml(const gchar *buf, gint len) {
    xmlDocPtr doc;

    /* get document */
    doc = xmlParseMemory(buf, len);

    if (doc == NULL) {
        printf("xmlParseMemory() failed\n");
        return NULL;
    }

    if (doc->xmlRootNode == NULL || doc->xmlRootNode->name == NULL
            || !STRCEQ((gchar*)doc->xmlRootNode->name, "folder-listing")) {
        printf("Invalid folder-listing xml document\n");
        xmlFreeDoc(doc);
        return NULL;
    }

    return doc;
}

static void fl_entry_print(FlEntry *e, gboolean show_type) {
    char tstr[13], *size;
    time_t t = -1;

    if (e->modified >= 0)
        t = e->modified;
    else 
        t = e->created;

    if (t != -1) {
        struct tm *tm = gmtime(&t);
        time_t now = time(NULL);

        if (now - (6*30*24*60*60) > t) /* Older than 6 months */
            strftime(tstr, sizeof(tstr), "%b %e  %Y", tm);
        else
            strftime(tstr, sizeof(tstr), "%b %e %H:%M", tm);
    }

    if (e->size >= 0)
        size = g_strdup_printf("%d", e->size);
    else
        size = g_strdup(" ");

    if (show_type)
        printf("%s %8s %12s %-20s %-29s\n",
                e->dir ? "D" : " ",
                size,
                t != -1 ? tstr : " ",
                e->type ? e->type : " ",
                e->name ? e->name : "(no name)");
    else
        printf("%s %8s %12s %-49s\n",
                e->dir ? "D" : " ",
                size,
                t != -1 ? tstr : " ",
                e->name ? e->name : "(no name)");

    g_free(size);
}

void fl_list_free(GSList *l) {
    g_slist_foreach(l, (GFunc)fl_entry_free, NULL);
    g_slist_free(l);
}

static gboolean fl_get_buf(ObcContext *ctx, const gchar *dir,
                           gboolean progress,
                           gchar **buf, gint *buf_size,
                           gint *err) {
    gboolean prog, ret;

    prog = ctx->show_progress;
    obc_show_progress(ctx, progress);
    ctx->object = g_strdup("x-obex/folder-listing");
    ret = gw_obex_read_dir(ctx->obex, dir, buf, buf_size, err);
    g_free(ctx->object);
    ctx->object = NULL;
    obc_show_progress(ctx, prog);
    
    return ret;
}

FlEntry *fl_list_find_name(GSList *l, const char *name) {
    GSList *c;

    for (c = l; c != NULL; c = c->next) {
        FlEntry *e = c->data;
        g_assert(e != NULL);
        if (e->name == NULL)
            continue;
        if (g_str_equal(e->name, name))
            return e;
    }

    return NULL;
}

GSList *fl_list_get(ObcContext *ctx, const gchar *dir, gboolean progress,
                    gint *err) {
    GSList *l;
    gchar *buf;
    gint len;
    xmlNodePtr root;
    xmlDocPtr  doc;

    if (!fl_get_buf(ctx, dir, progress, &buf, &len, err))
        return NULL;

    doc = parse_fl_xml(buf, len);
    g_free(buf);
    if (doc == NULL) {
        printf("Unable to parse folder-listing\n");
        xmlCleanupParser();
        return NULL;
    }

    root = xmlDocGetRootElement(doc);

    l = make_fl_list(root);

    xmlFreeDoc(doc);
    xmlCleanupParser();

    return l;
}

gboolean cmd_ls(ObcContext *ctx, gint argc, gchar *argv[], gint *err) {
    int c, i;
    char *dir;
    const char *nl;
    gboolean raw, show_type, show_prog, multiple, current;

    /* Defaults */
    nl = "";
    raw = FALSE;
    current = FALSE;
    show_type = FALSE;
    show_prog = FALSE;
    multiple = FALSE;

    while ((c = getopt(argc, argv, "Rrpt")) != -1) {
        switch (c) {
            case 'r':
                printf("Recursive listing not yet implemented\n");
                break;
            case 'R':
                raw = TRUE;
                break;
            case 'p':
                show_prog = TRUE;
                break;
            case 't':
                show_type = TRUE;
                break;
            default:
                printf("Unhandled option character: '%c'\n", c);
                break;
        }
    }

    c = optind;
    optind = 0;

    if (argc > c)
        multiple = TRUE;
    if (argc == c)
        current = TRUE;

    i = c;
    do {
        if (current)
            dir = NULL;
        else
            dir = argv[i];

        ctx->xfer_complete = FALSE;
        ctx->start = time(NULL);

        if (raw) {
            gchar *buf, *fmt;
            gint buf_size;

            if (!fl_get_buf(ctx, dir, show_prog, &buf, &buf_size, err))
                return FALSE;

            if (multiple)
                printf("%s%s:\n", nl, dir);

            fmt = g_strdup_printf("%%%ds", buf_size);
            printf(fmt, buf);
            g_free(fmt);

            g_free(buf);
        }
        else {
            GSList *l;

            l = fl_list_get(ctx, dir, show_prog, err);
            if (l == NULL && err && *err >= 0x40)
                return FALSE;

            if (multiple)
                printf("%s%s:\n", nl, dir);

            g_slist_foreach(l, (GFunc)fl_entry_print, (gpointer)show_type);

            if (current) {
                fl_list_free(ctx->fl_cache);
                ctx->fl_cache = l;
            }
            else
                fl_list_free(l);
        }

        if (current)
            break;

        if (i == c)
            nl = "\n";

        i++;
    } while (i < argc);

    return TRUE;
}

GSList *fl_list_match_glob(GSList *l, const char *glob) {
    GSList *m, *c;
    GPatternSpec *spec;

    spec = g_pattern_spec_new(glob);
    m = NULL;

    for (c = l; c != NULL; c = c->next) {
        FlEntry *e = c->data;
        g_assert(e != NULL);
        if (e->name == NULL)
            continue;
        if (g_pattern_match_string(spec, e->name))
            m = g_slist_append(m, e);
    }

    g_pattern_spec_free(spec);

    return m;
}

void ls_help(ObcContext *ctx, const char *name) {
    printf("%s [-r] [-p] [-t] [-R] [file..]\n"
           "List contents of remote folder\n"
           "Parameters:\n"
           "-r\tRecursive listing\n"
           "-p\tShow progress while fetching folder-listing\n"
           "-t\tShow file type (if available)\n"
           "-R\tShow raw XML\n",
           name);
}

