/*
 * This file is part of lessertunjo
 *
 * Copyright (C) 2006 Nokia Corporation.
 *
 * Contact: Kuisma Salonen <kuisma.salonen@nokia.com>
 * Author: Kuisma Salonen <kuisma.salonen@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */


/**
 * @file sappd_spawn.c
 * shadow application library, daemon side library,
 * child spawning part
 * <p>
 *
 * @author Kuisma Salonen <Kuisma.Salonen@nokia.com>
 */

/**********************************************************/

#define DBUS_API_SUBJECT_TO_CHANGE

#include <glib.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/un.h>
#include <unistd.h>
#include <libosso.h>
#include <sys/wait.h>
#include <sys/types.h>
#include <sys/socket.h>

#include <dbus/dbus.h>
#include <gconf/gconf-client.h>

#include "shadowappd.h"

static gboolean kill_no_mercy(gpointer data)
{
  sapp_d_data *ctx = (sapp_d_data *)data;

  kill(ctx->child_pid, SIGKILL);

  return FALSE;
}

int sappd_spawn(sapp_d_data *context, int command)
{
  int rc, rc2;
  int temp;
  char *arglist[5];

  if((command == SAPPD_COMMAND_RUN && (context->state == SAPPD_STATE_INACTIVE ||
                                       context->state == SAPPD_STATE_ACTIVE)) ||
     (command == SAPPD_COMMAND_RESET && (context->state == SAPPD_STATE_ACTIVE)) ||
     (command == SAPPD_COMMAND_RESTART && (context->state == SAPPD_STATE_INACTIVE ||
					   context->state == SAPPD_STATE_ACTIVE))) {

    context->connstate = SAPPD_CONNSTATE_ACTIVE;

    rc = fork();
    if(rc == -1) {
#ifdef SAPPD_DEBUG
      _sd_debugprint("fork() failed");
#endif
      return SAPP_ERROR_EXECUTION;
    } else if(rc == 0) { /* lapsi */
      arglist[0] = context->executable;
      arglist[1] = context->our_addr->sun_path;
      arglist[2] = (char *)malloc(2);
      sprintf(arglist[2], "%c", command + 0x30);
      arglist[3] = context->alt_addr->sun_path;
      arglist[4] = NULL;

      if(execv(context->executable, arglist) == -1) {
#ifdef SAPPD_DEBUG
        _sd_debugprint("cannot execute our glorious file");
#endif
        free(arglist[2]);
        exit(-1);
      }

      free(arglist[2]);
      exit(0);
    } else { /* äitee */
      context->connection_sd = accept(context->listener_sd,
                                      (struct sockaddr *)context->incoming_addr,
                                      &(context->socket_lenght));
      if(context->connection_sd == -1) { /* should exit automatically when no */
        waitpid(rc, NULL, 0);            /* connection available              */
        return SAPP_ERROR_CONNECTION;
      }

      context->child_pid = rc;

      rc2 = recv(context->connection_sd, &temp, 4, 0);
      if(rc2 != 4) {
        close(context->connection_sd);
        waitpid(rc, NULL, 0);
        return SAPP_ERROR_CONNANOTHER;
      }
      if(temp != rc) {
        temp = SAPP_ERROR_WRONGID;
        send(context->connection_sd, &temp, 4, 0);
        close(context->connection_sd);
        waitpid(rc, NULL, 0);
        return SAPP_ERROR_CONNANOTHER;
      }

      temp = SAPP_ERROR_NONE;
      send(context->connection_sd, &temp, 4, 0);

      context->state = SAPPD_STATE_RUNNING;

      /* apply networking */
      context->channel = g_io_channel_unix_new(context->connection_sd);
      g_io_add_watch(context->channel, G_IO_IN | G_IO_HUP | G_IO_ERR,
                     sappd_socket_callback, (gpointer)context);
      g_io_channel_unref(context->channel);
      context->connstate = SAPPD_CONNSTATE_ACTIVE;
    }
  } else if((command == SAPPD_COMMAND_RUN && context->state == SAPPD_STATE_PAUSED) ||
            (command == SAPPD_COMMAND_RESET && (context->state == SAPPD_STATE_PAUSED ||
                                                context->state == SAPPD_STATE_RUNNING)) ||
            (command == SAPPD_COMMAND_RESTART && (context->state == SAPPD_STATE_PAUSED ||
                                                  context->state == SAPPD_STATE_RUNNING))) {
    if(command == SAPPD_COMMAND_RUN) {
      send(context->connection_sd, "continue\n", 9, 0);
      context->state = SAPPD_STATE_RUNNING;
    } else if(command == SAPPD_COMMAND_RESTART) {
      send(context->connection_sd, "restart\n", 8, 0);
      context->state = SAPPD_STATE_RUNNING;
    } else {
      send(context->connection_sd, "quit\n", 5, 0);
      g_timeout_add(2000, kill_no_mercy, context);
      context->state = SAPPD_STATE_INACTIVE;
    }
  } else if(command == SAPPD_COMMAND_RUN && context->state == SAPPD_STATE_RUNNING) {
    send(context->connection_sd, "top\n", 4, 0);
  } else {
    return SAPP_ERROR_INVALIDPARAMS;
  }

  return SAPP_ERROR_NONE;
}

int sappd_rpc(sapp_d_data *context, char *mthd)
{
  DBusMessage *msg;
  dbus_bool_t b;
  char *path, *service, *iface;

  if(context == NULL || mthd == NULL) {
    return SAPP_ERROR_INVALIDPARAMS;
  }

  path = (char *)malloc(strlen(context->path) + 10);
  service = (char *)malloc(strlen(context->service) + 10);
  iface = (char *)malloc(strlen(context->iface) + 10);

  sprintf(path, "%s/startup", context->path);
  sprintf(iface, "%s.startup", context->iface);
  sprintf(service, "%s.startup", context->service);

  msg = dbus_message_new_method_call(service, path, iface, mthd);
  if(msg == NULL) {
    /* prevent leaking */
    free(path); free(iface); free(service);
    return SAPP_ERROR_NOMEMORY;
  }

  dbus_message_set_no_reply(msg, TRUE);
  b = dbus_connection_send(osso_get_dbus_connection(context->osso_context), msg, NULL);

  if(b) {
    dbus_connection_flush(osso_get_dbus_connection(context->osso_context));
  } else {
    free(path); free(iface); free(service);
    return SAPP_ERROR_NOMEMORY;
  }

  dbus_message_unref(msg);

  free(path); free(iface); free(service);

  return SAPP_ERROR_NONE;
}
